#include "main.h"

#include <contrib/libs/grpc/src/core/lib/iomgr/executor.h>

#include <infra/pod_agent/libs/client/client.h>
#include <infra/pod_agent/libs/config/config.pb.h>
#include <infra/pod_agent/libs/service/alive_service.h>
#include <infra/pod_agent/libs/service/service.h>
#include <infra/libs/service_iface/str_iface.h>

#include <library/cpp/proto_config/load.h>
#include <library/cpp/sighandler/async_signals_handler.h>

#include <util/folder/dirut.h>
#include <util/string/vector.h>

namespace NInfra::NPodAgent {

TLoggerConfig GetStderrLoggerConfig() {
    TLoggerConfig config;
    config.SetBackend(TLoggerConfig_ELogBackend_STDERR);
    return config;
}

int RunDaemon(int argc, const char* argv[]) {
    TConfig config;
    try {
        config = NProtoConfig::GetOpt<TConfig>(argc, argv, "/pod_agent/proto_config/config.json");
    } catch (yexception& e) {
        ythrow e << "While parsing pod_agent config.";
    }

    // Because mod chooser argv[0] is "<pod_agent_binary> <run_mode>"
    TVector<TString> argv0Parts = SplitString(argv[0], " ");
    Y_ENSURE(
        argv0Parts.size() == 2u
        , "Argv[0] must be '<pod_agent_binary> <run_mode>', but actual value is '" << argv[0] << "'"
    );
    TString selfBinaryFilePath = RealPath(argv0Parts[0]);

    // At this moment FS is not initialized (logdir not created)
    // so the only place where we can write log is stderr
    TLogger stderrLogger = GetStderrLoggerConfig();

    THolder<TAliveService> aliveService = TAliveService::MakeService(
        config
        , stderrLogger
    );

    grpc_core::Executor::SetThreadsLimit(config.GetGrpcCoreConfig().GetMaxGrpcCoreThreads());

    // alive service should be started before main service, because main service has variable time delay on creation for fs initialization
    aliveService->Start();

    THolder<TService> service = TService::MakeService(
        selfBinaryFilePath
        , config
        , stderrLogger
    );

    service->Start();

    std::function<void(int)> shutdown = [config](int) {
        try {
            const TString address = TGrpcService::AddressFromConfig(config.GetGrpcService());
            TClient client(address);

            TAttributes emptyAttrs;
            TString emptyStr;

            client.Shutdown(
                RequestPtr<TEmptyRequest<TReqShutdown>>(TStringBuf(), emptyStr, emptyAttrs)
                , ReplyPtr<TEmptyReply<TRspShutdown>> (emptyStr, emptyAttrs)
            );
        } catch(...) {
            Cerr << CurrentExceptionMessage() << "Throwed exception while signal handler run.\n";
        }
    };

    SetAsyncSignalFunction(SIGINT, shutdown);
    SetAsyncSignalFunction(SIGTERM, shutdown);

    service->Wait();
    aliveService->Stop();

    return EXIT_SUCCESS;
}

} // namespace NInfra::NPodAgent
