#include "main.h"

#include <util/generic/vector.h>
#include <util/stream/output.h>

#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>

namespace NInfra::NPodAgent {

int RunExecWrapper(int argc, const char* argv[]) {
    if (argc < 4) {
        Cerr << "[Exec wrapper] Usage: " << argv[0] << " <user> <group> exec_argv..." << Endl;
        return EXIT_FAILURE;
    }

    if (getuid() != 0) {
        Cerr << "[Exec wrapper] Error: you can't run exec wrapper without root privileges." << Endl;
        return EXIT_FAILURE;
    }

    const char* user = argv[1];
    const char* group = argv[2];

    const auto* userInfo = getpwnam(user);
    if (nullptr == userInfo) {
        Cerr << "[Exec wrapper] Error: user '" << user << "' not found." << Endl;
        return EXIT_FAILURE;
    }

    const auto* groupInfo = getgrnam(group);
    if (nullptr == groupInfo) {
        Cerr << "[Exec wrapper] Error: group '" << group << "' not found." << Endl;
        return EXIT_FAILURE;
    }

    if (initgroups(user, groupInfo->gr_gid) != 0) {
        Cerr << "[Exec wrapper] Error: initgroups failed with '" << strerror(errno) << "' (errno = " << errno << ")" << Endl;
        return errno;
    }

    // initgroups does not change the gid
    // We need to change it using setgid
    if (setgid(groupInfo->gr_gid) != 0) {
        Cerr << "[Exec wrapper] Error: setgid failed with '" << strerror(errno) << "' (errno = " << errno << ")" << Endl;
        return errno;
    }

    if (setuid(userInfo->pw_uid) != 0) {
        Cerr << "[Exec wrapper] Error: setuid failed with '" << strerror(errno) << "' (errno = " << errno << ")" << Endl;
        return errno;
    }

    const char* execFile = argv[3];
    TVector<char*> execArgv(Reserve(argc - 2)); // argv[3], argv[4], ..., argv[argc - 1], nullptr
    for (int i = 3; i < argc; ++i) {
        execArgv.push_back(const_cast<char*>(argv[i])); // execvp has non-const argv
    }
    // execvp(3): The array of pointers must be terminated by a NULL pointer.
    // However it is not guaranteed that argv[argc] == nullptr (Thanks to TModChooser)
    execArgv.push_back(nullptr);

    execvp(execFile, execArgv.data());
    Cerr << "[Exec wrapper] Error: exec failed with '" << strerror(errno) << "' (errno = " << errno << ")" << Endl;

    return errno;
}

} // namespace NInfra::NPodAgent
