#include "service.h"

#include <infra/pod_agent/libs/service_iface/protos/service.grpc.pb.h>
#include <infra/pod_agent/libs/service_iface/protos/service.pb.h>

#include <util/string/cast.h>
#include <util/system/guard.h>
#include <util/system/hostname.h>

namespace NInfra::NPodAgent {

using namespace grpc;

namespace {

template <typename TMessage>
class TGrpcRequest: public IRequest<TMessage> {
public:
    TGrpcRequest(const TMessage& message, const TAttributes& attrs)
        : IRequest<TMessage>("", attrs)
        , Message_(message)
    {
    }

    const TMessage& Get() const override {
        return Message_;
    }

private:
    const TMessage& Message_;
};

template <typename TMessage>
TRequestPtr<TMessage> GrpcRequest(const TMessage& message, const TAttributes& attrs) {
    return new TGrpcRequest<TMessage>(message, attrs);
}

template <typename TMessage>
class TGrpcReply: public IReply<TMessage> {
public:
    TGrpcReply(TMessage& message, TAttributes& attrs)
        : IReply<TMessage>(attrs)
        , Message_(message)
    {
    }

    void Set(const TMessage& message) override {
        Message_ = message;
    }

private:
    TMessage& Message_;
};

template <typename TMessage>
TReplyPtr<TMessage> GrpcReply(TMessage& message, TAttributes& attrs) {
    return new TGrpcReply<TMessage>(message, attrs);
}

} // namespace

class TGrpcService::TImpl: public TPodAgentService::Service {
public:
    TImpl(IService& service, TLogFramePtr logFrame)
        : Service_(service)
        , LogFrame_(logFrame)
    {
    }

    Status Config(ServerContext* ctx, const TReqConfig* request, TRspConfig* reply) override {
        return Call(ctx, request, reply, &IApi::Config);
    }

    Status Shutdown(ServerContext* ctx, const TReqShutdown* request, TRspShutdown* reply) override {
        return Call(ctx, request, reply, &IApi::Shutdown);
    }

    Status SetLogLevel(ServerContext* ctx, const TReqSetLogLevel* request, TRspSetLogLevel* reply) override {
        return Call(ctx, request, reply, &IApi::SetLogLevel);
    }

    Status UpdatePodAgentRequest(ServerContext* ctx, const API::TPodAgentRequest* request, API::TPodAgentStatus* reply) override {
        return Call(ctx, request, reply, &IApi::UpdatePodAgentRequest);
    }

    Status GetPodAgentStatus(ServerContext* ctx, const TReqGetPodAgentStatus* request, API::TPodAgentStatus* reply) override {
        return Call(ctx, request, reply, &IApi::GetPodAgentStatus);
    }

    Status Ping(ServerContext* ctx, const TReqPing* request, TRspPing* reply) override {
        return Call(ctx, request, reply, &IApi::Ping);
    }

    Status Sensors(ServerContext* ctx, const TReqSensors* request, TRspSensors* reply) override {
        return Call(ctx, request, reply, &IApi::Sensors);
    }

    Status UserSensors(ServerContext* ctx, const TReqSensors* request, TRspSensors* reply) override {
        return Call(ctx, request, reply, &IApi::UserSensors);
    }

    Status Version(ServerContext* ctx, const TReqVersion* request, TRspVersion* reply) override {
        return Call(ctx, request, reply, &IApi::Version);
    }

    Status ReopenLog(ServerContext* ctx, const TReqReopenLog* request, TRspReopenLog* reply) override {
        return Call(ctx, request, reply, &IApi::ReopenLog);
    }

private:
    template <typename TRequest, typename TReply>
    Status Call(ServerContext* ctx, const TRequest* request, TReply* reply, void (IApi::*call)(TRequestPtr<TRequest>, TReplyPtr<TReply>)) {
        try {
            TAttributes attributes;
            for (const auto& [key, value] : ctx->client_metadata()) {
                attributes.insert({TString(key.data(), key.size()), TString(value.data(), value.size())});
            }

            TAttributes replyAttributes;
            (Service_.*call)(GrpcRequest(*request, attributes), GrpcReply(*reply, replyAttributes));
            return Status::OK;
        } catch (...) {
            const TString message = CurrentExceptionMessage();
            LogFrame_->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TServiceError("grpc", 1, message));
            return Status(StatusCode::INTERNAL, std::string(message.data()));
        }
    }

private:
    IService& Service_;
    TLogFramePtr LogFrame_;
};

TGrpcService::TGrpcService(const TGrpcServiceConfig& config, IService& service)
    : Config_(config)
    , Service_(service)
{
}

void TGrpcService::Start(TLogFramePtr logFrame) {
    const TString address = AddressFromConfig(Config_);

    ServerBuilder_ = MakeHolder<ServerBuilder>();
    ServerBuilder_->AddListeningPort(address, InsecureServerCredentials());

    Impl_ = MakeHolder<TImpl>(Service_, logFrame);
    ServerBuilder_->RegisterService(Impl_.Get());

    std::unique_ptr<Server> server(ServerBuilder_->BuildAndStart());
    Server_.Reset(server.release());

    NLogEvent::TStartService ev("grpc", address);
    logFrame->LogEvent(ev);
}

void TGrpcService::Wait(TLogFramePtr logFrame) {
    {
        TGuard<TMutex> guard(ShutdownMutex_);

        while (!ShutdownCalled_) {
            ShutdownCondvar_.Wait(ShutdownMutex_);
        }
    }

    Stop();

    logFrame->LogEvent(NLogEvent::TStopService("grpc"));
}

void TGrpcService::Shutdown() {
    TGuard<TMutex> guard(ShutdownMutex_);

    if (!ShutdownCalled_) {
        ShutdownCalled_ = true;

        ShutdownCondvar_.Signal();
    }
}

void TGrpcService::Stop() {
    Server_->Shutdown();

    Server_->Wait();
}

TString TGrpcService::AddressFromConfig(const TGrpcServiceConfig& config) {
    return config.HasPort()
        ? (TString("dns:*:") + ToString(config.GetPort()) )
        : (TString("unix:") + (config.GetUnixSocket().StartsWith('/') ? "/" : "") + config.GetUnixSocket() );
}

TGrpcService::~TGrpcService() {
    try {
        Stop();
    } catch(...) {
    }
}

} // namespace NInfra::NPodAgent
