#include "cached_unique_event_iterator.h"

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/protos/events.ev.pb.h>
#include <infra/libs/logger/test_common.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/tempfile.h>

namespace NInfra::NPodAgent::NTestEventlogCachedUniqueEventIterator {

Y_UNIT_TEST_SUITE(EventlogCachedUniqueEventIteratorSuite) {

Y_UNIT_TEST(UniqueEvents) {
    TString logFile = "eventlog_unique_event_iter";

    TTempFile tmp(logFile);

    TLogger logger(NTestCommon::CreateLoggerConfig("DEBUG", logFile));
    const size_t eventsCount = 21;
    const size_t steps = 5;

    {

        for (size_t i = 1; i <= eventsCount / 3; ++i) {
            for (size_t j = 1; j <= steps; ++j) {
                auto frame = logger.SpawnFrame();
                NLogEvent::TBehaviourTreeTick msg;
                msg.set_title(ToString(j % i));
                msg.set_tick(i * 100 + j);
                frame->LogEvent(msg);
                logger.ReopenLog();
            }
        }
        for (size_t i = eventsCount / 3 + 1; i <= eventsCount / 3 * 2; ++i) {
            for (size_t j = 1; j <= steps; ++j) {
                auto frame = logger.SpawnFrame();
                NLogEvent::TBehaviourTreeTickError msg;
                msg.set_title(ToString(j % i));
                msg.set_tick(i * 100 + j);
                frame->LogEvent(msg);
                logger.ReopenLog();
            }
        }
        for (size_t i = eventsCount / 3 * 2 + 1; i <= eventsCount; ++i) {
            for (size_t j = 1; j <= steps; ++j) {
                auto frame = logger.SpawnFrame();
                NLogEvent::TTestMessage msg;
                msg.set_data(ToString(j % i));
                frame->LogEvent(msg);
                logger.ReopenLog();
            }
        }
    }

    logger.CloseLog();

    NInfra::TIteratorOptions options;
    options.SetFileName(logFile);

    TCachedUniqueEventIterator iter(options);
    const NInfra::TEvent* event;
    TStack<TString> messages;
    TSet<TString> uniqueMessages;

    for (;;) {
        try {
            event = iter.Next();

            if (event) {
                    messages.push(event->GetData());
                    uniqueMessages.insert(messages.top());
                } else {
                    break;
                }
            } catch(...) {
                Cout.Flush();
            }
        }

        UNIT_ASSERT_EQUAL(messages.size(), uniqueMessages.size());
        UNIT_ASSERT_C(messages.size() < eventsCount * steps, messages.size());

        messages.pop();

        auto removeTick = [](TString& str) {
            size_t tickPos = str.find("Tick");
            if (tickPos == TString::npos) {
                return;
            }
            size_t titlePos = str.find("Title");
            UNIT_ASSERT(titlePos != TString::npos);
            UNIT_ASSERT(titlePos < tickPos);
            str = str.substr(0, tickPos);
        };

        for (;;) {
            try {
                event = iter.Prev();

                if (event) {
                    UNIT_ASSERT(messages);
                    TString eventData = event->GetData();
                    removeTick(eventData);
                    TString oldData  = messages.top();
                    removeTick(oldData);
                    UNIT_ASSERT_EQUAL_C(oldData, eventData, TStringBuilder() << oldData << " :: " << eventData);
                    messages.pop();
                } else {
                    break;
                }
            } catch(...) {
                Cout.Flush();
            }
        }
        UNIT_ASSERT(!messages);
    }

}

} // namespace NInfra::NPodAgent::NTestEventlogCachedUniqueEventIterator
