#include "cached_unique_event_iterator.h"
#include "interactive_log.h"

#include <util/string/builder.h>

namespace NInfra::NPodAgent {

namespace {
    TString GetEnumName(const TString& name) {
        size_t firstUnderline = name.find_first_of('_');
        Y_ENSURE(firstUnderline != TString::npos, "Full enum name must have at least one '_': '" << name << "'");
        return name.substr(firstUnderline + 1);
    }

    TString SnakeCaseToCamleCase(const TString& st) {
        TString result;
        bool prevUnderline = true; // toupper first symbol
        for (const char& c : st) {
            if (c == '_') {
                prevUnderline = true;
            } else {
                if (prevUnderline) {
                    result.push_back(toupper(c));
                } else {
                    result.push_back(tolower(c));
                }

                prevUnderline = false;
            }
        }

        return result;
    }

    TString NodeTypeToString(NLogEvent::TBehaviourTreeTickV2::ENodeType nodeType) {
        return SnakeCaseToCamleCase(GetEnumName(NLogEvent::TBehaviourTreeTickV2::ENodeType_Name(nodeType)));
    }

    TString NodeStatusToString(NLogEvent::TBehaviourTreeTickV2::ENodeStatus nodeStatus) {
        return GetEnumName(NLogEvent::TBehaviourTreeTickV2::ENodeStatus_Name(nodeStatus));
    }

    void DFSRender(const NLogEvent::TBehaviourTreeTickV2::TNodeTick& node, i32 depth, TStringBuilder& result) {
        static const TString SINGLE_INDENT = "  ";
        result << SINGLE_INDENT * depth;

        TString currentTitle = node.GetTitle();
        if (currentTitle.StartsWith("Root:")) {
            // Parse as "Root:<tree name>:<node title>
            ssize_t pos = currentTitle.find_last_of(':');
            TString realTitle = currentTitle.substr(pos + 1);
            if (realTitle.empty()) {
                realTitle = NodeTypeToString(node.GetNodeType());
            } else {
                realTitle = TStringBuilder()
                    << NodeTypeToString(node.GetNodeType())
                    << "(" << realTitle << ")"
                ;
            }
            result << "RR> " << currentTitle.substr(0, pos + 1) << realTitle;
        } else {
            if (currentTitle.empty()) {
                currentTitle = NodeTypeToString(node.GetNodeType());
            } else {
                currentTitle = TStringBuilder()
                    << NodeTypeToString(node.GetNodeType())
                    << "(" << currentTitle << ")"
                ;
            }
            result << "--> " << currentTitle;
        }

        result << " [";
        TString message;
        if (node.HasSuccess()) {
            result << NodeStatusToString(node.GetSuccess().GetStatus());
            message = node.GetSuccess().GetMessage();
        } else if (node.HasError()) {
            result << "ERROR";
            message = node.GetError().GetMessage();
        }
        if (message) {
            result << " : " << message;
        }
        result << "]";

        for (auto&& child : node.children()) {
            result << Endl;
            DFSRender(child, depth + 1, result);
        }
    }
}

void TTermBoxWorker::Draw() {
    tb_clear();
    i32 x = -XOffset_, y = -YOffset_;
    for (size_t i = 0; i < CurrentString_.size(); ++i) {
        if (CurrentString_[i] == '\n') {
            x = -XOffset_;
            ++y;
            continue;
        }
        if (x >= 0 && y >= 0 && x <= tb_width() && y <= tb_height()) {
            tb_change_cell(x, y, (CurrentString_[i] == '\t' ? ' ' : CurrentString_[i]), TB_DEFAULT, TB_DEFAULT);
        }
        ++x;
    }
    tb_present();
}

TString RenderTree(const NLogEvent::TBehaviourTreeTickV2& tree) {
    TStringBuilder result;
    result << "TreeId: " << tree.GetTreeId() << "\n";
    result << "Tick: " << tree.GetTick() << "\n";
    DFSRender(tree.GetRoot(), 0, result);
    return result;
}

TString RenderEvent(const TEvent* event, TEvent::TEventContext& context) {
    Y_ENSURE(event, "null event pointer passed");
    TStringBuilder stringBuilder;

    if (context.PrevEventTime == 0)
        context.PrevEventTime = event->Timestamp();
    TStringOutput out(stringBuilder);
    event->PrintHeader(out, context);
    context.PrevEventTime = event->Timestamp();

    stringBuilder << event->GetProto()->GetDescriptor()->name() << '\n';

    if (auto tick = dynamic_cast<NLogEvent::TBehaviourTreeTick*>(event->GetProto())) {
        stringBuilder << "Id: " << tick->GetId() << "\n";
        stringBuilder << "TreeId: " << tick->GetTreeId() << "\n";
        stringBuilder << "Tick: " << tick->GetTick() << "\n";
        stringBuilder << "Type: " << tick->GetType() << "\n";
        stringBuilder << "Status: " << tick->GetStatus() << "\n";
        stringBuilder << "Title: \n" << tick->GetTitle() << "\n";
        return stringBuilder;
    }
    if (auto tick = dynamic_cast<NLogEvent::TBehaviourTreeTickError*>(event->GetProto())) {
        stringBuilder << "Id: " << tick->GetId() << "\n";
        stringBuilder << "TreeId: " << tick->GetTreeId() << "\n";
        stringBuilder << "Tick: " << tick->GetTick() << "\n";
        stringBuilder << "Type: " << tick->GetType() << "\n";
        stringBuilder << "Error: " << tick->GetError() << "\n";
        stringBuilder << "Title: \n" << tick->GetTitle() << "\n";
        return stringBuilder;
    }
    if (auto tick = dynamic_cast<NLogEvent::TBehaviourTreeTickV2*>(event->GetProto())) {
        stringBuilder << RenderTree(*tick);
        return stringBuilder;
    }
    stringBuilder << event->GetData();
    return stringBuilder;
}

void StartInteractiveLog(const TIteratorOptions& options, TEvent::TEventContext context,  bool uniqueOnly) {
    TTermBoxWorker termBox;

    TEventIteratorPtr iter;
    if (uniqueOnly) {
        iter.Reset(new TCachedUniqueEventIterator(options));
    } else {
        iter.Reset(new TCachedEventIterator(options));
    }

    {
        const TEvent* event = iter->SafeNext();
        if (event) {
            termBox.RenderString(RenderEvent(event, context));
        } else {
            termBox.RenderString("No event found");
        }
    }
    while (true) {
        tb_event userEvent;
        tb_poll_event(&userEvent);
        if (userEvent.type == TB_EVENT_KEY) {
            if (userEvent.key == TB_KEY_ESC || userEvent.ch == 'q') {
                return;
            }
            switch (userEvent.key) {
                case TB_KEY_ENTER: {
                    const TEvent* event = iter->SafeNext();
                    if (event) {
                        termBox.RenderString(RenderEvent(event, context));
                    } else {
                        termBox.RenderString("No event found");
                    }
                    break;
                }
                case TB_KEY_BACKSPACE:
                case TB_KEY_BACKSPACE2: {
                    const TEvent* event = iter->SafePrev();
                    if (event) {
                        termBox.RenderString(RenderEvent(event, context));
                    } else {
                        termBox.RenderString("No event found");
                    }
                    break;
                }
                case TB_KEY_ARROW_DOWN: {
                    termBox.IncreaseOffset(0, 1);
                    break;
                }
                case TB_KEY_ARROW_UP: {
                    termBox.IncreaseOffset(0, -1);
                    break;
                }
                case TB_KEY_ARROW_LEFT: {
                    termBox.IncreaseOffset(-1, 0);
                    break;
                }
                case TB_KEY_ARROW_RIGHT: {
                    termBox.IncreaseOffset(1, 0);
                    break;
                }
                case TB_KEY_PGDN: {
                    termBox.IncreaseOffset(0, tb_height() / 2);
                    break;
                }
                case TB_KEY_PGUP: {
                    termBox.IncreaseOffset(0, -tb_height() / 2);
                    break;
                }
                default:
                    break;
            }
        }
    }
}

} // namespace NInfra::NPodAgent
