#include "main.h"
#include "interactive_log.h"

#include <cstdlib>

#include <library/cpp/eventlog/dumper/common.h>
#include <library/cpp/getopt/small/last_getopt.h>

namespace NInfra::NPodAgent {

int RunInteractiveLog(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;

    opts.AddHelpOption('h');
    opts.SetTitle("Interactive log\n"
                  "Use arrow keys, page up and page down to scroll\n"
                  "Use Enter and Backspace to navigate\n"
                  "Esc or q to exit");

    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "<eventlog>", "Event log file");

    TString start;
    opts.AddLongOption(
            's', "start-time")
        .Optional()
        .StoreResult(&start);

    TString end;
    opts.AddLongOption(
            'e', "end-time")
        .Optional()
        .StoreResult(&end);

    TIteratorOptions options;

    TVector<TString> includeEventList;
    opts.AddLongOption(
            'i', "event-list",
            "Comma-separated list of included event class IDs or names "
            "(e.g. 265,287 or MainTaskError,ContextCreated)")
        .Optional()
        .SplitHandler(&includeEventList, ',')
        .StoreValue(&options.IncludeExcludeFlag, true);

    TVector<TString> excludeEventList;
    opts.AddLongOption(
            'x', "exclude-list",
            "Comma-separated list of excluded event class IDs or names (see also --event-list option)")
        .Optional()
        .SplitHandler(&excludeEventList, ',')
        .StoreValue(&options.IncludeExcludeFlag, false);

    TString treeIds;
    bool includeTrees = false;
    opts.AddLongOption(
            'p', "include-tree-list",
            "Comma-separated list of included tree ids")
        .Optional()
        .StoreResult(&treeIds)
        .StoreValue(&includeTrees, true);

    bool excludeTrees = false;
    opts.AddLongOption(
            'c', "exclude-tree-list",
            "Comma-separated list of excluded tree ids")
        .Optional()
        .StoreResult(&treeIds)
        .StoreValue(&excludeTrees, true);

    opts.AddLongOption(
            'v', "version",
            "Print program version")
        .NoArgument()
        .Optional()
        .Handler(&NPrivate::PrintVersionAndExit);

    opts.AddLongOption(
            't', "tail",
            "Open file like `tail -f` does")
        .NoArgument()
        .Optional()
        .StoreValue(&options.TailFMode, true);

    bool humanReadable = false;
    opts.AddLongOption(
            'r', "human-readable",
            "Print some fields (e.g. timestamp) in human-readable format, add time offsets")
        .NoArgument()
        .StoreValue(&humanReadable, true);

    bool uniqueOnly = false;
    opts.AddLongOption(
        'u', "unique",
        "Print only unique events")
        .NoArgument()
        .StoreValue(&uniqueOnly, true);

    NLastGetopt::TOptsParseResult parsedOpts(&opts, argc, argv);

    if (includeEventList && excludeEventList) {
        throw NLastGetopt::TUsageException() << "You can use either include (-i) or exclude (-x) events list. ";
    }
    if (includeTrees && excludeTrees) {
        throw NLastGetopt::TUsageException() << "You can use either include (-p) or exclude (-c) tree id list. ";
    }

    options.SetFileName(parsedOpts.GetFreeArgs()[0])
        .SetStartTime(ParseTime(start, TIteratorOptions::MIN_START_TIME))
        .SetEndTime(ParseTime(end, TIteratorOptions::MAX_END_TIME))
        .SetEventList(options.IncludeExcludeFlag ? includeEventList : excludeEventList);

    if (includeTrees || excludeTrees) {
        TVector<TString> allowedTreeIdList;
        Split(treeIds.data(), ",", allowedTreeIdList);
        THashSet<TString> treeIdFilter(allowedTreeIdList.begin(), allowedTreeIdList.end());
        options.SetEventPredicate(
            [treeIdFilter=std::move(treeIdFilter), exclude=excludeTrees](const TEvent* event) {
                auto field = event->GetProto()->GetDescriptor()->FindFieldByName("TreeId");
                if (!field)
                    return true;
                if (field->cpp_type() != google::protobuf::FieldDescriptor::CPPTYPE_STRING)
                    return true;
                auto reflection = event->GetProto()->GetReflection();
                TString value = reflection->GetString(*event->GetProto(), field);
                // invert the result if should exclude
                return treeIdFilter.contains(value) != exclude;
            }
        );
    }

    TEvent::TEventContext context{.HumanReadable = humanReadable, .NoTTy = true};
    StartInteractiveLog(options, context, uniqueOnly);

    return EXIT_SUCCESS;
}

} // namespace NInfra::NPodAgent
