#pragma once

#include "client.h"

#include <library/cpp/threading/future/async.h>

namespace NInfra::NPodAgent {

using NThreading::TFuture;

class TAsyncIpClient;
using TAsyncIpClientPtr = TIntrusivePtr<TAsyncIpClient>;

/*
    Runs Client_ functions in a given MtpQueue
*/
class TAsyncIpClient: public TAtomicRefCount<TAsyncIpClient> {
public:
    TAsyncIpClient(
        TIpClientPtr client
        , TAtomicSharedPtr<IThreadPool> mtpQueue
    )
        : Client_(client)
        , MtpQueuePtr_(mtpQueue)
    {
        Y_ENSURE(MtpQueuePtr_, "MtpQueue not defined for TAsyncIpClient");
    }

    TFuture<TExpected<void, TIpClientError>> AddAddress(const TString& device, const TIpDescription& ip) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->AddAddress(device, ip);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<void, TIpClientError>> RemoveAddress(const TString& device, const TIpDescription& ip) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->RemoveAddress(device, ip);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TVector<TIpDescription>, TIpClientError>> ListAddress(const TString& device) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ListAddress(device);
            }
            , *MtpQueuePtr_
        );
    }

private:
    TIpClientPtr Client_;
    TAtomicSharedPtr<IThreadPool> MtpQueuePtr_;
};

} // namespace NInfra::NPodAgent
