#pragma once

#include <library/cpp/unistat/unistat.h>

#include <util/generic/hash.h>
#include <util/generic/singleton.h>

namespace NInfra::NPodAgent {

using TUnistatPtr = TAtomicSharedPtr<TUnistat>;

class TMultiUnistat {
public:
    enum ESignalNamespace: ui8 {
        INFRA    /* "infra" */,
        USER     /* "user" */,
    };
    enum ESignalPriority: ui8 {
        DEBUG = 1 /* "debug" */,

        // Due to problems with signal separation they are again combined in one place
        // Because of this we get two different levels of info signals
        // For more information: DEPLOY-1897 and DEPLOY-2045
        USER_INFO = 2 /* "user_info" */,
        INFRA_INFO = 3 /* "infra_info" */,
    };

public:
    TMultiUnistat()
    {
        for (const auto& signalNamespace : ALL_SIGNAL_NAMESPACES) {
            Unistats_[signalNamespace] = new TUnistat();
        }
    }

    static TMultiUnistat& Instance() {
        return *Singleton<TMultiUnistat>();
    };

    NUnistat::IHolePtr DrillFloatHole(
        const ESignalNamespace signalNamespace
        , const TString& name
        , const TString& description
        , const TString& suffix
        , NUnistat::TPriority priority
        , NUnistat::TStartValue startValue = NUnistat::TStartValue(0)
        , EAggregationType type = EAggregationType::Sum
        , bool alwaysVisible = false
    );

    NUnistat::IHolePtr DrillFloatHole(
        const ESignalNamespace signalNamespace
        , const TString& name
        , const TString& suffix
        , NUnistat::TPriority priority
        , NUnistat::TStartValue startValue = NUnistat::TStartValue(0)
        , EAggregationType type = EAggregationType::Sum
        , bool alwaysVisible = false
    );

    NUnistat::IHolePtr DrillHistogramHole(
        const ESignalNamespace signalNamespace
        , const TString& name
        , const TString& description
        , const TString& suffix
        , NUnistat::TPriority priority
        , const NUnistat::TIntervals& intervals
        , EAggregationType type = EAggregationType::HostHistogram
        , bool alwaysVisible = false
    );

    NUnistat::IHolePtr DrillHistogramHole(
        const ESignalNamespace signalNamespace
        , const TString& name
        , const TString& suffix
        , NUnistat::TPriority priority
        , const NUnistat::TIntervals& intervals
        , EAggregationType type = EAggregationType::HostHistogram
        , bool alwaysVisible = false
    );

    template<typename T>
    bool PushSignalUnsafe(const ESignalNamespace signalNamespace, const T& holename, double signal) {
        return GetUnistatInstance(signalNamespace)->PushSignalUnsafe(holename, signal);
    };

    template<typename T>
    bool ResetSignalUnsafe(const ESignalNamespace signalNamespace, const T& holename) {
        return GetUnistatInstance(signalNamespace)->ResetSignalUnsafe(holename);
    };

    TVector<TString> GetHolenames(const ESignalNamespace signalNamespace) const;

    TString CreateJsonDump(const ESignalNamespace signalNamespace, int level, bool allHoles = true) const;

    TString CreateInfoDump(const ESignalNamespace signalNamespace, int level) const;

    bool EraseHole(const ESignalNamespace signalNamespace, const TString& name);

    void Reset(const ESignalNamespace signalNamespace);

    void ResetAll();

    void ResetSignals(const ESignalNamespace signalNamespace);

    void ResetAllSignals();

    void ExportToProto(const ESignalNamespace signalNamespace, TInstanceStats& stats, int level) const;

private:
    const TUnistatPtr GetUnistatInstance(ESignalNamespace signalNamespace) const;

private:
    static inline const TVector<TMultiUnistat::ESignalNamespace> ALL_SIGNAL_NAMESPACES = {
        TMultiUnistat::ESignalNamespace::INFRA
        , TMultiUnistat::ESignalNamespace::USER
    };

    THashMap<ESignalNamespace, TUnistatPtr> Unistats_;
};

};
