#include "multi_unistat.h"

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NMultiUnistatTest {

const TVector<TString> USER_SIGNALS = {"same_", "some_a", "some_b"};
const TVector<TString> INFRA_SIGNALS = {"d_free", "other_one", "same_"};

void SetupSignals(TMultiUnistat& instance, TVector<TString> signals, TMultiUnistat::ESignalNamespace signalNamespace) {
    for (const TString& signal : signals) {
        instance.DrillFloatHole(
            signalNamespace
            , signal
            , "deee"
            , NUnistat::TPriority(1)
            , NUnistat::TStartValue(0)
            , EAggregationType::Sum
        );
    }
}

TMultiUnistat& GetMultiUnistat() {
    return TMultiUnistat::Instance();
}

TMultiUnistat& SetupMultiUnistat(TMultiUnistat& multiUnistat) {
    multiUnistat.ResetAll();
    SetupSignals(multiUnistat, USER_SIGNALS, TMultiUnistat::ESignalNamespace::USER);
    SetupSignals(multiUnistat, INFRA_SIGNALS, TMultiUnistat::ESignalNamespace::INFRA);
    return multiUnistat;
}

Y_UNIT_TEST_SUITE(MultiUnistatSuite) {

Y_UNIT_TEST(TestHoleNames) {
    auto& multiUnistat = SetupMultiUnistat(GetMultiUnistat());

    TVector<TString> infraHoleNames = multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA);
    Sort(infraHoleNames.begin(), infraHoleNames.end());
    UNIT_ASSERT_EQUAL(infraHoleNames, INFRA_SIGNALS);

    TVector<TString> userHoleNames = multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::USER);
    Sort(userHoleNames.begin(), userHoleNames.end());
    UNIT_ASSERT_EQUAL(userHoleNames, USER_SIGNALS);
}

Y_UNIT_TEST(TestResetByType) {
    auto& multiUnistat = SetupMultiUnistat(GetMultiUnistat());

    {
        TVector<TString> infraHoleNames = multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA);
        Sort(infraHoleNames.begin(), infraHoleNames.end());
        UNIT_ASSERT_EQUAL(infraHoleNames, INFRA_SIGNALS);
    }

    {
        multiUnistat.Reset(TMultiUnistat::ESignalNamespace::USER);
        TVector<TString> infraHoleNames = multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA);
        Sort(infraHoleNames.begin(), infraHoleNames.end());
        UNIT_ASSERT_EQUAL(infraHoleNames, INFRA_SIGNALS);
    }

    {
        multiUnistat.Reset(TMultiUnistat::ESignalNamespace::INFRA);
        TVector<TString> infraHoleNames = multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA);
        Sort(infraHoleNames.begin(), infraHoleNames.end());
        UNIT_ASSERT_EQUAL(infraHoleNames, TVector<TString>());
    }
}

Y_UNIT_TEST(TestResetAll) {
    auto& multiUnistat = SetupMultiUnistat(GetMultiUnistat());

    UNIT_ASSERT_EQUAL(multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::USER).size(), 3);
    UNIT_ASSERT_EQUAL(multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA).size(), 3);

    multiUnistat.ResetAll();
    UNIT_ASSERT_EQUAL(multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::USER).size(), 0);
    UNIT_ASSERT_EQUAL(multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA).size(), 0);
}

Y_UNIT_TEST(TestEraseHole) {
    auto& multiUnistat = SetupMultiUnistat(GetMultiUnistat());

    UNIT_ASSERT(!multiUnistat.EraseHole(TMultiUnistat::ESignalNamespace::INFRA, "unknown_signal"));
    UNIT_ASSERT(multiUnistat.EraseHole(TMultiUnistat::ESignalNamespace::INFRA, "d_free"));
    UNIT_ASSERT(!multiUnistat.EraseHole(TMultiUnistat::ESignalNamespace::INFRA, "d_free"));

    UNIT_ASSERT_EQUAL(multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::INFRA), TVector<TString>({"other_one", "same_"}));
    UNIT_ASSERT_EQUAL(multiUnistat.GetHolenames(TMultiUnistat::ESignalNamespace::USER).size(), 3);
}

}

} // `namespace NInfra::NPodAgent::NMultiUnistatTest
