#pragma once

#include "network_client_types.h"
#include "network_request.h"

#include <infra/libs/outcome/result.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>

namespace NInfra::NPodAgent {

class INetworkClient;
using TNetworkClientPtr = TIntrusivePtr<INetworkClient>;

class INetworkClient: public TAtomicRefCount<INetworkClient> {
public:
    enum ERequestState {
        IN_QUEUE      /* "in_queue" */,
        RUNNING       /* "running" */,
        COMPLETED     /* "completed" */,
    };

    struct TRequestPublicInfo {
        TRequestPublicInfo(
            const TString& key
            , const TString& hash
            , const TString& additionalInfo
        )
            : Key_(key)
            , Hash_(hash)
            , AdditionalInfo_(additionalInfo)
        {}

        const TString Key_;
        const TString Hash_;
        const TString AdditionalInfo_;
    };

public:
    virtual ~INetworkClient() = default;

    virtual TExpected<TString, TNetworkClientError> GetLocalHostName() = 0;

    // Add http request
    // If already exist check hash.
    // If hash mismatched, remove and add new with correct hash
    virtual TExpected<void, TNetworkClientError> CheckAndAddHttpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , const TString& host
        , ui32 port
        , const TString& path
        , TDuration timeout
    ) = 0;
    // Add tcp request
    // If already exist check hash.
    // If hash mismatched, remove and add new with correct hash
    virtual TExpected<void, TNetworkClientError> CheckAndAddTcpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , ui32 port
        , TDuration timeout
    ) = 0;
    // Remove request by key (do not check hash)
    virtual TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) = 0;
    // Return list of added requests
    virtual TExpected<TVector<TRequestPublicInfo>, TNetworkClientError> ListRequests() const = 0;

    // Get request state. Return error if hash mismatched.
    virtual TExpected<ERequestState, TNetworkClientError> GetRequestState(const TString& requestKey, const TString& requestHash) = 0;
    // Get request description. Return error if hash mismatched.
    virtual TExpected<TString, TNetworkClientError> GetRequestDescription(const TString& requestKey, const TString& requestHash) = 0;
    // Get response for request and remove it from client memory. Return error if hash mismatched or request not completed.
    virtual TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) = 0;
};

} // namespace NInfra::NPodAgent
