#include "network_http_request.h"

#include <library/cpp/neh/http2.h>
#include <library/cpp/http/io/stream.h>

namespace NInfra::NPodAgent {

TNetworkHttpRequest::TNetworkHttpRequest(
    const TString& host
    , ui32 port
    , const TString& path
)
    : Host_(host)
    , Port_(port)
    , Path_(path)
    , Address_(MakeAddress())
{}

TExpected<void, TNetworkClientError> TNetworkHttpRequest::Init() {

    auto msg = NNeh::TMessage(
        Address_
        , ""    
    );

    bool ok = NNeh::NHttp::MakeFullRequest(
        msg,
        "Connection: close",
        "",
        NNeh::NHttp::DefaultContentType,
        NNeh::NHttp::ERequestType::Get
    );

    if (!ok) {
        return TNetworkClientError(
            ENetworkClientError::RequestError
            , Description() + ": Request error, unable to make full request"
        );
    }

    Request_ = NNeh::Request(msg);

    return TExpected<void, TNetworkClientError>::DefaultSuccess();
}

bool TNetworkHttpRequest::Execute(const TDuration& timeout) {
    return Request_->Wait(Response_, timeout);
}

void TNetworkHttpRequest::Cancel() {
    Request_->Cancel();
    AtomicSet(Cancelled_, 1);
}

bool TNetworkHttpRequest::IsError() const {
    return AtomicGet(Cancelled_) || Response_->IsError();
}

TString TNetworkHttpRequest::GetErrorText() const {
    if (AtomicGet(Cancelled_)) {
        return TStringBuilder() << Description() << ": Connection cancelled";
    }
    return TStringBuilder() << Description() << ": " << Response_->GetErrorText();
}

TString TNetworkHttpRequest::Data() const {
    return Response_->Data;
}

TString TNetworkHttpRequest::Description() const {
    return TStringBuilder() << "HTTP request to " << Address_;
}

TString TNetworkHttpRequest::MakeAddress() const {
    return TStringBuilder()
        << NNeh::Http2Protocol()->Scheme() << "://" << Host_ << ":" << ToString(Port_)
        << (Path_.StartsWith("/") ? "" : "/") << Path_;
}

} // namespace NInfra::NPodAgent
