#include "simple_client.h"

#include <library/cpp/threading/future/async.h>
#include <library/cpp/http/io/stream.h>
 
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/system/hostname.h>

namespace NInfra::NPodAgent {

TExpected<TString, TNetworkClientError> TSimpleNetworkClient::GetLocalHostName() {
    TString hostName;
    try {
        hostName = HostName();
    } catch(...) {
        return TNetworkClientError(
            ENetworkClientError::GetLocalHostNameError
            , CurrentExceptionMessage()
        );
    }

    if (hostName.empty()) {
        return TNetworkClientError(
            ENetworkClientError::GetLocalHostNameError
            , "hostname is empty"
        );
    }

    return hostName;
}

TExpected<void, TNetworkClientError> TSimpleNetworkClient::CheckAndAddHttpRequest(
    const TString& requestKey
    , const TString& requestHash
    , const TString& additionalInfo
    , const TString& host
    , ui32 port
    , const TString& path
    , TDuration timeout
) {
    TNetworkRequestPtr request = MakeIntrusive<TNetworkHttpRequest>(host, port, path);

    return CheckAndAddRequest(requestKey, requestHash, additionalInfo, timeout, request);
}

TExpected<void, TNetworkClientError> TSimpleNetworkClient::CheckAndAddTcpRequest(
    const TString& requestKey
    , const TString& requestHash
    , const TString& additionalInfo
    , ui32 port
    , TDuration timeout
) {
    TNetworkRequestPtr request = MakeIntrusive<TNetworkTcpRequest>(port);

    return CheckAndAddRequest(requestKey, requestHash, additionalInfo, timeout, request);
}

TExpected<void, TNetworkClientError> TSimpleNetworkClient::CheckAndAddRequest(
    const TString& requestKey
    , const TString& requestHash
    , const TString& additionalInfo
    , TDuration timeout
    , TNetworkRequestPtr request
) {
    TWriteGuardBase<TLightRWLock> guard(GlobalRequestLock_);
    if (auto ptr = RequestInfo_.FindPtr(requestKey)) {
        if (ptr->Hash_ == requestHash) {
            return TNetworkClientError(
                ENetworkClientError::RequestAlreadyExist
                , TStringBuilder()
                    << request->Description() << ": "
                    << "pod agent error: "
                    << "request with key '" << requestKey << "' already exist"
            );
        } else {
            OUTCOME_TRYV(RemoveRequestNoLock(requestKey));
        }
    }

    OUTCOME_TRYV(request->Init());
    TSimpleNetworkClient::TRequestInfo& info = RequestInfo_[requestKey];
    LocalRequestLock_[requestKey];

    info.Hash_ = requestHash;
    info.AdditionalInfo_ = additionalInfo;
    info.State_ = TSimpleNetworkClient::ERequestState::IN_QUEUE;

    info.Request_ = request;
    info.RequestFuture_ = NThreading::Async(
        [this, requestKey, request, timeout]() -> TExpected<TString, TNetworkClientError> {
            {
                TReadGuardBase<TLightRWLock> guard(GlobalRequestLock_);
                auto mutex = OUTCOME_TRYX(GetRequestMutex(requestKey));
                TGuard<TMutex> g(*(mutex));
                RequestInfo_[requestKey].State_ = TSimpleNetworkClient::ERequestState::RUNNING;
            }

            bool isTimeout = !request->Execute(timeout);

            {
                TReadGuardBase<TLightRWLock> guard(GlobalRequestLock_);
                auto mutex = OUTCOME_TRYX(GetRequestMutex(requestKey));
                TGuard<TMutex> g(*(mutex));
                RequestInfo_[requestKey].State_ = TSimpleNetworkClient::ERequestState::COMPLETED;

                if (isTimeout) {
                    return TNetworkClientError(
                        ENetworkClientError::RequestTimeout
                        , request->Description() + ": Request timeout"
                    );
                } else if (request->IsError()) {
                    return TNetworkClientError(
                        ENetworkClientError::RequestError
                        , request->GetErrorText()
                    );
                } else {
                    return request->Data();
                }
            }
        }
        , *MtpQueuePtr_
    );

    return TExpected<void, TNetworkClientError>::DefaultSuccess();
}

TExpected<void, TNetworkClientError> TSimpleNetworkClient::RemoveRequest(const TString& requestKey) {
    TWriteGuardBase<TLightRWLock> guard(GlobalRequestLock_);
    return RemoveRequestNoLock(requestKey);
}

TExpected<TVector<TSimpleNetworkClient::TRequestPublicInfo>, TNetworkClientError> TSimpleNetworkClient::ListRequests() const {
    TReadGuardBase<TLightRWLock> guard(GlobalRequestLock_);

    TVector<TSimpleNetworkClient::TRequestPublicInfo> keys;
    keys.reserve(RequestInfo_.size());
    for (const auto& [key, info] : RequestInfo_) {
        keys.emplace_back(
            key
            , info.Hash_
            , info.AdditionalInfo_
        );
    }

    return keys;
}

TExpected<TSimpleNetworkClient::ERequestState, TNetworkClientError> TSimpleNetworkClient::GetRequestState(const TString& requestKey, const TString& requestHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalRequestLock_);
    auto mutex = OUTCOME_TRYX(GetRequestMutex(requestKey)); // check that exist
    TGuard<TMutex> g(mutex);

    auto ptr = RequestInfo_.FindPtr(requestKey);
    OUTCOME_TRYV(CheckRequestHashNoLock(ptr, requestKey, requestHash));

    return ptr->State_;
}

TExpected<TString, TNetworkClientError> TSimpleNetworkClient::GetRequestDescription(const TString& requestKey, const TString& requestHash) {
    TWriteGuardBase<TLightRWLock> guard(GlobalRequestLock_);

    auto ptr = OUTCOME_TRYX(GetRequestInfoNoLock(requestKey));
    OUTCOME_TRYV(CheckRequestHashNoLock(ptr, requestKey, requestHash));
    return ptr->Request_->Description();
}

TExpected<TString, TNetworkClientError> TSimpleNetworkClient::GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) {
    TWriteGuardBase<TLightRWLock> guard(GlobalRequestLock_);

    auto ptr = OUTCOME_TRYX(GetRequestInfoNoLock(requestKey));
    OUTCOME_TRYV(CheckRequestHashNoLock(ptr, requestKey, requestHash));

    // No mutex because WriteLock
    if (ptr->State_ != TSimpleNetworkClient::ERequestState::COMPLETED || !ptr->RequestFuture_.HasValue()) {
        return TNetworkClientError(
            ENetworkClientError::RequestIncompleted
            , TStringBuilder()
                << ptr->Request_->Description() << ": "
                << "pod agent error: "
                << "request with key '" << requestKey << "' incomleted, currect state '" << ToString(ptr->State_) << "'"
        );
    }

    auto result = ptr->RequestFuture_.GetValueSync();

    RequestInfo_.erase(requestKey);
    LocalRequestLock_.erase(requestKey);

    return result;
}

TExpected<void, TNetworkClientError> TSimpleNetworkClient::RemoveRequestNoLock(const TString& requestKey) {
    auto ptr = OUTCOME_TRYX(GetRequestInfoNoLock(requestKey));

    // We need Cancel request to interrupt future
    ptr->Request_->Cancel();

    // Remove request from internal state
    RequestInfo_.erase(requestKey);
    LocalRequestLock_.erase(requestKey);

    return TExpected<void, TNetworkClientError>::DefaultSuccess();
}

TExpected<const TMutex*, TNetworkClientError> TSimpleNetworkClient::GetRequestMutex(const TString& requestKey) const {
    auto ptr = LocalRequestLock_.FindPtr(requestKey);
    if (!ptr) {
        return TNetworkClientError(
            ENetworkClientError::RequestDoesNotExist
            , TStringBuilder()
                << "pod agent error: "
                << "request with key '" << requestKey << "' does not exist"
        );
    }

    return ptr;
}

TExpected<const TSimpleNetworkClient::TRequestInfo*, TNetworkClientError> TSimpleNetworkClient::GetRequestInfoNoLock(const TString& requestKey) const {
    auto ptr = RequestInfo_.FindPtr(requestKey);
    if (!ptr) {
        return TNetworkClientError(
            ENetworkClientError::RequestDoesNotExist
            , TStringBuilder()
                << "pod agent error: "
                << "request with key '" << requestKey << "' does not exist"
        );
    }
    return ptr;
}

TExpected<void, TNetworkClientError> TSimpleNetworkClient::CheckRequestHashNoLock(const TRequestInfo* requestInfoPtr, const TString& requestKey, const TString& requestHash) const {
    if (requestInfoPtr->Hash_!= requestHash) {
        return TNetworkClientError(
            ENetworkClientError::RequestHashMismatched
            , TStringBuilder()
                << requestInfoPtr->Request_->Description() << ": "
                << "pod agent error: "
                << "request with key '" << requestKey << "' hash mismatched, "
                << "expected: '" << requestHash << "', got: '" << requestInfoPtr->Hash_ << "'"
        );
    }
    return TExpected<void, TNetworkClientError>::DefaultSuccess();
}


} // namespace NInfra::NPodAgent
