#pragma once

#include "client.h"
#include "network_http_request.h"
#include "network_tcp_request.h"

#include <library/cpp/threading/future/future.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/generic/hash.h>
#include <util/system/mutex.h>
#include <util/thread/pool.h>

namespace NInfra::NPodAgent {

class TSimpleNetworkClient;
using TSimpleNetworkClientPtr = TIntrusivePtr<TSimpleNetworkClient>;

class TSimpleNetworkClient: public INetworkClient {
public:
    TSimpleNetworkClient(
         TAtomicSharedPtr<IThreadPool> mtpQueue
    )
        : INetworkClient()
        , MtpQueuePtr_(mtpQueue)
    {
    }

    virtual ~TSimpleNetworkClient() = default;

    virtual TExpected<TString, TNetworkClientError> GetLocalHostName() override;

    virtual TExpected<void, TNetworkClientError> CheckAndAddHttpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , const TString& host
        , ui32 port
        , const TString& path
        , TDuration timeout
    ) override;
    virtual TExpected<void, TNetworkClientError> CheckAndAddTcpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , ui32 port
        , TDuration timeout
    ) override;
    virtual TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override;
    virtual TExpected<TVector<TRequestPublicInfo>, TNetworkClientError> ListRequests() const override;

    virtual TExpected<ERequestState, TNetworkClientError> GetRequestState(const TString& requestKey, const TString& requestHash) override;
    virtual TExpected<TString, TNetworkClientError> GetRequestDescription(const TString& requestKey, const TString& requestHash) override;
    virtual TExpected<TString, TNetworkClientError> GetAndRemoveRequestResponse(const TString& requestKey, const TString& requestHash) override;

private:
    struct TRequestInfo {
        TString AdditionalInfo_;
        TString Hash_;

        NThreading::TFuture<TExpected<TString, TNetworkClientError>> RequestFuture_;
        TNetworkRequestPtr Request_;
        ERequestState State_;
    };

public: //public for tests
    TExpected<void, TNetworkClientError> CheckAndAddRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , TDuration timeout
        , TNetworkRequestPtr request
    );
private:
    TExpected<void, TNetworkClientError> RemoveRequestNoLock(const TString& requestKey);
    TExpected<const TMutex*, TNetworkClientError> GetRequestMutex(const TString& requestKey) const;
    TExpected<const TRequestInfo*, TNetworkClientError> GetRequestInfoNoLock(const TString& requestKey) const;
    TExpected<void, TNetworkClientError> CheckRequestHashNoLock(const TRequestInfo* requestInfoPtr, const TString& requestKey, const TString& requestHash) const;

private:
    TAtomicSharedPtr<IThreadPool> MtpQueuePtr_;

    TLightRWLock GlobalRequestLock_;
    THashMap<TString, TMutex> LocalRequestLock_;
    THashMap<TString, TRequestInfo> RequestInfo_;
};

} // namespace NInfra::NPodAgent
