#include "simple_client.h"
#include "mock_network_request.h"

#include <infra/libs/http_service/service.h>
#include <infra/libs/http_service/test_common.h>
#include <infra/libs/logger/logger.h>
#include <infra/libs/service_iface/fake_routers.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NNetworkClientTest {

static TLogger logger({});

class TFakeBlockRouter: public TFakeBaseRouter {
public:
    TFakeBlockRouter(const TDuration& timeout)
        : NeedQuit_(0)
        , Timeout_(timeout)
    {}

    void ReleaseBlock() {
        AtomicSet(NeedQuit_, 1);
    }

    TRouterResponse Handle(TStringBuf, const TString&, const TVector<std::pair<TString, TString>>&) const override {
        TInstant deadline = TInstant::Now() + Timeout_;
        while (AtomicGet(NeedQuit_) == 0 && TInstant::Now() < deadline) {
            Sleep(TDuration::MilliSeconds(500));
        }
        return {TString("Data"), TAttributes()};
    }

protected:
    bool DoHas(TStringBuf) const override {
        return true;
    }

protected:
    TAtomic NeedQuit_;
    TDuration Timeout_;
};

class ITestNetworkClientCanon {
public:
    ITestNetworkClientCanon()
        : MtpQueuePtr_(new TThreadPool())
        , FakePort_(NTesting::GetFreePort())
    {
        NetworkClient_ = new TSimpleNetworkClient(MtpQueuePtr_);
        MtpQueuePtr_->Start(MAX_THREADS);
    }

    virtual ~ITestNetworkClientCanon() = default;

    void DoTest() {
        Test();
        HttpServicesShutdown();
        MtpQueuePtr_->Stop();
    }

protected:
    virtual void Test() = 0;

    void AddHttpService(
        TSimpleSharedPtr<TFakeBaseRouter> router
    ) {
        Routers_.push_back(router);
        HttpServicePorts_.push_back(NTesting::GetFreePort());
        HttpServiceConfigs_.push_back(NTestCommon::GenerateHttpServiceConfig(HttpServicePorts_.back()));
        HttpServices_.push_back(MakeHolder<THttpService>(
            HttpServiceConfigs_.back()
            , Routers_.back()
        ));
        HttpServices_.back()->Start(logger.SpawnFrame());
    }

    void HttpServicesShutdown() {
        for (auto& httpService : HttpServices_) {
            httpService->ShutDown();
        }
    }

    void AddHttpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , const ui32 httpServiceId = 0
        , const TString& suffix = ""
        , const TDuration& timeout = DEFAULT_TIMEOUT
        , bool alreadyExist = false
    ) {
        auto result = NetworkClient_->CheckAndAddHttpRequest(
            requestKey
            , requestHash
            , additionalInfo
            , "localhost"
            , httpServiceId < HttpServiceConfigs_.size() ? HttpServiceConfigs_[httpServiceId].GetPort() : FakePort_
            , DEFAULT_PATH + (suffix.empty() ? "" : "_" + suffix)
            , timeout
        );
        if (!alreadyExist) {
            UNIT_ASSERT_C(result, result.Error().Message);
        } else {
            UNIT_ASSERT_C(!result, "CheckAndAddHttpRequest must fail");
            UNIT_ASSERT_C(result.Error().Errno == ENetworkClientError::RequestAlreadyExist, result.Error().Message);
        }
    }

    void AddTcpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , const ui16 port
        , const TDuration& timeout = DEFAULT_TIMEOUT
        , bool alreadyExist = false
    ) {
        auto result = NetworkClient_->CheckAndAddTcpRequest(
            requestKey
            , requestHash
            , additionalInfo
            , port
            , timeout
        );
        if (!alreadyExist) {
            UNIT_ASSERT_C(result, result.Error().Message);
        } else {
            UNIT_ASSERT_C(!result, "CheckAndAddTcpRequest must fail");
            UNIT_ASSERT_C(result.Error().Errno == ENetworkClientError::RequestAlreadyExist, result.Error().Message);
        }
    }

    void AddManyHttpRequests(
        size_t cntRequestAdds
        , size_t cntRequest
        , const TString& requestKeyPrefix = "key"
        , const TString requestHashPrefix = "hash"
        , const TString& additionalInfoPrefix = "info"
        , const ui32 httpServiceId = 0
        , const TString& suffixPrefix = ""
        , const TDuration& timeout = DEFAULT_TIMEOUT
    ) {
        for (size_t i = 0; i < cntRequestAdds; ++i) {
            for (size_t j = 0; j < cntRequest; ++j) {
                AddHttpRequest(
                    requestKeyPrefix + ToString(j)
                    , requestHashPrefix + ToString(j)
                    , additionalInfoPrefix + ToString(j)
                    , httpServiceId
                    , suffixPrefix + ToString(j)
                    , timeout
                    , i != 0
                );
            }
        }
    }

    bool WaitRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TInstant& deadline
        , const INetworkClient::ERequestState state = INetworkClient::ERequestState::COMPLETED
    ) {
        while (TInstant::Now() < deadline) {
            auto result = NetworkClient_->GetRequestState(requestKey, requestHash);
            UNIT_ASSERT_C(result, result.Error().Message);
            if (result.Success() == state) {
                return true;
            } else {
                Sleep(DEFAULT_TIMEOUT);
            }
        }

        return false;
    }

    NTesting::TPortHolder EnableTcpServer() {
        NTesting::TPortHolder port = NTesting::GetFreePort();
        TSockAddrInet6 addr("::", port);

        UNIT_ASSERT_NO_EXCEPTION(ListenTcpSocket_.CheckSock());
        {
            SOCKET sock = ListenTcpSocket_;
            const int yes = 1;
            ::setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (const char*)&yes, sizeof(yes));
        }
        UNIT_ASSERT_EQUAL(ListenTcpSocket_.Bind(&addr), 0);
        UNIT_ASSERT_EQUAL(ListenTcpSocket_.Listen(1), 0);

        return port;
    }

public:
    static const ui32 MAX_THREADS;
    static const TDuration DEFAULT_TIMEOUT;
    static const TString DEFAULT_PATH;

protected:
    TNetworkClientPtr NetworkClient_;

private:
    TAtomicSharedPtr<IThreadPool> MtpQueuePtr_;
    NTesting::TPortHolder FakePort_;
    TVector<TRequestRouterPtr> Routers_;
    TVector<NTesting::TPortHolder> HttpServicePorts_;
    TVector<THttpServiceConfig> HttpServiceConfigs_;
    TVector<THolder<THttpService>> HttpServices_;

    TInet6StreamSocket ListenTcpSocket_;
};

const ui32 ITestNetworkClientCanon::MAX_THREADS = 2;
const TDuration ITestNetworkClientCanon::DEFAULT_TIMEOUT = TDuration::MilliSeconds(500);
const TString ITestNetworkClientCanon::DEFAULT_PATH = "/path";

Y_UNIT_TEST_SUITE(NetworkClientTestSuite) {

Y_UNIT_TEST(TestCheckAndAddHttpRequestWithSuccess) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            const size_t cntRequest = 10;
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            AddHttpService(router);

            AddManyHttpRequests(2, cntRequest);
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), cntRequest, requestList.size());
            for (auto requestInfo : requestList) {
                const TString keySuffix = requestInfo.Key_.substr(strlen("key"));
                const TString hashSuffix = requestInfo.Hash_.substr(strlen("hash"));
                const TString infoSuffix = requestInfo.AdditionalInfo_.substr(strlen("info"));

                UNIT_ASSERT_EQUAL_C(keySuffix, hashSuffix, keySuffix << " != " << hashSuffix);
                UNIT_ASSERT_EQUAL_C(infoSuffix, hashSuffix, infoSuffix << " != " << hashSuffix);
            }

            TDuration reqTime = DEFAULT_TIMEOUT * (cntRequest + 2);
            TInstant deadline = TInstant::Now() + reqTime;
            size_t cntRequestOk = 0;
            for (auto requestInfo : requestList) {
                if (WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline)) {
                    auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
                    UNIT_ASSERT_C(result, result.Error().Message);
                    UNIT_ASSERT_EQUAL_C(result.Success(), "Data", result.Success());
                    ++cntRequestOk;

                    auto currentPaths = router->GetPaths();
                    const TString path = DEFAULT_PATH + "_" + requestInfo.Key_.substr(strlen("key"));
                    UNIT_ASSERT_C(Find(currentPaths.begin(), currentPaths.end(), path) != currentPaths.end(), "Path " << path << " not found in router");
                } else {
                    break;
                }

                size_t cntRequestInternal = NetworkClient_->ListRequests().Success().size();
                UNIT_ASSERT_EQUAL_C(cntRequestInternal, cntRequest - cntRequestOk, cntRequestInternal);
            }

            UNIT_ASSERT_EQUAL_C(cntRequestOk, cntRequest, "Fail to complete all request in " << reqTime << ", complete only " << cntRequestOk);
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestCheckAndAddHttpRequestWithSlowSuccess) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            const size_t cntRequest = 10;
            auto router = MakeSimpleShared<TFakeBlockRouter>(DEFAULT_TIMEOUT * 20);
            AddHttpService(router);

            AddManyHttpRequests(2, cntRequest, "key", "hash", "info", 0, "", TDuration::Max());
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), cntRequest, requestList.size());

            // Wait running state
            Sleep(DEFAULT_TIMEOUT * 5);

            size_t cntRunning = 0;
            for (auto requestInfo : requestList) {
                INetworkClient::ERequestState state = NetworkClient_->GetRequestState(requestInfo.Key_, requestInfo.Hash_).Success();
                UNIT_ASSERT_UNEQUAL_C(state, INetworkClient::ERequestState::COMPLETED, "Request can't be completed so fast");
                if (state == INetworkClient::ERequestState::RUNNING) {
                    ++cntRunning;
                }
            }
            UNIT_ASSERT_C(cntRunning >= 1, "At least one request must be in running state");
            router->ReleaseBlock();

            TDuration reqTime = DEFAULT_TIMEOUT * (cntRequest + 2);
            TInstant deadline = TInstant::Now() + reqTime;
            size_t cntRequestOk = 0;
            for (auto requestInfo : requestList) {
                if (WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline)) {
                    auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
                    UNIT_ASSERT_C(result, result.Error().Message);
                    UNIT_ASSERT_EQUAL_C(result.Success(), "Data", result.Success());
                    ++cntRequestOk;

                    auto currentPaths = router->GetPaths();
                    const TString path = DEFAULT_PATH + "_" + requestInfo.Key_.substr(strlen("key"));
                    UNIT_ASSERT_C(Find(currentPaths.begin(), currentPaths.end(), path) != currentPaths.end(), "Path " << path << " not found in router");
                } else {
                    break;
                }

                size_t cntRequestInternal = NetworkClient_->ListRequests().Success().size();
                UNIT_ASSERT_EQUAL_C(cntRequestInternal, cntRequest - cntRequestOk, cntRequestInternal);
            }

            UNIT_ASSERT_EQUAL_C(cntRequestOk, cntRequest, "Fail to complete all request in " << reqTime << ", complete only " << cntRequestOk);
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestHttpRequestWithFailure) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            const size_t cntRequest = 10;
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            AddHttpService(router);

            AddManyHttpRequests(2, cntRequest);
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), cntRequest, requestList.size());

            TDuration reqTime = DEFAULT_TIMEOUT * (cntRequest + 2);
            TInstant deadline = TInstant::Now() + reqTime;
            size_t cntRequestFail = 0;
            for (auto requestInfo : requestList) {
                if (WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline)) {
                    auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
                    UNIT_ASSERT_C(!result, result.Success());
                    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "404 Not found");
                    ++cntRequestFail;

                    auto currentPaths = router->GetPaths();
                    const TString path = DEFAULT_PATH + "_" + requestInfo.Key_.substr(strlen("key"));
                    UNIT_ASSERT_C(Find(currentPaths.begin(), currentPaths.end(), path) != currentPaths.end(), "Path " << path << " not found in router");
                } else {
                    break;
                }

                size_t cntRequestInternal = NetworkClient_->ListRequests().Success().size();
                UNIT_ASSERT_EQUAL_C(cntRequestInternal, cntRequest - cntRequestFail, cntRequestInternal);
            }

            UNIT_ASSERT_EQUAL_C(cntRequestFail, cntRequest, "Fail to complete all request in " << reqTime << ", complete only " << cntRequestFail);
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestHttpRequestWithNoService) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            const size_t cntRequest = 10;

            AddManyHttpRequests(2, cntRequest);
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), cntRequest, requestList.size());

            TDuration reqTime = DEFAULT_TIMEOUT * (cntRequest + 2);
            TInstant deadline = TInstant::Now() + reqTime;
            size_t cntRequestFail = 0;
            for (auto requestInfo : requestList) {
                if (WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline)) {
                    auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
                    UNIT_ASSERT_C(!result, result.Success());
                    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Connection refused");
                    ++cntRequestFail;
                } else {
                    break;
                }

                size_t cntRequestInternal = NetworkClient_->ListRequests().Success().size();
                UNIT_ASSERT_EQUAL_C(cntRequestInternal, cntRequest - cntRequestFail, cntRequestInternal);
            }

            UNIT_ASSERT_EQUAL_C(cntRequestFail, cntRequest, "Fail to complete all request in " << reqTime << ", complete only " << cntRequestFail);
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestHttpRequestWithTimeout) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            const size_t cntRequest = 10;
            auto router = MakeSimpleShared<TFakeBlockRouter>(DEFAULT_TIMEOUT * 10);
            AddHttpService(router);

            AddManyHttpRequests(2, cntRequest);
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), cntRequest, requestList.size());

            TDuration reqTime = DEFAULT_TIMEOUT * (cntRequest + 2);
            TInstant deadline = TInstant::Now() + reqTime;
            size_t cntRequestTimeout = 0;
            for (auto requestInfo : requestList) {
                if (WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline)) {
                    auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
                    UNIT_ASSERT_C(!result, result.Success());
                    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "timeout");
                    ++cntRequestTimeout;
                } else {
                    break;
                }

                size_t cntRequestInternal = NetworkClient_->ListRequests().Success().size();
                UNIT_ASSERT_EQUAL_C(cntRequestInternal, cntRequest - cntRequestTimeout, cntRequestInternal);
            }

            UNIT_ASSERT_EQUAL_C(cntRequestTimeout, cntRequest, "Timeout to complete all request in " << reqTime << ", complete only " << cntRequestTimeout);
            router->ReleaseBlock();
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestHttpRequestRemove) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            const size_t cntRequest = 10;
            auto blockRouter = MakeSimpleShared<TFakeBlockRouter>(DEFAULT_TIMEOUT * 10);
            AddHttpService(blockRouter);
            AddHttpService(MakeSimpleShared<TFakeSuccessRouter>());

            UNIT_ASSERT_C(cntRequest >= MAX_THREADS, "This test need at least MAX_THREADS requests");

            size_t cntRunning = 0;
            for (size_t i = 0; i < cntRequest; ++i) {
                AddHttpRequest("key" + ToString(i), "hash" + ToString(i), "info" + ToString(i), 0, ToString(i), TDuration::Max());
                if (cntRunning < MAX_THREADS) {
                    cntRunning += WaitRequest("key" + ToString(i), "hash" + ToString(i), TInstant::Now() + DEFAULT_TIMEOUT * 2, INetworkClient::ERequestState::RUNNING);
                }
            }
            UNIT_ASSERT_EQUAL_C(cntRunning, MAX_THREADS, "Not all threads are busy: " << cntRunning << " of " << MAX_THREADS);

            for (size_t i = 0; i < cntRequest; ++i) {
                NetworkClient_->RemoveRequest("key" + ToString(i));
            }
            UNIT_ASSERT_EQUAL_C(NetworkClient_->ListRequests().Success().size(), 0, "Request must be removed from internal state");

            AddHttpRequest("key", "hash", "info", 1);
            UNIT_ASSERT_C(WaitRequest("key", "hash", TInstant::Now() + DEFAULT_TIMEOUT * 2), "Request not completed");

            auto result = NetworkClient_->GetAndRemoveRequestResponse("key", "hash");
            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_EQUAL_C(result.Success(), "Data", result.Success());

            blockRouter->ReleaseBlock();
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestHttpRequestWrongHash) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            AddHttpService(MakeSimpleShared<TFakeSuccessRouter>());

            AddHttpRequest("key", "hash", "info");
            UNIT_ASSERT_C(WaitRequest("key", "hash", TInstant::Now() + DEFAULT_TIMEOUT * 2), "Request not completed");

            {
                // Check hash with state
                auto result = NetworkClient_->GetRequestState("key", "wrong_hash");
                UNIT_ASSERT_C(!result, "Get state must fail");
                UNIT_ASSERT_EQUAL_C(result.Error().Errno, ENetworkClientError::RequestHashMismatched, ToString(result.Error().Errno));
            }

            {
                // Check hash with response
                auto result = NetworkClient_->GetAndRemoveRequestResponse("key", "wrong_hash");
                UNIT_ASSERT_C(!result, "Get state must fail");
                UNIT_ASSERT_EQUAL_C(result.Error().Errno, ENetworkClientError::RequestHashMismatched, ToString(result.Error().Errno));
                UNIT_ASSERT_EQUAL_C(NetworkClient_->ListRequests().Success().size(), 1, "Request must stay in internal state");
            }

            AddHttpRequest("key", "other_hash", "other_info");
            UNIT_ASSERT_C(WaitRequest("key", "other_hash", TInstant::Now() + DEFAULT_TIMEOUT * 2), "Request not completed");

            {
                // Check override hash and info
                auto result = NetworkClient_->ListRequests();
                UNIT_ASSERT_C(result, result.Error().Message);
                UNIT_ASSERT_EQUAL_C(result.Success().size(), 1, result.Success().size());
                UNIT_ASSERT_EQUAL_C(result.Success()[0].AdditionalInfo_, "other_info", result.Success()[0].AdditionalInfo_);
            }

            {
                // Check state and response
                auto stateResult = NetworkClient_->GetRequestState("key", "other_hash");
                UNIT_ASSERT_C(stateResult, stateResult.Error().Message);
                UNIT_ASSERT_EQUAL_C(stateResult.Success(), INetworkClient::ERequestState::COMPLETED, ToString(stateResult.Success()));

                auto responseResult = NetworkClient_->GetAndRemoveRequestResponse("key", "other_hash");
                UNIT_ASSERT_C(responseResult, responseResult.Error().Message);
                UNIT_ASSERT_EQUAL_C(responseResult.Success(), "Data", responseResult.Success());

                UNIT_ASSERT_EQUAL_C(NetworkClient_->ListRequests().Success().size(), 0, "Request must be removed from internal state");
            }
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestCheckAndAddTcpRequestWithSuccess) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            NTesting::TPortHolder port = EnableTcpServer();

            AddTcpRequest("key1", "hash1", "info1", port, DEFAULT_TIMEOUT, false);
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), 1, requestList.size());

            auto requestInfo = requestList[0];
            const TString keySuffix = requestInfo.Key_.substr(strlen("key"));
            const TString hashSuffix = requestInfo.Hash_.substr(strlen("hash"));
            const TString infoSuffix = requestInfo.AdditionalInfo_.substr(strlen("info"));

            UNIT_ASSERT_EQUAL_C(keySuffix, hashSuffix, keySuffix << " != " << hashSuffix);
            UNIT_ASSERT_EQUAL_C(infoSuffix, hashSuffix, infoSuffix << " != " << hashSuffix);

            TDuration reqTime = DEFAULT_TIMEOUT * 2;
            TInstant deadline = TInstant::Now() + reqTime;
            UNIT_ASSERT(WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline));
            auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_EQUAL_C(result.Success(), "", result.Success());
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestCheckAndAddTcpRequestWithNoServer) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            NTesting::TPortHolder port = NTesting::GetFreePort();

            AddTcpRequest("key1", "hash1", "info1", port, DEFAULT_TIMEOUT, false);
            auto requestList = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL_C(requestList.size(), 1, requestList.size());

            auto requestInfo = requestList[0];
            const TString keySuffix = requestInfo.Key_.substr(strlen("key"));
            const TString hashSuffix = requestInfo.Hash_.substr(strlen("hash"));
            const TString infoSuffix = requestInfo.AdditionalInfo_.substr(strlen("info"));

            UNIT_ASSERT_EQUAL_C(keySuffix, hashSuffix, keySuffix << " != " << hashSuffix);
            UNIT_ASSERT_EQUAL_C(infoSuffix, hashSuffix, infoSuffix << " != " << hashSuffix);

            TDuration reqTime = DEFAULT_TIMEOUT * 2;
            TInstant deadline = TInstant::Now() + reqTime;
            UNIT_ASSERT(WaitRequest(requestInfo.Key_, requestInfo.Hash_, deadline));
            auto result = NetworkClient_->GetAndRemoveRequestResponse(requestInfo.Key_, requestInfo.Hash_);
            UNIT_ASSERT_C(!result, result.Success());
            UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Connection refused");
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestGetLocalHostName) {
    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            auto result = NetworkClient_->GetLocalHostName();

            UNIT_ASSERT_C((bool)result, ToString(result.Error()));
            UNIT_ASSERT_C(!result.Success().empty(), "Hostname must be non-empty");
        }
    };

    TTest().DoTest();
}

Y_UNIT_TEST(TestCheckAndAddRequestWhenNetworkRequestInitFail) {
    struct TFailNetworkRequest : public TMockNetworkRequest {
        TFailNetworkRequest()
        { }

        virtual TExpected<void, TNetworkClientError> Init() override {
            return TNetworkClientError(
                ENetworkClientError::RequestError
                , "error"
            );
        }
    };

    class TTest : public ITestNetworkClientCanon {
    public:
        TTest() : ITestNetworkClientCanon()
        {
        }

    private:
        void Test() final {
            auto result = ((TSimpleNetworkClient*)NetworkClient_.Get())->CheckAndAddRequest(
                "key"
                , "hash"
                , "info"
                , DEFAULT_TIMEOUT
                , new TFailNetworkRequest
            );

            UNIT_ASSERT(!result);
            UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "error");
        }
    };

    TTest().DoTest();
}

}

} // namespace NInfra::NPodAgent::NNetworkClientTest
