#include "path_holder.h"

#include <util/string/cast.h>
#include <util/string/builder.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent {


TPathHolder::TPathHolder(
    const TString& dom0PodRoot
    , const TMap<TString, TString>& virtualDisksToPlace
    , const TMap<TString, TString>& placesToDownloadVolumePath
    , const TString& volumeStorage
    , const TString& persistentStoragePrefix
    , const TString& layerPrefix
    , const TString& containersPrefix
    , const TString& portoLogFilesDir
    , const TString& rbindVolumeStorageDir
    , const bool isBoxAgentMode
)
    : Dom0PodRoot_(TPathHolder::PatchString(dom0PodRoot, ""))
    , VirtualDisksToPlace_(virtualDisksToPlace)
    , PlacesToDownloadVolumePath_(TPathHolder::PatchDownloadVolumesPaths(placesToDownloadVolumePath, NFs::CurrentWorkingDirectory()))
    , VolumeStorage_(TPathHolder::PatchString(volumeStorage, NFs::CurrentWorkingDirectory()))
    , PersistentStoragePrefix_(persistentStoragePrefix)
    , LayerPrefix_(layerPrefix)
    , ContainersPrefix_(containersPrefix)
    , PortoLogFilesDir_(TPathHolder::PatchString(portoLogFilesDir, ""))
    , RbindVolumeStorageDir_(TPathHolder::PatchString(rbindVolumeStorageDir, NFs::CurrentWorkingDirectory()))
    , IsBoxAgentMode_(isBoxAgentMode)
{
    Y_ENSURE(virtualDisksToPlace.size() == placesToDownloadVolumePath.size()
        , "Different number of virtual disks and places: virtual disks: " << virtualDisksToPlace.size()
        << ", places " << placesToDownloadVolumePath.size()
    );

    for (const auto& [virtualDisk, place] : virtualDisksToPlace) {
        Y_ENSURE(placesToDownloadVolumePath.contains(place), "Virtual disk '" << virtualDisk << "' correspond to non-existent place '" << place << "'");
        if (!place.empty()) {
            Y_ENSURE(place.StartsWith("///"), "Place specified without '///' at the beginning: '" << place << "'");
        }
    }

    CheckParam(PersistentStoragePrefix_);
    CheckParam(LayerPrefix_);
}

bool TPathHolder::HasVirtualDisk(const TString& virtualDisk) const {
    return VirtualDisksToPlace_.contains(virtualDisk);
}

bool TPathHolder::HasPlace(const TString& place) const {
    return PlacesToDownloadVolumePath_.contains(place);
}

TString TPathHolder::GetPlaceFromVirtualDisk(const TString& virtualDisk) const {
    Y_ENSURE(VirtualDisksToPlace_.contains(virtualDisk), "unknown virtual disk " << virtualDisk << "'");
    return VirtualDisksToPlace_.at(virtualDisk);
}

TString TPathHolder::GetDom0PlaceFromVirtualDisk(const TString& virtualDisk) const {
    const TString place = GetPlaceFromVirtualDisk(virtualDisk);
    if (place.empty()) {
        // Return empty place as is
        return place;
    } else {
        // Sanity check
        // Must be validated in the constructor
        Y_ENSURE(place.StartsWith("///"), "Place specified without '///' at the beginning: '" << place << "'");
        // Return Dom0PodRoot_ + place without slashes
        return TStringBuilder() << Dom0PodRoot_ << place.substr(3);
    }
}

TString TPathHolder::GetPathFromPlace(const TString& place) const {
    Y_ENSURE(PlacesToDownloadVolumePath_.contains(place), "unknown place " << place << "'");
    return PlacesToDownloadVolumePath_.at(place);
}

TString TPathHolder::GetLayersDirectory(const TString& place) const {
    return TStringBuilder() << GetPathFromPlace(place) << LAYER_RESOURCE_TAG;
}

TString TPathHolder::GetStaticResourcesDirectory(const TString& place) const {
    return TStringBuilder() << GetPathFromPlace(place) << STATIC_RESOURCE_TAG;
}

TPortoContainerName TPathHolder::GetResourceGangMetaContainer() const {
    return {TStringBuilder() << ContainersPrefix_ << RESOURCE_GANG_META_CONTAINER};
}

TPortoContainerName TPathHolder::GetDownloadContainerLookupMask() const {
    return {GetResourceGangMetaContainer(), TPortoContainerName::NoEscape("*_download")};
}

TPortoContainerName TPathHolder::GetVerifyContainerLookupMask() const {
    return {GetResourceGangMetaContainer(), TPortoContainerName::NoEscape("*_verify")};
}

TString TPathHolder::GetStaticResourceDirectoryFromHash(const TString& staticResourceDownloadHash, const TString& place) const {
    CheckParam(staticResourceDownloadHash);
    return TStringBuilder() << GetStaticResourcesDirectory(place) << "/" << staticResourceDownloadHash;
}

TString TPathHolder::GetStaticResourceDownloadDirectoryFromHash(const TString& staticResourceDownloadHash, const TString& place) const {
    CheckParam(staticResourceDownloadHash);
    return TStringBuilder() << GetStaticResourceDirectoryFromHash(staticResourceDownloadHash, place) << "/downloaded";
}

TString TPathHolder::GetFinalStaticResourcePathFromHash(const TString& staticResourceDownloadHash, const TString& place) const {
    CheckParam(staticResourceDownloadHash);
    return TStringBuilder() << GetStaticResourceDirectoryFromHash(staticResourceDownloadHash, place) << "/downloaded_result";
}

TString TPathHolder::GetStaticResourceAccessModeFlagPathFromHash(const TString& staticResourceDownloadHash, const TString& place, const TString& fileAccessMode) const {
    CheckParam(staticResourceDownloadHash);
    return TStringBuilder() << GetStaticResourceAccessModeFlagDirectoryFromHash(staticResourceDownloadHash, place) << "/" << fileAccessMode;
}

TString TPathHolder::GetStaticResourceAccessModeFlagDirectoryFromHash(const TString& staticResourceDownloadHash, const TString& place) const {
    CheckParam(staticResourceDownloadHash);
    return TStringBuilder() << GetStaticResourceDirectoryFromHash(staticResourceDownloadHash, place) << "/access_mode";
}

TString TPathHolder::GetLayerDirectoryFromHash(const TString& layerDownloadHash, const TString& place) const {
    CheckParam(layerDownloadHash);
    return TStringBuilder() << GetLayersDirectory(place) << "/" << layerDownloadHash;
}

TString TPathHolder::GetLayerDownloadDirectoryFromHash(const TString& layerDownloadHash, const TString& place) const {
    CheckParam(layerDownloadHash);
    return TStringBuilder() << GetLayerDirectoryFromHash(layerDownloadHash, place) << "/downloaded";
}

TString TPathHolder::GetFinalLayerPathFromHash(const TString& layerDownloadHash, const TString& place) const {
    CheckParam(layerDownloadHash);
    return TStringBuilder() << GetLayerDirectoryFromHash(layerDownloadHash, place) << "/downloaded_result/layer_link";
}

TString TPathHolder::GetVolumePath(const TString& volumeId) const {
    CheckParam(volumeId);
    return TStringBuilder() << VolumeStorage_ << VOLUME_PREFIX << volumeId;
}

TString TPathHolder::GetVolumePersistentStorage(const TString& volumeId) const {
    CheckParam(volumeId);
    return TStringBuilder() << PersistentStoragePrefix_ << VOLUME_PREFIX << volumeId;
}

TString TPathHolder::GetVolumeStoragePath() const {
    return VolumeStorage_;
}

TString TPathHolder::GetRbindVolumeStorage(const TString rbindVolumeRef) const {
    CheckParam(rbindVolumeRef);
    return TStringBuilder() << RbindVolumeStorageDir_ << rbindVolumeRef;
}

TString TPathHolder::GetRbindVolumeStorageDir() const {
    return RbindVolumeStorageDir_;
}

TString TPathHolder::GetBoxRootfsPath(const TString& boxId) const {
    CheckParam(boxId);
    return TStringBuilder() << VolumeStorage_ << ROOTFS_PREFIX << boxId;
}

TString TPathHolder::GetBoxRootfsPersistentStorage(const TString& boxId) const {
    CheckParam(boxId);
    return TStringBuilder() << PersistentStoragePrefix_ << ROOTFS_PREFIX << boxId;
}

TString TPathHolder::GetWorkloadLogsFileName(const TString& workloadId, const TString& typeofLog) const {
    CheckParam(workloadId);
    CheckParam(typeofLog);
    Y_ENSURE(typeofLog.find(SEPARATOR) == TString::npos, typeofLog << " contains separator '" << SEPARATOR << "'");
    return TStringBuilder() << workloadId << SEPARATOR << typeofLog << LOG_FILE_EXTENSION;
}

TString TPathHolder::GetWorkloadLogsFilePathInBox(const TString& workloadId, const TString& typeofLog) const {
    return TStringBuilder() << PortoLogFilesDir_ << GetWorkloadLogsFileName(workloadId, typeofLog);
}

TString TPathHolder::GetWorkloadLogsFilePathAtBoxRootfs(const TString& boxId, const TString& workloadId, const TString& typeofLog) const {
    CheckParam(boxId);
    CheckParam(workloadId);
    CheckParam(typeofLog);
    Y_ENSURE(typeofLog.find(SEPARATOR) == TString::npos, typeofLog << " contains separator '" << SEPARATOR << "'");
    return TStringBuilder() << GetBoxRootfsPath(boxId) << GetWorkloadLogsFilePathInBox(workloadId, typeofLog);
}

TString TPathHolder::GetWorkloadLogsFilePathAtLogsVolumePath(const TString& workloadId, const TString& typeofLog) const {
    CheckParam(workloadId);
    CheckParam(typeofLog);
    Y_ENSURE(typeofLog.find(SEPARATOR) == TString::npos, typeofLog << " contains separator '" << SEPARATOR << "'");
    return TStringBuilder() << GetVolumePath(LOGS_VOLUME_ID) << "/" << GetWorkloadLogsFileName(workloadId, typeofLog);
}

TString TPathHolder::GetWorkloadLogsMetaPathAtLogsVolumePath() const {
    return TStringBuilder() << GetVolumePath(LOGS_VOLUME_ID);
}

TPortoContainerName TPathHolder::GetLayerContainerWithNameFromHash(const TString& layerDownloadHash, const TString& name) const {
    CheckParam(layerDownloadHash);
    CheckParam(name);
    Y_ENSURE(name.find(SEPARATOR) == TString::npos, name << " contains separator '" << SEPARATOR << "'");
    return TPortoContainerName::NoEscape(ContainersPrefix_ + RESOURCE_GANG_META_CONTAINER + "/" + TString(LAYER_RESOURCE_TAG) + SEPARATOR + layerDownloadHash + SEPARATOR + name);
}

TPortoContainerName TPathHolder::GetStaticResourceContainerWithNameFromHash(const TString& staticResourceDownloadHash, const TString& name) const {
    CheckParam(staticResourceDownloadHash);
    CheckParam(name);
    Y_ENSURE(name.find(SEPARATOR) == TString::npos, name << " contains separator '" << SEPARATOR << "'");
    return TPortoContainerName::NoEscape(ContainersPrefix_ + RESOURCE_GANG_META_CONTAINER + "/" + TString(STATIC_RESOURCE_TAG) + SEPARATOR + staticResourceDownloadHash + SEPARATOR + name);
}

TPortoContainerName TPathHolder::GetBoxContainer(const TString& boxId) const {
    CheckParam(boxId);
    return {ContainersPrefix_ + BOX_PREFIX + boxId};
}

TPortoContainerName TPathHolder::GetBoxInitContainer(const TString& boxId, size_t initId) const {
    CheckParam(boxId);
    return {GetBoxContainer(boxId), BOX_INIT_PREFIX + ToString(initId)};
}

TPortoContainerName TPathHolder::GetWorkloadContainerWithName(const TString& boxId, const TString& workloadId, const TString& name) const {
    CheckParam(boxId);
    CheckParam(workloadId);
    CheckParam(name);
    Y_ENSURE(name.find(SEPARATOR) == TString::npos, name << " contains separator '" << SEPARATOR << "'");
    if (IsBoxAgentMode_) {
        Y_ENSURE(boxId.empty(), "Workload must not have box when pod agent runs in box agent mode, but it has: '" << boxId << "'");
        return {ContainersPrefix_ + WORKLOAD_PREFIX + workloadId + SEPARATOR + name};
    } else {
        return {GetBoxContainer(boxId), WORKLOAD_PREFIX + workloadId + SEPARATOR + name};
    }
}

TPortoContainerName TPathHolder::GetWorkloadInitContainer(const TString& boxId, const TString& workloadId, size_t initId) const {
    CheckParam(boxId);
    CheckParam(workloadId);
    if (IsBoxAgentMode_) {
        Y_ENSURE(boxId.empty(), "Workload must not have box when pod agent runs in box agent mode, but it has: '" << boxId << "'");
        return {ContainersPrefix_ + WORKLOAD_PREFIX + workloadId + SEPARATOR + BOX_INIT_PREFIX + ToString(initId)};
    } else {
        return {GetBoxContainer(boxId), WORKLOAD_PREFIX + workloadId + SEPARATOR + BOX_INIT_PREFIX + ToString(initId)};
    }
}

TPortoContainerName TPathHolder::GetWorkloadContainerLookupMask(const TString& boxId, const TString& workloadId) const {
    CheckParam(boxId);
    CheckParam(workloadId);
    if (IsBoxAgentMode_) {
        Y_ENSURE(boxId.empty(), "Workload must not have box when pod agent runs in box agent mode, but it has: '" << boxId << "'");
        return TPortoContainerName::NoEscape(ContainersPrefix_ + WORKLOAD_PREFIX + workloadId + "***");
    } else {
        return {GetBoxContainer(boxId), TPortoContainerName::NoEscape(WORKLOAD_PREFIX + workloadId + "***")};
    }
}

TIdExtractionResult TPathHolder::ExtractVolumeIdFromPath(const TString& path) const {
    if (path.StartsWith(VolumeStorage_ + VOLUME_PREFIX)) {
        TString id = path.substr((VolumeStorage_ + VOLUME_PREFIX).size());
        if (id != "") {
            return id;
        }
    }
    return TIdExtractionError{
        TStringBuilder() << "Unable to extract VolumeId from path:"
                         << "  " << path
    };
}

TIdExtractionResult TPathHolder::ExtractIdOrHashFromPath(const TString& path, const TStringBuf prefix) const {
    if (path.StartsWith(prefix)) {
        size_t pos = path.find_first_of('/', prefix.size());
        TString id = (pos == TString::npos) ? path.substr(prefix.size())
                                            : path.substr(prefix.size(), pos - prefix.size());
        if (id != "") {
            return id;
        }
    }
    return TIdExtractionError{
        TStringBuilder() << "Unable to extract id or hash from path: " << path
    };
}

TIdExtractionResult TPathHolder::ExtractBoxIdFromPath(const TString& path) const {
    const TString prefix = VolumeStorage_ + ROOTFS_PREFIX;
    return ExtractIdOrHashFromPath(path, prefix);
}

TIdExtractionResult TPathHolder::ExtractStaticResourceHashFromPath(const TString& path, const TString& place) const {
    const TString prefix = GetStaticResourcesDirectory(place) + "/";
    return ExtractIdOrHashFromPath(path, prefix);
}

TIdExtractionResult TPathHolder::ExtractIdOrHashFromContainer(const TPortoContainerName& containerName, const TStringBuf prefix) const {
    TString containerStringName = TString(containerName);
    if (containerStringName.StartsWith(prefix)) {
        const size_t pos = containerStringName.find_last_of(SEPARATOR);
        if (pos != TString::npos && pos > prefix.length()) {
            TString id = containerStringName.substr(prefix.length(), pos - prefix.length());
            if (id != "") {
                return id;
            }
        }
    }
    return TIdExtractionError{
        TStringBuilder()
            << "Unable to extract id or hash from container"
            << " '" << containerStringName << "'"
            << " with prefix " << prefix
    };
}

TIdExtractionResult TPathHolder::ExtractLayerHashFromContainer(const TPortoContainerName& containerName) const {
    static const size_t prefixLen = strlen(LAYER_RESOURCE_TAG);
    return ExtractIdOrHashFromContainer(containerName, ContainersPrefix_ + RESOURCE_GANG_META_CONTAINER + "/" + TString(LAYER_RESOURCE_TAG, prefixLen) + SEPARATOR);
}

TIdExtractionResult TPathHolder::ExtractStaticResourceHashFromContainer(const TPortoContainerName& containerName) const {
    static const size_t prefixLen = strlen(STATIC_RESOURCE_TAG);
    return ExtractIdOrHashFromContainer(containerName, ContainersPrefix_ + RESOURCE_GANG_META_CONTAINER + "/" + TString(STATIC_RESOURCE_TAG, prefixLen) + SEPARATOR);
}

TIdExtractionResult TPathHolder::ExtractBoxIdFromContainer(const TPortoContainerName& containerName) const {
    const TString boxContainerPrefix = ContainersPrefix_ + BOX_PREFIX;
    const size_t boxPrefixLen = boxContainerPrefix.size();
    const TString containerStringName = TString(containerName);

    if (containerStringName.StartsWith(boxContainerPrefix)) {
        const size_t pos = containerStringName.find_first_of('/', boxPrefixLen);
        if (pos == TString::npos) {
            if (containerStringName.length() > boxPrefixLen) {
                return containerStringName.substr(boxPrefixLen);
            }
        } else if (pos > boxPrefixLen) {
            return containerStringName.substr(boxPrefixLen, pos - boxPrefixLen);
        }
    }

    return TIdExtractionError{
        TStringBuilder() << "Unable to extract BoxId from container '" << containerStringName << "'"
    };
}

TExpected<TPortoContainerName, TIdExtractionError> TPathHolder::ExtractBoxSimpleSubContainer(const TPortoContainerName& containerName) const {
    const TString boxId = OUTCOME_TRYX(ExtractBoxIdFromContainer(containerName));
    if (containerName.IsSimple()) {
        return TIdExtractionError{
            TStringBuilder() << "Unable to extract box subcontainer from container '" << containerName << "' which is simple"
        };
    }
    const TPortoContainerName child = containerName.GetChild();
    if (!child.IsSimple()) {
        return TIdExtractionError{
            TStringBuilder() << "Unable to extract box simple subcontainer from container '" << containerName << "' which is not simple"
        };
    }
    return child;
}

TIdExtractionResult TPathHolder::ExtractBoxInitIdFromContainer(const TPortoContainerName& containerName) const {
    const TPortoContainerName child = OUTCOME_TRYX(ExtractBoxSimpleSubContainer(containerName));

    static const size_t boxInitPrefixLen = strlen(BOX_INIT_PREFIX);
    if (child.StartsWith(BOX_INIT_PREFIX)) {
        return TString(child).substr(boxInitPrefixLen);
    }

    return TIdExtractionError{
        TStringBuilder() << "Unable to extract BoxInitId from container '" << containerName << "'"
    };
}

TIdExtractionResult TPathHolder::ExtractWorkloadIdFromContainer(const TPortoContainerName& containerName) const {
    if (IsBoxAgentMode_) {
        if (!containerName.IsSimple()) {
            return TIdExtractionError {
                TStringBuilder()
                    << "Workload container must be simple when pod agent runs in box agent mode, but it is '"
                    << TString(containerName) << "'"
            };
        }
        const TString containerPrefix = ContainersPrefix_ + WORKLOAD_PREFIX;
        return ExtractIdOrHashFromContainer(containerName, containerPrefix);
    } else {
        const TPortoContainerName child = OUTCOME_TRYX(ExtractBoxSimpleSubContainer(containerName));
        static const size_t prefixLen = strlen(WORKLOAD_PREFIX);
        return ExtractIdOrHashFromContainer(child, TStringBuf(WORKLOAD_PREFIX, prefixLen));
    }
}

TIdExtractionResult TPathHolder::ExtractVolumeIdFromStorage(const TString& storageName) const {
    const TString prefix = PersistentStoragePrefix_ + TString(VOLUME_PREFIX);
    if (storageName.StartsWith(prefix)) {
        const TString id = storageName.substr(prefix.size());
        if (id != "") {
            return id;
        }
    }
    return TIdExtractionError{
        TStringBuilder() << "Unable to extract VolumeId from storage:"
                         << "  " << storageName
    };
}

TIdExtractionResult TPathHolder::ExtractBoxIdFromStorage(const TString& storageName) const {
    const TString prefix = PersistentStoragePrefix_ + TString(ROOTFS_PREFIX);
    if (storageName.StartsWith(prefix)) {
        const TString id = storageName.substr(prefix.size());
        if (id != "")
            return id;
    }
    return TIdExtractionError{
        TStringBuilder() << "Unable to extract BoxId from storage:"
                         << "  " << storageName
    };
}

TIdExtractionResult TPathHolder::ExtractWorkloadSuffixFromContainer(const TPortoContainerName& containerName) const {
    TPortoContainerName child = containerName;
    if (!IsBoxAgentMode_) {
        child = OUTCOME_TRYX(ExtractBoxSimpleSubContainer(containerName));
    } else {
        if (!containerName.IsSimple()) {
            return TIdExtractionError {
                TStringBuilder()
                    << "Workload container must be simple when pod agent runs in box agent mode, but it is '"
                    << TString(containerName) << "'"
            };
        }
    }

    static const size_t workloadPrefixLen = strlen(WORKLOAD_PREFIX);
    TString workloadId = OUTCOME_TRYX(ExtractIdOrHashFromContainer(child, TStringBuf(WORKLOAD_PREFIX, workloadPrefixLen)));

    const size_t prefixLen = workloadPrefixLen + workloadId.length() + SEPARATOR_SIZE;
    TString suffix = TString(child).substr(prefixLen);

    if (!suffix.empty()) {
        return suffix;
    }

    return TIdExtractionError {
        TStringBuilder()
            << "Unable to extract suffix from container"
            << " '" << containerName << "'"
    };
}

TString TPathHolder::GetLayerNameFromHash(const TString& layerDownloadHash) const {
    return LayerPrefix_ + layerDownloadHash;
}

TString TPathHolder::PatchString(const TString& string, const TString& prefix) {
    TString copyString = string;

    if (!copyString.StartsWith('/')) {
        copyString.prepend(prefix + '/');
    }
    if (!copyString.EndsWith('/')) {
        copyString.append('/');
    }

    return copyString;
}

TMap<TString, TString> TPathHolder::PatchDownloadVolumesPaths(const TMap<TString, TString>& placesToDownloadVolumePath, const TString& prefix) {
    TMap<TString, TString> patchedPlaces;

    for (const auto& it : placesToDownloadVolumePath) {
        patchedPlaces[it.first] = PatchString(it.second, prefix);
    }

    return patchedPlaces;
}

void TPathHolder::CheckParam(const TString& param) const {
    size_t pos = param.find_first_not_of(PORTO_ALLOW_SYMBOLS);
    Y_ENSURE(pos == TString::npos, param << " doesn't fit porto name template: char " << param[pos] << " at position " << pos);
}

} // namespace NInfra::NPodAgent

