#pragma once

#include <infra/pod_agent/libs/porto_client/porto_types.h>
#include <infra/libs/outcome/result.h>

namespace NInfra::NPodAgent {

struct TIdExtractionError {
    TString Message = "";

    bool operator==(const TIdExtractionError& rhs) const {
            return this->Message == rhs.Message;
    }
};

using TIdExtractionResult = TExpected<TString, TIdExtractionError>;

class TPathHolder;
using TPathHolderPtr = TAtomicSharedPtr<TPathHolder>;

class TPathHolder {
public:
    TPathHolder(
        const TString& dom0PodRoot
        , const TMap<TString, TString>& virtualDisksToPlace
        , const TMap<TString, TString>& placesToDownloadVolumePath
        , const TString& volumeStorage
        , const TString& persistentStoragePrefix
        , const TString& layerPrefix
        , const TString& containersPrefix
        , const TString& portoLogFilesDir
        , const TString& rbindVolumeStorageDir
        , const bool isBoxAgentMode = false
    );

    bool HasVirtualDisk(const TString& virtualDisk) const;
    bool HasPlace(const TString& place) const;
    TString GetPlaceFromVirtualDisk(const TString& virtualDisk) const;
    // We need this isolation breaking method
    // because we have isolated box inside isolated pod we can't simply use some porto magic like "///" or "./",
    // we need to specify box meta container place and io limits with dom0 prefix
    TString GetDom0PlaceFromVirtualDisk(const TString& virtualDisk) const;
    TString GetPathFromPlace(const TString& place) const;

    TString GetLayersDirectory(const TString& place) const;
    TString GetStaticResourcesDirectory(const TString& place) const;
    TPortoContainerName GetResourceGangMetaContainer() const;
    TPortoContainerName GetDownloadContainerLookupMask() const;
    TPortoContainerName GetVerifyContainerLookupMask() const;

    TString GetStaticResourceDirectoryFromHash(const TString& staticResourceDownloadHash, const TString& place) const;
    // Special directory for downloading static resource files
    // Files by this path may still be downloading
    // All files remain in it as is (without modifications or renaming)
    TString GetStaticResourceDownloadDirectoryFromHash(const TString& staticResourceDownloadHash, const TString& place) const;
    // Symbolic link to the static resource
    // When access mode flag exists and final static resource path exists - box can use static resource by final resource path
    TString GetFinalStaticResourcePathFromHash(const TString& staticResourceDownloadHash, const TString& place) const;
    // Symbolic link to the static resource, that indicates that access mode to static resource were set
    // When access mode flag exists and final static resource path exists - box can use static resource by final resource path
    TString GetStaticResourceAccessModeFlagPathFromHash(const TString& staticResourceDownloadHash, const TString& place, const TString& fileAccessMode) const;
    TString GetStaticResourceAccessModeFlagDirectoryFromHash(const TString& staticResourceDownloadHash, const TString& place) const;

    TString GetLayerDirectoryFromHash(const TString& layerDownloadHash, const TString& place) const;
    // Special directory for downloading the layer
    // Files by this path may still be downloading
    // The source layer file remains in it as is (without modifications or renaming)
    TString GetLayerDownloadDirectoryFromHash(const TString& layerDownloadHash, const TString& place) const;
    // Symbolic link to the layer
    // File by this path is guaranteed to be ready for import
    TString GetFinalLayerPathFromHash(const TString& layerDownloadHash, const TString& place) const;

    TString GetBoxRootfsPath(const TString& boxId) const;
    TString GetBoxRootfsPersistentStorage(const TString& boxId) const;

    TString GetVolumePath(const TString& volumeId) const;
    TString GetVolumePersistentStorage(const TString& volumeId) const;
    TString GetVolumeStoragePath() const;

    TString GetRbindVolumeStorage(const TString rbindVolumeRef) const;
    TString GetRbindVolumeStorageDir() const;

    // Returns the path to the log file relative to the root of the box, since workload start container is nested container of the box meta container
    TString GetWorkloadLogsFilePathInBox(const TString& workloadId, const TString& typeofLog) const;
    TString GetWorkloadLogsFileName(const TString& workloadId, const TString& typeofLog) const;
    TString GetWorkloadLogsFilePathAtBoxRootfs(const TString& boxId, const TString& workloadId, const TString& typeofLog) const;
    TString GetWorkloadLogsMetaPathAtLogsVolumePath() const;
    TString GetWorkloadLogsFilePathAtLogsVolumePath(const TString& workloadId, const TString& typeofLog) const;


    TPortoContainerName GetBoxContainer(const TString& boxId) const;
    TPortoContainerName GetBoxInitContainer(const TString& boxId, size_t initId) const;
    TPortoContainerName GetLayerContainerWithNameFromHash(const TString& layerDownloadHash, const TString& containerNameSuffix) const;
    TPortoContainerName GetStaticResourceContainerWithNameFromHash(const TString& staticResourceDownloadHash, const TString& containerNameSuffix) const;
    TPortoContainerName GetWorkloadContainerWithName(const TString& boxId, const TString& workloadId, const TString& containerNameSuffix) const;
    TPortoContainerName GetWorkloadInitContainer(const TString& boxId, const TString& workloadId, size_t initId) const;
    TPortoContainerName GetWorkloadContainerLookupMask(const TString& boxId, const TString& workloadId) const;

    TIdExtractionResult ExtractVolumeIdFromPath(const TString& path) const;
    TIdExtractionResult ExtractBoxIdFromPath(const TString& path) const;
    TIdExtractionResult ExtractStaticResourceHashFromPath(const TString& path, const TString& place) const;

    TIdExtractionResult ExtractLayerHashFromContainer(const TPortoContainerName& containerName) const;
    TIdExtractionResult ExtractStaticResourceHashFromContainer(const TPortoContainerName& containerName) const;
    TIdExtractionResult ExtractBoxIdFromContainer(const TPortoContainerName& containerName) const;
    TIdExtractionResult ExtractBoxInitIdFromContainer(const TPortoContainerName& containerName) const;
    TIdExtractionResult ExtractWorkloadIdFromContainer(const TPortoContainerName& containerName) const;
    TIdExtractionResult ExtractVolumeIdFromStorage(const TString& storageName) const;
    TIdExtractionResult ExtractBoxIdFromStorage(const TString& storageName) const;

    TIdExtractionResult ExtractWorkloadSuffixFromContainer(const TPortoContainerName& containerName) const;

    TString GetLayerNameFromHash(const TString& layerDownloadHash) const;

private:
    TExpected<TPortoContainerName, TIdExtractionError> ExtractBoxSimpleSubContainer(const TPortoContainerName& containerName) const;
    TIdExtractionResult ExtractIdOrHashFromPath(const TString& path, const TStringBuf prefix) const;
    TIdExtractionResult ExtractIdOrHashFromContainer(const TPortoContainerName& containerName, const TStringBuf prefix) const;

    static TString PatchString(const TString& string, const TString& prefix);
    static TMap<TString, TString> PatchDownloadVolumesPaths(const TMap<TString, TString>& placesToDownloadVolumePath, const TString& prefix);

    void CheckParam(const TString& param) const;

public:
    static constexpr const char* LOG_FILE_EXTENSION = ".portolog";
    static constexpr const char* LOGS_VOLUME_ID = "logbroker_push_agent_porto_logs_volume_id";

private:
    static constexpr const char SEPARATOR = '_';
    static constexpr const i32 SEPARATOR_SIZE = 1;

    // from https://bb.yandex-team.ru/projects/PORTO/repos/porto/browse/porto.md
    static constexpr const char* PORTO_ALLOW_SYMBOLS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-@:.";

    static constexpr const char* RESOURCE_GANG_META_CONTAINER = "resource_gang_meta";
    static constexpr const char* LAYER_RESOURCE_TAG = "layer";
    static constexpr const char* STATIC_RESOURCE_TAG = "static";

    static constexpr const char* ROOTFS_PREFIX = "rootfs_";
    static constexpr const char* VOLUME_PREFIX = "volume_";
    static constexpr const char* WORKLOAD_PREFIX = "workload_";
    static constexpr const char* BOX_PREFIX = "box_";
    static constexpr const char* BOX_INIT_PREFIX = "init";

    const TString Dom0PodRoot_;
    const TMap<TString, TString> VirtualDisksToPlace_;
    const TMap<TString, TString> PlacesToDownloadVolumePath_;
    const TString VolumeStorage_;
    const TString PersistentStoragePrefix_;
    const TString LayerPrefix_;
    const TString ContainersPrefix_;
    const TString PortoLogFilesDir_;
    const TString RbindVolumeStorageDir_;
    const bool IsBoxAgentMode_;
};

} // namespace NInfra::NPodAgent

