#include "path_holder.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestPathHolder {

TString AppendToCwd(const TString& path) {
    const TString cwd = NFs::CurrentWorkingDirectory();
    return TStringBuilder() << cwd << path;
}

Y_UNIT_TEST_SUITE(PathHolderSuite) {

Y_UNIT_TEST(PathGenerationShouldWorkAsExpected) {
    {
        TMap<TString, TString> virtualDisksToPlace = {
            {"virtual_disk__0", ""}
            , {"virtual_disk_1", "///other_place"}
        };
        TMap<TString, TString> placesToDownloadVolumePath = {
            {"", "/resource_storage"}
            , {"///other_place", "/other_resource_storage"}
        };
        TPathHolderPtr pathHolder = new TPathHolder(
            "/root"
            , virtualDisksToPlace
            , placesToDownloadVolumePath
            , "/volume_storage"
            , "persistent_storage"
            , ""
            , ""
            , "porto_logs_storage"
            , "/rbind_volumes_storage"
        );

        for (const auto& it: placesToDownloadVolumePath) {
            // Layer
            UNIT_ASSERT_EQUAL_C(pathHolder->GetLayersDirectory(it.first), TStringBuilder() << it.second << "/layer", pathHolder->GetLayersDirectory(""));
            UNIT_ASSERT_EQUAL(pathHolder->GetLayerDirectoryFromHash("hash", it.first), TStringBuilder() << it.second << "/layer/hash");
            UNIT_ASSERT_EQUAL(pathHolder->GetLayerDownloadDirectoryFromHash("hash", it.first), TStringBuilder() << it.second << "/layer/hash/downloaded");
            UNIT_ASSERT_EQUAL(pathHolder->GetFinalLayerPathFromHash("hash", it.first), TStringBuilder() << it.second << "/layer/hash/downloaded_result/layer_link");

            // Static resource
            UNIT_ASSERT_EQUAL_C(pathHolder->GetStaticResourcesDirectory(it.first), TStringBuilder() << it.second << "/static", pathHolder->GetStaticResourcesDirectory(""));
            UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourceDirectoryFromHash("hash", it.first), TStringBuilder() << it.second << "/static/hash");
            UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourceDownloadDirectoryFromHash("hash", it.first), TStringBuilder() << it.second << "/static/hash/downloaded");
            UNIT_ASSERT_EQUAL(pathHolder->GetFinalStaticResourcePathFromHash("hash", it.first), TStringBuilder() << it.second << "/static/hash/downloaded_result");
            UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourceAccessModeFlagDirectoryFromHash("hash", it.first), TStringBuilder() << it.second << "/static/hash/access_mode");
            UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourceAccessModeFlagPathFromHash("hash", it.first, "664"), TStringBuilder() << it.second << "/static/hash/access_mode/664");

            // Volume
            UNIT_ASSERT_EQUAL(pathHolder->GetVolumePath("id"), "/volume_storage/volume_id");
            UNIT_ASSERT_EQUAL(pathHolder->GetVolumePersistentStorage("id"), "persistent_storagevolume_id");
            UNIT_ASSERT_EQUAL(pathHolder->GetVolumeStoragePath(), "/volume_storage/");

            // Rbind volume
            UNIT_ASSERT_EQUAL(pathHolder->GetRbindVolumeStorage("rbind_volume_ref"), "/rbind_volumes_storage/rbind_volume_ref");
            UNIT_ASSERT_EQUAL(pathHolder->GetRbindVolumeStorageDir(), "/rbind_volumes_storage/");

            // Box
            UNIT_ASSERT_EQUAL(pathHolder->GetBoxRootfsPath("id"), "/volume_storage/rootfs_id");
            UNIT_ASSERT_EQUAL(pathHolder->GetBoxRootfsPersistentStorage("id"), "persistent_storagerootfs_id");

            // Workload
            UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFileName("my_workload", "stdout"), "my_workload_stdout.portolog");
            UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathInBox("my_workload", "stdout"), "/porto_logs_storage/my_workload_stdout.portolog");
            UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathAtBoxRootfs("id", "my_workload", "stdout"), "/volume_storage/rootfs_id/porto_logs_storage/my_workload_stdout.portolog");

            UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathAtLogsVolumePath("my_workload", "stdout"), "/volume_storage/volume_logbroker_push_agent_porto_logs_volume_id/my_workload_stdout.portolog");
            UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsMetaPathAtLogsVolumePath(), "/volume_storage/volume_logbroker_push_agent_porto_logs_volume_id");
        }
    }

    {
        TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "", "", "");

        // Layer
        UNIT_ASSERT_EQUAL(pathHolder->GetLayersDirectory(""), AppendToCwd("/layer"));
        UNIT_ASSERT_EQUAL(pathHolder->GetLayerDirectoryFromHash("hash", ""), AppendToCwd("/layer/hash"));
        UNIT_ASSERT_EQUAL(pathHolder->GetLayerDownloadDirectoryFromHash("hash", ""), AppendToCwd("/layer/hash/downloaded"));
        UNIT_ASSERT_EQUAL(pathHolder->GetFinalLayerPathFromHash("hash", ""), AppendToCwd("/layer/hash/downloaded_result/layer_link"));

        // Static resource
        UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourcesDirectory(""), AppendToCwd("/static"));
        UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourceDirectoryFromHash("hash", ""), AppendToCwd("/static/hash"));
        UNIT_ASSERT_EQUAL(pathHolder->GetStaticResourceDownloadDirectoryFromHash("hash", ""), AppendToCwd("/static/hash/downloaded"));
        UNIT_ASSERT_EQUAL(pathHolder->GetFinalStaticResourcePathFromHash("hash", ""), AppendToCwd("/static/hash/downloaded_result"));

        // Volume
        UNIT_ASSERT_EQUAL(pathHolder->GetVolumePath("id"), AppendToCwd("/volume_id"));
        UNIT_ASSERT_EQUAL(pathHolder->GetVolumePersistentStorage("id"), "volume_id");
        UNIT_ASSERT_EQUAL(pathHolder->GetVolumeStoragePath(), AppendToCwd("/"));

        // Rbind volume
        UNIT_ASSERT_EQUAL(pathHolder->GetRbindVolumeStorage("rbind_volume_ref"), AppendToCwd("/rbind_volume_ref"));
        UNIT_ASSERT_EQUAL(pathHolder->GetRbindVolumeStorageDir(), AppendToCwd("/"));

        // Box
        UNIT_ASSERT_EQUAL(pathHolder->GetBoxRootfsPath("id"), AppendToCwd("/rootfs_id"));
        UNIT_ASSERT_EQUAL(pathHolder->GetBoxRootfsPersistentStorage("id"), "rootfs_id");

        // Workload
        UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFileName("my_workload", "stdout"), "my_workload_stdout.portolog");
        UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathInBox("my_workload", "stdout"), "/my_workload_stdout.portolog");
        UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathAtBoxRootfs("id", "my_workload", "stdout"), AppendToCwd("/rootfs_id/my_workload_stdout.portolog"));

        UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathAtLogsVolumePath("my_workload", "stdout"), AppendToCwd("/volume_logbroker_push_agent_porto_logs_volume_id/my_workload_stdout.portolog"));
        UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsMetaPathAtLogsVolumePath(), AppendToCwd("/volume_logbroker_push_agent_porto_logs_volume_id"));
    }
}

Y_UNIT_TEST(LayerName) {
    const TString layerPrefix = "TestLayerPrefix_";
    const TString layerHash = "my_layer_hash";
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", layerPrefix, "", "", "");
    UNIT_ASSERT_EQUAL(pathHolder->GetLayerNameFromHash(layerHash), layerPrefix + layerHash);
}

Y_UNIT_TEST(TestVolumeIdFromPathExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", "/resource_storage"}}, "/volume_storage", "persistent_storage", "", "", "", "");
    {
        const TString id = "my_volume";
        const TString path = pathHolder->GetVolumePath(id);
        auto extractionResult = pathHolder->ExtractVolumeIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString path = "path";
        auto extractionResult = pathHolder->ExtractVolumeIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract VolumeId from path:  path";
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, errorMessage);
    }

    {
        const TString path = "/volume_storage";
        auto extractionResult = pathHolder->ExtractVolumeIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract VolumeId from path:  /volume_storage";
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, errorMessage);
    }

    {
        const TString path = "/volume_storage/test";
        auto extractionResult = pathHolder->ExtractVolumeIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract VolumeId from path:  /volume_storage/test";
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, errorMessage);
    }
}

Y_UNIT_TEST(TestStaticResourceHashFromPathExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder(
        "/root"
        , {
            {"virtual_disk__0", ""}
            , {"virtual_disk_1", "///other_place"}
        }
        , {
            {"", "/resource_storage"}
            , {"///other_place", "/other_resource_storage"}
        }
        , "/volume_storage"
        , "persistent_storage"
        , ""
        , ""
        , ""
        , ""
    );

    {
        const TString hash = "my_resource_hash";
        const TString path = pathHolder->GetStaticResourceDownloadDirectoryFromHash(hash, "");
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromPath(path, "");
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), hash);
    }

    {
        const TString path = "path";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromPath(path, "");
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract id or hash from path: path";
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, errorMessage);
    }

    {
        const TString hash = "my_resource_hash";
        const TString path = "/resource_storage/static/my_resource_hash";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromPath(path, "");
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), hash);
    }

    {
        const TString hash = "my_resource_hash";
        const TString path = "/other_resource_storage/static/my_resource_hash";

        {
            auto extractionResult = pathHolder->ExtractStaticResourceHashFromPath(path, "");
            UNIT_ASSERT_EQUAL((bool)extractionResult, false);
            const TString errorMessage = TStringBuilder() << "Unable to extract id or hash from path: " << path;
            UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, errorMessage);
        }
        {
            auto extractionResult = pathHolder->ExtractStaticResourceHashFromPath(path, "///other_place");
            UNIT_ASSERT_EQUAL((bool)extractionResult, true);
            UNIT_ASSERT_EQUAL(extractionResult.Success(), hash);
        }
    }

    {
        const TString hash = "my_resource_hash";
        const TString path = "/resource_storage/static/my_resource_hash/subdir/";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromPath(path, "");
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), hash);
    }
}

Y_UNIT_TEST(TestBoxIdFromPathExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", "/resource_storage"}}, "/volume_storage", "persistent_storage", "", "", "", "");
    {
        const TString id = "my_box";
        const TString path = pathHolder->GetBoxRootfsPath(id);
        auto extractionResult = pathHolder->ExtractBoxIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString id = "my_box";
        const TString path = pathHolder->GetBoxRootfsPath(id) + "/link";
        auto extractionResult = pathHolder->ExtractBoxIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString id = "my_box";
        const TString path = pathHolder->GetBoxRootfsPath(id) + "/recursive/link";
        auto extractionResult = pathHolder->ExtractBoxIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString path = "path";
        auto extractionResult = pathHolder->ExtractBoxIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract id or hash from path: path";
        UNIT_ASSERT_EQUAL(extractionResult.Error().Message, errorMessage);
    }

    {
        const TString path = "/volume_storage/rootfs_";
        auto extractionResult = pathHolder->ExtractBoxIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract id or hash from path: /volume_storage/rootfs_";
        UNIT_ASSERT_EQUAL(extractionResult.Error().Message, errorMessage);
    }

    {
        const TString path = "/volume_storage/test";
        auto extractionResult = pathHolder->ExtractBoxIdFromPath(path);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        const TString errorMessage = "Unable to extract id or hash from path: /volume_storage/test";
        UNIT_ASSERT_EQUAL(extractionResult.Error().Message, errorMessage);
    }
}

Y_UNIT_TEST(ContainerGenerationShouldWorkAsExpected) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "", "", "");

    // ResourceGang
    UNIT_ASSERT_EQUAL_C(pathHolder->GetResourceGangMetaContainer(), TPortoContainerName("resource_gang_meta"), pathHolder->GetResourceGangMetaContainer());
    UNIT_ASSERT_EQUAL_C(pathHolder->GetLayerContainerWithNameFromHash("hash", "download"), TPortoContainerName::NoEscape("resource_gang_meta/layer_hash_download"), pathHolder->GetLayerContainerWithNameFromHash("hash", "download"));
    UNIT_ASSERT_EQUAL_C(pathHolder->GetStaticResourceContainerWithNameFromHash("hash", "download"), TPortoContainerName::NoEscape("resource_gang_meta/static_hash_download"), pathHolder->GetStaticResourceContainerWithNameFromHash("hash", "download"));

    // Box
    UNIT_ASSERT_EQUAL(pathHolder->GetBoxContainer("id"), TPortoContainerName("box_id"));
    UNIT_ASSERT_EQUAL(pathHolder->GetBoxInitContainer("id", 5), TPortoContainerName({"box_id"}, "init5"));

    // Workload
    UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadContainerWithName("boxId", "workloadId", "start"), TPortoContainerName({"box_boxId"}, "workload_workloadId_start"));
    UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadInitContainer("boxId", "workloadId", 1), TPortoContainerName({"box_boxId"}, "workload_workloadId_init1"));
}

Y_UNIT_TEST(TestWorkloadContainerLookupMask) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "", "", "");
    TPortoContainerName canondata = TPortoContainerName::NoEscape("box_MyBox/workload_MyWorkload***");
    UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadContainerLookupMask("MyBox", "MyWorkload"), canondata);
}

Y_UNIT_TEST(TestDownloadContainerLookupMask) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "", "", "");
    TPortoContainerName canondata = TPortoContainerName::NoEscape("resource_gang_meta/*_download");
    UNIT_ASSERT_EQUAL(pathHolder->GetDownloadContainerLookupMask(), canondata);
}

Y_UNIT_TEST(TestVerifyContainerLookupMask) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "", "", "");
    TPortoContainerName canondata = TPortoContainerName::NoEscape("resource_gang_meta/*_verify");
    UNIT_ASSERT_EQUAL(pathHolder->GetVerifyContainerLookupMask(), canondata);
}

Y_UNIT_TEST(TestLayerFromContainerExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "MyPrefix_", "", "");

    {
        const TString hash = "my_layer_hash";
        const TPortoContainerName container = pathHolder->GetLayerContainerWithNameFromHash(hash, "name");
        auto extractionResult = pathHolder->ExtractLayerHashFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL_C(extractionResult.Success(), hash, extractionResult.Success());
    }

    {
        const TString hash = "my_layer_hash";
        UNIT_ASSERT_EXCEPTION_CONTAINS(pathHolder->GetLayerContainerWithNameFromHash(hash, "random_bad_name"), yexception, "contains separator");
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/layers_download";
        auto extractionResult = pathHolder->ExtractLayerHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container));
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/layers";
        auto extractionResult = pathHolder->ExtractLayerHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container));
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/Resources_";
        auto extractionResult = pathHolder->ExtractLayerHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container));
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/Resources_layerHash";
        auto extractionResult = pathHolder->ExtractLayerHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container));
    }

    {
        const TString container = "random_layerHash_download";
        auto extractionResult = pathHolder->ExtractLayerHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container));
    }
}

Y_UNIT_TEST(TestStaticResourceFromContainerExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "MyPrefix_", "", "");

    {
        const TString hash = "my_resource_hash";
        TPortoContainerName container = pathHolder->GetStaticResourceContainerWithNameFromHash(hash, "name");
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(container);
        UNIT_ASSERT_EQUAL_C((bool)extractionResult, true, extractionResult.Error().Message);
        UNIT_ASSERT_EQUAL_C(extractionResult.Success(), hash, extractionResult.Success());
    }

    {
        const TString hash = "my_resource_hash";
        UNIT_ASSERT_EXCEPTION_CONTAINS(pathHolder->GetStaticResourceContainerWithNameFromHash(hash, "random_bad_name"), yexception, "contains separator");
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/static_download";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + container);
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/Resources";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + container);
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/Resources_";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + container);
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/Resources_resourceHash";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + container);
    }

    {
        const TString container = "MyPrefix_resource_gang_meta/random_resourceHash_download";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + container);
    }

    {
        const TString container = "random_resourceHash_download";
        auto extractionResult = pathHolder->ExtractStaticResourceHashFromContainer(TPortoContainerName::NoEscape(container));
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + container);
    }
}

Y_UNIT_TEST(TestWorkloadSuffixFromContainerExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "", "", "");

    {
        const TPortoContainerName containerName = pathHolder->GetWorkloadContainerWithName("boxId", "workloadId", "start");
        auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
        UNIT_ASSERT(extractionResult);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), "start");
    }

    {
        const TPortoContainerName containerName = pathHolder->GetWorkloadContainerWithName("boxId", "workloadId", "");
        auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
        UNIT_ASSERT(!extractionResult);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract suffix from container '" + TString(containerName) + "'");
    }

    {
        const TPortoContainerName containerName = pathHolder->GetWorkloadInitContainer("boxId", "workloadId", 2);
        auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
        UNIT_ASSERT(extractionResult);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), "init2");
    }
}

Y_UNIT_TEST(TestBoxFromContainerExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "MyPrefix_", "", "");

    {
        const TString id = "my_box";
        const TPortoContainerName container = pathHolder->GetBoxContainer(id);
        auto extractionResult = pathHolder->ExtractBoxIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString id = "my_box";
        const TPortoContainerName container = pathHolder->GetBoxInitContainer(id, 42);
        auto extractionResult = pathHolder->ExtractBoxIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString id = "my_box";
        const TPortoContainerName container = pathHolder->GetWorkloadContainerWithName(id, "workloadId", "start");
        auto extractionResult = pathHolder->ExtractBoxIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TPortoContainerName container("MyPrefix_box_");
        auto extractionResult = pathHolder->ExtractBoxIdFromContainer(container);
        UNIT_ASSERT_EQUAL_C((bool)extractionResult, false, extractionResult.Success());
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract BoxId from container '" + TString(container) + "'");
    }

    {
        const TPortoContainerName container{{"MyPrefix_box_"}, ""};
        auto extractionResult = pathHolder->ExtractBoxIdFromContainer(container);
        UNIT_ASSERT_EQUAL_C((bool)extractionResult, false, extractionResult.Success());
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract BoxId from container '" + TString(container) + "'");
    }

    {
        const TPortoContainerName container{{"MyPrefix_random"}, "boxId"};
        auto extractionResult = pathHolder->ExtractBoxIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract BoxId from container '" + TString(container) + "'");
    }
}

Y_UNIT_TEST(TestBoxInitFromContainerExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "MyPrefix_", "", "");

    {
        const TString id = "my_box";
        const TPortoContainerName container = pathHolder->GetBoxInitContainer(id, 42);
        auto extractionResult = pathHolder->ExtractBoxInitIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), "42");
    }

    {
        const TString id = "my_box";
        const TPortoContainerName container = {pathHolder->GetBoxInitContainer(id, 42), "subcontainer"};
        auto extractionResult = pathHolder->ExtractBoxInitIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract box simple subcontainer from container '" + TString(container) + "'");
    }

    {
        const TPortoContainerName container{"MyPrefix_box_boxId"};
        auto extractionResult = pathHolder->ExtractBoxInitIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract box subcontainer from container '" + TString(container) + "'");
    }

    {
        const TPortoContainerName container{{"MyPrefix_box_boxId"}, "trash1"};
        auto extractionResult = pathHolder->ExtractBoxInitIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract BoxInitId from container '" + TString(container) + "'");
    }
}

Y_UNIT_TEST(TestWorkloadFromContainerExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "MyPrefix_", "", "");
    {
        const TString id = "my_workload";
        const TPortoContainerName container = pathHolder->GetWorkloadContainerWithName("boxId", id, "start");
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL_C(extractionResult.Success(), id, extractionResult.Success());
    }
    
    {
        const TString id = "my_workload";
        const TPortoContainerName container = pathHolder->GetWorkloadContainerWithName("boxId", id, "");
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }
    
    {
        const TString id = "my_workload";
        const TPortoContainerName container = {pathHolder->GetWorkloadContainerWithName("boxId", id, ""), "subcontainer"};
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract box simple subcontainer from container '" + TString(container) + "'");
    }
    
    {
        const TPortoContainerName container("MyPrefix_box_workload_workloadId_start");
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract box subcontainer from container '" + TString(container) + "'");
    }
    
    {
        const TPortoContainerName container("MyPrefix_box_");
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract BoxId from container '" + TString(container) + "'");
    }
    
    {
        const TPortoContainerName container{{"MyPrefix_box_"}, ""};
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract BoxId from container '" + TString(container) + "'");
    }
    
    {
        const TPortoContainerName container{{"MyPrefix_box_boxid"}, "trash"};
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container.GetChild()) + "'");
    }
    
    {
        const TPortoContainerName container{{"MyPrefix_box_boxid"}, "WorkloadworkloadId_start"};
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container.GetChild()) + "'");
    }
    
    {
        const TPortoContainerName container{{"MyPrefix_box_boxid"}, "workload_workloadIdstart"};
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container.GetChild()) + "'");
    }
    
    {
        const TPortoContainerName container{{"MyPrefix_box_boxid"}, "Trash_workloadId_start"};
        auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract id or hash from container '" + TString(container.GetChild()) + "'");
    }
}

Y_UNIT_TEST(TestIdFromStorageExtraction) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "storages_", "", "", "", "");

    {
        const TString id = "my_volume";
        const TString storage = pathHolder->GetVolumePersistentStorage(id);
        auto extractionResult = pathHolder->ExtractVolumeIdFromStorage(storage);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString id = "my_box";
        const TString storage = pathHolder->GetBoxRootfsPersistentStorage(id);
        auto extractionResult = pathHolder->ExtractBoxIdFromStorage(storage);
        UNIT_ASSERT_EQUAL((bool)extractionResult, true);
        UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
    }

    {
        const TString storage("storages_volume_");
        auto extractionResult = pathHolder->ExtractVolumeIdFromStorage(storage);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, "Unable to extract VolumeId from storage:  storages_volume_");
    }

    {
        const TString storage("storages_rootfs_");
        auto extractionResult = pathHolder->ExtractBoxIdFromStorage(storage);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, "Unable to extract BoxId from storage:  storages_rootfs_");
    }

    {
        const TString storage("wrong_storage");
        auto extractionResult = pathHolder->ExtractVolumeIdFromStorage(storage);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, "Unable to extract VolumeId from storage:  wrong_storage");
    }
    {
        const TString storage("wrong_storage");
        auto extractionResult = pathHolder->ExtractBoxIdFromStorage(storage);
        UNIT_ASSERT_EQUAL((bool)extractionResult, false);
        UNIT_ASSERT_EQUAL(((TIdExtractionError)extractionResult).Message, "Unable to extract BoxId from storage:  wrong_storage");
    }
}

Y_UNIT_TEST(ContainerGenerationWithPrefixShouldWorkAsExpected) {
    TPathHolderPtr pathHolder = new TPathHolder("", {{"", ""}}, {{"", ""}}, "", "", "", "MyPrefix_", "", "");

    // ResourceGang
    UNIT_ASSERT_EQUAL_C(pathHolder->GetResourceGangMetaContainer(), TPortoContainerName("MyPrefix_resource_gang_meta"), pathHolder->GetResourceGangMetaContainer());
    UNIT_ASSERT_EQUAL_C(pathHolder->GetLayerContainerWithNameFromHash("hash", "download"), TPortoContainerName::NoEscape("MyPrefix_resource_gang_meta/layer_hash_download"), pathHolder->GetLayerContainerWithNameFromHash("hash", "download"));
    UNIT_ASSERT_EQUAL_C(pathHolder->GetStaticResourceContainerWithNameFromHash("hash", "download"), TPortoContainerName::NoEscape("MyPrefix_resource_gang_meta/static_hash_download"), pathHolder->GetStaticResourceContainerWithNameFromHash("hash", "download"));

    // Box
    UNIT_ASSERT_EQUAL(pathHolder->GetBoxContainer("id"), TPortoContainerName("MyPrefix_box_id"));
    UNIT_ASSERT_EQUAL(pathHolder->GetBoxInitContainer("id", 5), TPortoContainerName({"MyPrefix_box_id"}, "init5"));

    // Workload
    UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadContainerWithName("boxId", "workloadId", "start"), TPortoContainerName({"MyPrefix_box_boxId"}, "workload_workloadId_start"));
    UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadInitContainer("boxId", "workloadId", 1), TPortoContainerName({"MyPrefix_box_boxId"}, "workload_workloadId_init1"));
}

Y_UNIT_TEST(TestHasPlace) {
    TPathHolderPtr pathHolder = new TPathHolder(
        "/root"
        , {
            {"virtual_disk__0", ""}
            , {"virtual_disk_1", "///other_place"}
        }
        , {
            {"", "/resource_storage"}
            , {"///other_place", "/other_resource_storage"}
        }
        , "/volume_storage"
        , "persistent_storage"
        , ""
        , ""
        , ""
        , ""
    );

    UNIT_ASSERT(pathHolder->HasPlace(""));
    UNIT_ASSERT(pathHolder->HasPlace("///other_place"));
    UNIT_ASSERT(!pathHolder->HasPlace("///bad_place"));
    UNIT_ASSERT(!pathHolder->HasPlace("/resource_storage"));
}

Y_UNIT_TEST(TestVirtualDisk) {
    TPathHolderPtr pathHolder = new TPathHolder(
        "/root"
        , {
            {"virtual_disk_0", ""}
            , {"virtual_disk_1", "///other_place"}
        }
        , {
            {"", "/resource_storage"}
            , {"///other_place", "/other_resource_storage"}
        }
        , "/volume_storage"
        , "persistent_storage"
        , ""
        , ""
        , ""
        , ""
    );

    UNIT_ASSERT(pathHolder->HasVirtualDisk("virtual_disk_0"));
    UNIT_ASSERT(pathHolder->HasVirtualDisk("virtual_disk_1"));
    UNIT_ASSERT(!pathHolder->HasVirtualDisk("bad_disk"));
    UNIT_ASSERT(!pathHolder->HasVirtualDisk("/resource_storage"));

    UNIT_ASSERT_EQUAL_C(
        pathHolder->GetPlaceFromVirtualDisk("virtual_disk_0")
        , ""
        , pathHolder->GetPlaceFromVirtualDisk("virtual_disk_0")
    );
    UNIT_ASSERT_EQUAL_C(
        pathHolder->GetDom0PlaceFromVirtualDisk("virtual_disk_0")
        , ""
        , pathHolder->GetDom0PlaceFromVirtualDisk("virtual_disk_0")
    );
    UNIT_ASSERT_EQUAL_C(
        pathHolder->GetPlaceFromVirtualDisk("virtual_disk_1")
        , "///other_place"
        , pathHolder->GetPlaceFromVirtualDisk("virtual_disk_1")
    );
    UNIT_ASSERT_EQUAL_C(
        pathHolder->GetDom0PlaceFromVirtualDisk("virtual_disk_1")
        , "/root/other_place"
        , pathHolder->GetPlaceFromVirtualDisk("virtual_disk_1")
    );
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        pathHolder->GetPlaceFromVirtualDisk("bad_disk")
        , yexception
        , "unknown virtual disk"
    );
}

Y_UNIT_TEST(TestVirtualDiskValidation) {
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        TPathHolder("/root", {}, {{"", ""}}, "", "", "", "", "", "")
        , yexception
        , "Different number"
    );
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        TPathHolder("/root", {{"virtual_disk_0", "///bad_place"}}, {{"", ""}}, "", "", "", "", "", "")
        , yexception
        , "non-existent place"
    );
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        TPathHolder("/root", {{"virtual_disk_0", "no_slash_place"}}, {{"no_slash_place", "/no_slash_resource"}}, "", "", "", "", "", "")
        , yexception
        , "Place specified without '///' at the beginning"
    );
}

Y_UNIT_TEST(TestWorkloadMethodsWithBoxMode) {
     TPathHolderPtr pathHolder = new TPathHolder(
         ""
         , {}
         , {}
         , ""
         , ""
         , ""
         , ""
         , "porto_logs_storage"
         , ""
         , true /* isBoxAgentMode */
     );

     UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFileName("my_workload", "stdout"), "my_workload_stdout.portolog");
     UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadLogsFilePathInBox("my_workload", "stdout"), "/porto_logs_storage/my_workload_stdout.portolog");

     UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadContainerWithName("", "workloadId", "start"), TPortoContainerName("workload_workloadId_start"));
     UNIT_ASSERT_EXCEPTION_CONTAINS(
         pathHolder->GetWorkloadContainerWithName("boxId", "workloadId", "start")
         , yexception
         , "Workload must not have box when pod agent runs in box agent mode, but it has: 'boxId'"
     );

     UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadInitContainer("", "workloadId", 1), TPortoContainerName("workload_workloadId_init1"));
     UNIT_ASSERT_EXCEPTION_CONTAINS(
         pathHolder->GetWorkloadInitContainer("boxId", "workloadId", 1)
         , yexception
         , "Workload must not have box when pod agent runs in box agent mode, but it has: 'boxId'"
     );

     UNIT_ASSERT_EQUAL(pathHolder->GetWorkloadContainerLookupMask("", "MyWorkload"), TPortoContainerName::NoEscape("workload_MyWorkload***"));
     UNIT_ASSERT_EXCEPTION_CONTAINS(
         pathHolder->GetWorkloadContainerLookupMask("boxId", "MyWorkload")
         , yexception
         , "Workload must not have box when pod agent runs in box agent mode, but it has: 'boxId'"
     );

     // Suffix extraction
     {
         const TPortoContainerName containerName = TPortoContainerName({"box_boxId"}, "workload_workloadId_start");
         auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
         UNIT_ASSERT(!extractionResult);
         UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Workload container must be simple when pod agent runs in box agent mode, but it is '" + TString(containerName) + "'");
     }

     {
         const TPortoContainerName containerName = pathHolder->GetWorkloadContainerWithName("", "workloadId", "start");
         auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
         UNIT_ASSERT(extractionResult);
         UNIT_ASSERT_EQUAL(extractionResult.Success(), "start");
     }

     {
         const TPortoContainerName containerName = pathHolder->GetWorkloadContainerWithName("", "workloadId", "");
         auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
         UNIT_ASSERT(!extractionResult);
         UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Unable to extract suffix from container '" + TString(containerName) + "'");
     }

     {
         const TPortoContainerName containerName = pathHolder->GetWorkloadInitContainer("", "workloadId", 2);
         auto extractionResult = pathHolder->ExtractWorkloadSuffixFromContainer(containerName);
         UNIT_ASSERT(extractionResult);
         UNIT_ASSERT_EQUAL(extractionResult.Success(), "init2");
     }

     // Id extraction
     const TString id = "my_workload";
     {
         const TPortoContainerName container = pathHolder->GetWorkloadContainerWithName("", id, "start");
         auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
         UNIT_ASSERT_EQUAL((bool)extractionResult, true);
         UNIT_ASSERT_EQUAL_C(extractionResult.Success(), id, extractionResult.Success());
     }

     {
         const TPortoContainerName container = pathHolder->GetWorkloadContainerWithName("", id, "");
         auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
         UNIT_ASSERT_EQUAL((bool)extractionResult, true);
         UNIT_ASSERT_EQUAL(extractionResult.Success(), id);
     }

     {
         const TPortoContainerName container = {pathHolder->GetWorkloadContainerWithName("", id, ""), "subcontainer"};
         auto extractionResult = pathHolder->ExtractWorkloadIdFromContainer(container);
         UNIT_ASSERT_EQUAL((bool)extractionResult, false);
         UNIT_ASSERT_STRING_CONTAINS(extractionResult.Error().Message, "Workload container must be simple when pod agent runs in box agent mode, but it is '" + TString(container) + "'");
     }
}

}

} // namespace NInfra::NPodAgent::NTestPathHolder
