#include "cache_file.h"

#include <infra/pod_agent/libs/pod_agent/cache_file/ut/proto/test_message.pb.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/random/mersenne.h>

namespace NInfra::NPodAgent::NCacheFileTest {

const TString CWD = NFs::CurrentWorkingDirectory();

// Stable unique file path for every test
TString GetFilePath() {
    static TMersenne<ui64> rnd(12345);

    // Add folder + path to cwd to check folder creation
    const TString folder = ToString(rnd.GenRand());
    const TString fileName = ToString(rnd.GenRand());
    const TString dir = CWD + "/" + folder;

    if (NFs::Exists(dir)) {
        NFs::RemoveRecursive(dir);
    }

    return dir + "/" + fileName;
}

TestMessage GetTestProto(bool checkFloat) {
    const size_t cntIter = 10;
    TestMessage message;

    message.SetField("Field");
    for (size_t i = 0; i < cntIter; ++i) {
        message.MutableMap()->insert({ToString(i), ToString(i + 1)});
        message.MutableArray()->Add(ToString(i));
    }

    if (checkFloat) {
        message.SetFloat(1.1234567891234);
    }

    return message;
}

Y_UNIT_TEST_SUITE(CacheFileSuite) {

Y_UNIT_TEST(SafeLoadJson) {
    const TString filePath = GetFilePath();
    const TestMessage message = GetTestProto(false);

    NCacheFile::SaveToFileJson<TestMessage>(filePath, message);
    TestMessage loadedMessage = NCacheFile::LoadFromFileJson<TestMessage>(filePath);

    UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(message, loadedMessage));
}

Y_UNIT_TEST(SafeLoadBin) {
    const TString filePath = GetFilePath();
    const TestMessage message = GetTestProto(true);

    NCacheFile::SaveToFileBin<TestMessage>(filePath, message);
    TestMessage loadedMessage = NCacheFile::LoadFromFileBin<TestMessage>(filePath);

    UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(message, loadedMessage));
}

Y_UNIT_TEST(LoadJsonError) {
    const TString filePath = GetFilePath();

    {
        TString dir = GetDirName(filePath);
        if (!NFs::Exists(dir)) {
            NFs::MakeDirectoryRecursive(dir);
        }
        TUnbufferedFileOutput fileOutput(filePath);
        fileOutput << "trash";
    }

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NCacheFile::LoadFromFileJson<TestMessage>(filePath)
        , yexception
        , "TJsonException"
    );
}

Y_UNIT_TEST(LoadBinError) {
    const TString filePath = GetFilePath();

    {
        TString dir = GetDirName(filePath);
        if (!NFs::Exists(dir)) {
            NFs::MakeDirectoryRecursive(dir);
        }
        TUnbufferedFileOutput fileOutput(filePath);
        fileOutput << "trash";
    }

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NCacheFile::LoadFromFileBin<TestMessage>(filePath)
        , yexception
        , "Fail to parse proto from file"
    );
}

}

} // namespace NInfra::NPodAgent::NCacheFileTest
