#include "core_service.h"

#include <infra/pod_agent/libs/pod_agent/cache_file/cache_file.h>
#include <infra/pod_agent/libs/pod_agent/garbage_collector_job/garbage_collector_job.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/logs_transmitter_job/logs_transmitter_job_factory.h>
#include <infra/pod_agent/libs/pod_agent/trees_update_job/trees_update_job.h>

#include <infra/pod_agent/libs/ip_client/simple_client.h>
#include <infra/pod_agent/libs/network_client/simple_client.h>
#include <infra/pod_agent/libs/porto_client/simple_client.h>
#include <infra/pod_agent/libs/system_logs_sender/common_system_logs_session.h>
#include <infra/pod_agent/libs/system_logs_sender/one_system_logs_session_collection.h>
#include <infra/pod_agent/libs/system_logs_sender/sidecars_system_logs_filter_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/statistics/system_logs_statistics_printer_impl.h>

#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/system/fs.h>

namespace NInfra::NPodAgent {

namespace {

void PatchUnreleasedPodAgentStatusFields(API::TPodAgentStatus& status) {
    // TODO(DEPLOY-3561) remove after release
    status.clear_target_state();
}

} // namespace

TCoreService::TCoreService(
    TLogger& logger
    , TLogger& treeTraceLogger
    , TLogger& logsTransmitterJobWorkerEventsLogger
    , const TConfig& config
    , const TString& dom0PodRoot
    , const TMap<TString, TString>& virtualDisksToPlace
    , const TMap<TString, TString>& placesToDownloadVolumePath
)
    : Logger_(logger)
    , TreeTraceLogger_(treeTraceLogger)
    , LogsTransmitterJobWorkerEventsLogger_(logsTransmitterJobWorkerEventsLogger)
    , Config_(config)
    , Dom0PodRoot_(dom0PodRoot)
    , VirtualDisksToPlace_(virtualDisksToPlace)
    , PlacesToDownloadVolumePath_(placesToDownloadVolumePath)
{
}

void TCoreService::Start() {
    TGuard<TMutex> gMutex(Mutex_);

    auto logFrame = Logger_.SpawnFrame();

    TreesWorkersMtpQueue_ = new TThreadPool();
    TreesWorkersMtpQueue_->Start(Config_.GetBehaviorTicker().GetTreesWorkersPoolSize());

    TPortoConnectionPoolPtr pool = new TPortoConnectionPool(Config_.GetPorto().GetPoolSize());
    TPortoClientPtr porto = new TSimplePortoClient(logFrame, pool, PORTO_CLIENT_DEFAULT_TIMEOUT);
    TIpClientPtr ipClient = new TSimpleIpClient();
    TNetworkClientPtr networkClient = new TSimpleNetworkClient(TreesWorkersMtpQueue_);

    BehaviorTicker_ = new TMtpPeriodTicker(Config_.GetBehaviorTicker(), Logger_.SpawnFrame(), TreeTraceLogger_.SpawnFrame());
    BehaviorTicker_->Start();

    TStatusRepositoryPtr statusRepository = new TStatusRepository(
        new TUpdateHolder()
        , new TBoxStatusRepository()
        , new TLayerStatusRepository()
        , new TStaticResourceStatusRepository()
        , new TVolumeStatusRepository()
        , new TWorkloadStatusRepository()
    );

    TSystemLogsStatisticsPtr systemLogsStatistics = new TSystemLogsStatistics();
    TSystemLogsSessionPtr commonSystemLogsSession = new TCommonSystemLogsSession();
    TSystemLogsSessionCollectionPtr boxSystemLogsSessionCollection = new TOneSystemLogsSessionCollection(commonSystemLogsSession);
    TSystemLogsSessionCollectionPtr workloadSystemLogsSessionCollection = new TOneSystemLogsSessionCollection(commonSystemLogsSession);
    TSidecarsSystemLogsFilterPtr sidecarsSystemLogsFilter = new TSidecarsSystemLogsFilterImpl();

    const bool isBoxAgentMode = Config_.GetPodAgentMode() == EPodAgentMode::BOX_MODE;

    TStatusNTickerHolderPtr statusNTickerHolder = new TStatusNTickerHolder(
        BehaviorTicker_
        , statusRepository
        , new TWorkloadStatusRepositoryInternal()
        , new TSystemLogsSender(workloadSystemLogsSessionCollection, TreesWorkersMtpQueue_, systemLogsStatistics, sidecarsSystemLogsFilter)
        , new TSystemLogsSender(boxSystemLogsSessionCollection, TreesWorkersMtpQueue_, systemLogsStatistics, sidecarsSystemLogsFilter)
        , isBoxAgentMode
    );

    TPathHolderPtr pathHolder = new TPathHolder(
        Dom0PodRoot_
        , VirtualDisksToPlace_
        , PlacesToDownloadVolumePath_
        , Config_.GetVolumes().GetStorage()
        , Config_.GetPorto().GetPersistentStoragePrefix()
        , Config_.GetPorto().GetLayerPrefix()
        , Config_.GetPorto().GetContainersPrefix()
        , Config_.GetLogsTransmitter().GetPortoLogFilesDir()
        , Config_.GetRbindVolumes().GetStorage()
        , isBoxAgentMode
    );

    TPosixWorkerPtr posixWorker = new TPosixWorker(TreesWorkersMtpQueue_);

    TString hostname = networkClient->GetLocalHostName().Success();

    TreesGenerator_ = MakeHolder<TTreesGenerator>(
        Logger_
        , Config_.GetCache()
        , Config_.GetLogsTransmitter()
        , Config_.GetResources()
        , hostname
        , Config_.GetYtBind().GetYtPath()
        , Config_.GetBaseSearchBind().GetBaseSearchPath()
        , Config_.GetPublicVolume().GetVolumePath()
        , Config_.GetPublicVolume().GetMountPath()
        , Config_.GetPublicVolume().GetPodAgentBinaryFileName()
        , Config_.GetIp().GetDevice()
        , Config_.GetIsPossibleToTransmitSystemLogs()
        , new TAsyncIpClient(ipClient, TreesWorkersMtpQueue_)
        , new TAsyncPortoClient(porto, TreesWorkersMtpQueue_)
        , networkClient
        , pathHolder
        , posixWorker
        , statusNTickerHolder
        , new TTemplateBTStorage(Config_.GetTemplateBTStorage())
        , isBoxAgentMode
    );

    PeriodJobWorker_ = MakeHolder<TPeriodJobWorker>(
        TDuration::MilliSeconds(Config_.GetPeriodJobWorker().GetMainLoopPeriodMs())
        , Logger_.SpawnFrame()
    );

    GarbageCollectorJob_ = new TGarbageCollectorJob(
        TDuration::MilliSeconds(Config_.GetGarbageCollector().GetPeriodMs())
        , Logger_.SpawnFrame()
        , statusNTickerHolder
        , porto
        , posixWorker
        , pathHolder
        , ipClient
        , networkClient
        , PlacesToDownloadVolumePath_
        , Config_.GetVolumes().GetStorage()
        , Config_.GetPorto().GetLayerPrefix()
        , Config_.GetPorto().GetPersistentStoragePrefix()
        , Config_.GetCache().GetStorage()
        , Config_.GetResources().GetDownloadStoragePrefix()
        , Config_.GetPorto().GetContainersPrefix()
        , Config_.GetIp().GetDevice()
    );
    // Make sure that garbage collector unactive
    // It will be activated only after the first successful spec update
    GarbageCollectorJob_->SetActive(false);

    PeriodJobWorker_->AddJob(GarbageCollectorJob_);

    TSystemLogsStatisticsPrinterPtr systemLogsStatisticsPrinter = new TSystemLogsStatisticsPrinter(
        systemLogsStatistics
        , boxSystemLogsSessionCollection
        , workloadSystemLogsSessionCollection
        , Logger_.SpawnFrame()
        , TSystemLogsStatisticsPrinter::SYSTEM_LOGS_PRINTER_DEFAULT_PRINT_PERIOD_SEC
    );
    PeriodJobWorker_->AddJob(
        new TTreesUpdateJob(
            TDuration::MilliSeconds(Config_.GetTreesUpdate().GetPeriodMs())
            , Logger_.SpawnFrame()
            , LogsTransmitterJobWorkerEventsLogger_.SpawnFrame()
            , statusNTickerHolder
            , systemLogsStatisticsPrinter
            , UpdateSpecMutex_
        )
    );

    TryUpdateSpecFromSavePath(logFrame);

    // Note: logs transmitter job should be created after saved spec update,
    // because while creation logs transmitter job should know about current workload containers for meta deserialization
    PeriodJobWorker_->AddJob(
        TLogsTransmitterJobFactory(
            Config_
            , Logger_.SpawnFrame()
            , LogsTransmitterJobWorkerEventsLogger_.SpawnFrame()
            , pathHolder
            , hostname
            , statusRepository
            , statusNTickerHolder
            , porto
            , isBoxAgentMode
        ).Create()
    );

    PeriodJobWorker_->Start();

    logFrame->LogEvent(
        ELogPriority::TLOG_INFO
        , NLogEvent::TStartDaemon(
            GetProgramSvnRevision()
            , GetBranch()
        )
    );
}

void TCoreService::Wait() {
    auto logFrame = Logger_.SpawnFrame();

    PeriodJobWorker_->Wait();

    BehaviorTicker_->Stop();

    TreesWorkersMtpQueue_->Stop();

    logFrame->LogEvent(
        ELogPriority::TLOG_INFO
        , NLogEvent::TStopDaemon()
    );
}

void TCoreService::Shutdown() {
    TGuard<TMutex> gMutex(Mutex_);

    PeriodJobWorker_->Shutdown();
}

API::TPodAgentStatus TCoreService::UpdatePodAgentRequest(const API::TPodAgentRequest& spec) {
    TGuard<TMutex> gUpdateSpecMutex(UpdateSpecMutex_);
    TGuard<TMutex> gMutex(Mutex_);
    auto logFrame = Logger_.SpawnFrame();

    API::TPodAgentStatus result;
    try {
        result = TreesGenerator_->UpdatePodAgentRequest(spec, logFrame);
    } catch (yexception& e) {
        logFrame->LogEvent(ELogPriority::TLOG_CRIT, NLogEvent::TUpdatePodAgentRequestError(CurrentExceptionMessage()));

        throw e; // rethrow exception
    }

    // Save spec only after successful UpdatePodAgentRequest
    CurrentRequest_ = spec;
    SaveSpecToSavePath(spec, logFrame);

    // Activate garbage collector after any successful spec update
    GarbageCollectorJob_->SetActive(true);

    PatchUnreleasedPodAgentStatusFields(result);
    return result;
}

API::TPodAgentStatus TCoreService::GetPodAgentStatus() const {
    API::TPodAgentStatus status = TreesGenerator_->GetStatus(false);
    PatchUnreleasedPodAgentStatusFields(status);
    return status;
}

TString TCoreService::PodStatusJson() const {
    TGuard<TMutex> gMutex(Mutex_);

    auto status = TreesGenerator_->GetStatus(false);
    NJson::TJsonValue result;

    {
        NJson::TJsonValue& boxes = result["boxes"];
        for (const auto& box : status.boxes()) {
            auto element = NJson::TJsonValue(NJson::JSON_MAP);

            element["id"] = box.id();
            element["revision"] = box.revision();

            boxes.AppendValue(element);
        }
    }

    {
        NJson::TJsonValue& workloads = result["workloads"];
        for (const auto& workload : status.workloads()) {
            auto element = NJson::TJsonValue(NJson::JSON_MAP);

            element["id"] = workload.id();
            element["revision"] = workload.revision();

            workloads.AppendValue(element);
        }
    }

    return ToString(result);
}

TString TCoreService::PodAttributesJson() const {
    TGuard<TMutex> gMutex(Mutex_);
    NJson::TJsonValue result;

    {
        NJson::TJsonValue& resourceRequirements = result["resource_requirements"];
        NJson::TJsonValue& cpu = resourceRequirements["cpu"];
        cpu["cpu_guarantee_millicores"] = CurrentRequest_.resource_requests().vcpu_guarantee() / CurrentRequest_.node_entry().cpu().cpu_to_vcpu_factor();
        cpu["cpu_limit_millicores"] = CurrentRequest_.resource_requests().vcpu_limit() / CurrentRequest_.node_entry().cpu().cpu_to_vcpu_factor();
        NJson::TJsonValue& memory = resourceRequirements["memory"];
        memory["memory_guarantee_bytes"] = CurrentRequest_.resource_requests().memory_guarantee();
        memory["memory_limit_bytes"] = CurrentRequest_.resource_requests().memory_limit();
    }

    {
        NJson::TJsonValue& metadata = result["metadata"];
        metadata["pod_id"] = CurrentRequest_.immutable_meta().pod_id();
        NJson::TJsonValue& labels = metadata["labels"] = NJson::TJsonValue(NJson::JSON_MAP);
        for (const auto& it : CurrentRequest_.pod_dynamic_attributes().labels().attributes()) {
            NJson2Yson::DeserializeYsonAsJsonValue(it.value(), &(labels[it.key()]), true);
        }
        NJson::TJsonValue& annotations = metadata["annotations"] = NJson::TJsonValue(NJson::JSON_MAP);
        for (const auto& it : CurrentRequest_.pod_dynamic_attributes().annotations().attributes()) {
            NJson2Yson::DeserializeYsonAsJsonValue(it.value(), &(annotations[it.key()]), true);
        }
    }

    {
        NJson::TJsonValue& boxResourceRequirements = result["box_resource_requirements"] = NJson::TJsonValue(NJson::JSON_MAP);
        for (const auto& it : CurrentRequest_.spec().boxes()) {
            NJson::TJsonValue& boxReq = boxResourceRequirements[it.id()];
            NJson::TJsonValue& cpu = boxReq["cpu"];
            cpu["cpu_guarantee_millicores"] = it.compute_resources().vcpu_guarantee() / CurrentRequest_.node_entry().cpu().cpu_to_vcpu_factor();
            cpu["cpu_limit_millicores"] = it.compute_resources().vcpu_limit() / CurrentRequest_.node_entry().cpu().cpu_to_vcpu_factor();
            NJson::TJsonValue& memory = boxReq["memory"];
            memory["memory_guarantee_bytes"] = it.compute_resources().memory_guarantee();
            memory["memory_limit_bytes"] = it.compute_resources().memory_limit();
        }
    }

    {
        NJson::TJsonValue& ip6AddressAllocations = result["ip6_address_allocations"] = NJson::TJsonValue(NJson::JSON_ARRAY);
        for (const auto& ip6AddressAllocation : CurrentRequest_.ip6_address_allocations()) {
            NJson::TJsonValue currentIp6AddressAllocation;

            currentIp6AddressAllocation["address"] = ip6AddressAllocation.address();
            currentIp6AddressAllocation["vlan_id"] = ip6AddressAllocation.vlan_id();

            {
                NJson::TJsonValue& labels = currentIp6AddressAllocation["labels"] = NJson::TJsonValue(NJson::JSON_MAP);
                for (const auto& label : ip6AddressAllocation.labels().attributes()) {
                    NJson2Yson::DeserializeYsonAsJsonValue(label.value(), &(labels[label.key()]), true);
                }
            }

            currentIp6AddressAllocation["persistent_fqdn"] = ip6AddressAllocation.persistent_fqdn();
            currentIp6AddressAllocation["transient_fqdn"] = ip6AddressAllocation.transient_fqdn();

            if (ip6AddressAllocation.has_internet_address()) {
                NJson::TJsonValue& internetAddress = currentIp6AddressAllocation["internet_address"] = NJson::TJsonValue(NJson::JSON_MAP);

                internetAddress["id"] = ip6AddressAllocation.internet_address().id();
                internetAddress["ip4_address"] = ip6AddressAllocation.internet_address().ip4_address();
            }

            {
                NJson::TJsonValue& virtualServices = currentIp6AddressAllocation["virtual_services"] = NJson::TJsonValue(NJson::JSON_ARRAY);
                for (const auto& virtualService : ip6AddressAllocation.virtual_services()) {
                    NJson::TJsonValue virtualServiceJson = NJson::TJsonValue(NJson::JSON_MAP);
                    NJson::TJsonValue& ip6Addresses = virtualServiceJson["ip6_addresses"] = NJson::TJsonValue(NJson::JSON_ARRAY);
                    for (const auto& ip6Address : virtualService.ip6_addresses()) {
                        ip6Addresses.AppendValue(ip6Address);
                    }

                    NJson::TJsonValue& ip4Addresses = virtualServiceJson["ip4_addresses"] = NJson::TJsonValue(NJson::JSON_ARRAY);
                    for (const auto& ip4Address : virtualService.ip4_addresses()) {
                        ip4Addresses.AppendValue(ip4Address);
                    }

                    virtualServices.AppendValue(virtualServiceJson);
                }
            }

            ip6AddressAllocations.AppendValue(currentIp6AddressAllocation);
        }
    }

    {
        NJson::TJsonValue& ip6SubnetAllocations = result["ip6_subnet_allocations"] = NJson::TJsonValue(NJson::JSON_ARRAY);
        for (const auto& ip6SubnetAllocation : CurrentRequest_.ip6_subnet_allocations()) {
            NJson::TJsonValue currentIp6SubnetAllocation;

            currentIp6SubnetAllocation["subnet"] = ip6SubnetAllocation.subnet();
            currentIp6SubnetAllocation["vlan_id"] = ip6SubnetAllocation.vlan_id();

            {
                NJson::TJsonValue& labels = currentIp6SubnetAllocation["labels"] = NJson::TJsonValue(NJson::JSON_MAP);
                for (const auto& label : ip6SubnetAllocation.labels().attributes()) {
                    NJson2Yson::DeserializeYsonAsJsonValue(label.value(), &(labels[label.key()]), true);
                }
            }

            ip6SubnetAllocations.AppendValue(currentIp6SubnetAllocation);
        }
    }


    {
        NJson::TJsonValue& nodeMeta = result["node_meta"] = NJson::TJsonValue(NJson::JSON_MAP);
        nodeMeta["fqdn"] = CurrentRequest_.immutable_meta().node_meta().fqdn();
        nodeMeta["cluster"] = CurrentRequest_.immutable_meta().node_meta().cluster();
        nodeMeta["dc"] = CurrentRequest_.immutable_meta().node_meta().dc();
    }

    {
        NJson::TJsonValue& diskVolumeAllocations = result["disk_volume_allocations"] = NJson::TJsonValue(NJson::JSON_ARRAY);
        for (const auto& diskVolumeAllocation: CurrentRequest_.disk_volume_allocations()) {
            NJson::TJsonValue currentDiskVolumeAllocation;

            currentDiskVolumeAllocation["id"] = diskVolumeAllocation.id();

            {
                NJson::TJsonValue& labels = currentDiskVolumeAllocation["labels"] = NJson::TJsonValue(NJson::JSON_MAP);
                for (const auto& label : diskVolumeAllocation.labels().attributes()) {
                    NJson2Yson::DeserializeYsonAsJsonValue(label.value(), &(labels[label.key()]), true);
                }
            }

            currentDiskVolumeAllocation["capacity"] = diskVolumeAllocation.capacity();

            currentDiskVolumeAllocation["resource_id"] = diskVolumeAllocation.resource_id();
            currentDiskVolumeAllocation["volume_id"] = diskVolumeAllocation.volume_id();
            currentDiskVolumeAllocation["device"] = diskVolumeAllocation.device();

            currentDiskVolumeAllocation["read_bandwidth_guarantee"] = diskVolumeAllocation.read_bandwidth_guarantee();
            currentDiskVolumeAllocation["read_bandwidth_limit"] = diskVolumeAllocation.read_bandwidth_limit();

            currentDiskVolumeAllocation["write_bandwidth_guarantee"] = diskVolumeAllocation.write_bandwidth_guarantee();
            currentDiskVolumeAllocation["write_bandwidth_limit"] = diskVolumeAllocation.write_bandwidth_limit();

            currentDiskVolumeAllocation["read_operation_rate_guarantee"] = diskVolumeAllocation.read_operation_rate_guarantee();
            currentDiskVolumeAllocation["read_operation_rate_limit"] = diskVolumeAllocation.read_operation_rate_limit();

            currentDiskVolumeAllocation["write_operation_rate_guarantee"] = diskVolumeAllocation.write_operation_rate_guarantee();
            currentDiskVolumeAllocation["write_operation_rate_limit"] = diskVolumeAllocation.write_operation_rate_limit();

            diskVolumeAllocations.AppendValue(currentDiskVolumeAllocation);
        }
    }

    return ToString(result);
}

void TCoreService::UpdateConditionSensors() {
    // Get lightweight status with conditions only to update sensors
    TreesGenerator_->GetStatus(true);
}

void TCoreService::TryUpdateSpecFromSavePath(TLogFramePtr logFrame) {
    try {
        const TString saveFilePath = Config_.GetCache().GetVolumePath() + "/" + Config_.GetCache().GetSaveSpecFileName();
        if (NFs::Exists(saveFilePath)) {
            API::TPodAgentRequest spec = NCacheFile::LoadFromFileBin<API::TPodAgentRequest>(saveFilePath);
            UpdatePodAgentRequest(spec);
        }
    } catch (const yexception& e) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TSaveSpecReadError(e.what()));
    }
}

void TCoreService::SaveSpecToSavePath(const API::TPodAgentRequest& spec, TLogFramePtr logFrame) {
    try {
        NCacheFile::SaveToFileBin(Config_.GetCache().GetVolumePath() + "/" + Config_.GetCache().GetSaveSpecFileName(), spec);
        NCacheFile::SaveToFileJson(Config_.GetCache().GetVolumePath() + "/" + Config_.GetCache().GetHumanReadableSaveSpecFileName(), spec);
        API::TPodAgentRequest patchedSpecToSave = HideSecrets(spec);
        patchedSpecToSave = PatchPodDynamicAttributes(patchedSpecToSave);
        NCacheFile::SaveToFileJson(Config_.GetPublicVolume().GetVolumePath() + "/" + Config_.GetPublicVolume().GetHumanReadableSaveSpecFileName(), patchedSpecToSave);
    } catch (const yexception& e) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TSaveSpecWriteError(e.what()));
    }
}

API::TPodAgentRequest TCoreService::HideSecrets(const API::TPodAgentRequest& spec) {
    API::TPodAgentRequest result = spec;

    for (auto& mutableSecret : *(result.mutable_secrets())) {
        mutableSecret.set_id(HIDDEN_SECRET_ATTRIBUTE_ID_PREFIX + mutableSecret.id());
        for (auto& value : *(mutableSecret.mutable_values())) {
            value.set_value(HIDDEN_SECRET_ATTRIBUTE_VALUE);
            value.set_encoding(HIDDEN_SECRET_ATTRIBUTE_VALUE);
        }
        for (auto& [key, value] : *mutableSecret.mutable_payload()) {
            value = HIDDEN_SECRET_ATTRIBUTE_VALUE;
        }
    }

    return result;
}


API::TPodAgentRequest TCoreService::PatchPodDynamicAttributes(const API::TPodAgentRequest& spec) {
    API::TPodAgentRequest result = spec;

    for (auto& it : *(result.mutable_pod_dynamic_attributes()->mutable_labels()->mutable_attributes())) {
        NJson::TJsonValue label;
        NJson2Yson::DeserializeYsonAsJsonValue(it.value(), &label, true);
        it.set_value(label.GetStringRobust());
    }

    for (auto& it : *(result.mutable_pod_dynamic_attributes()->mutable_annotations()->mutable_attributes())) {
        NJson::TJsonValue annotation;
        NJson2Yson::DeserializeYsonAsJsonValue(it.value(), &annotation, true);
        it.set_value(annotation.GetStringRobust());
    }

    return result;
}

} // namespace NInfra::NPodAgent
