#include "garbage_collector_job.h"

#include <util/datetime/base.h>
#include <util/generic/set.h>
#include <util/string/vector.h>

namespace NInfra::NPodAgent {

void TGarbageCollectorJob::Run() {
    if (!AtomicGet(Active_)) {
        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TGarbageCollectorJobUnactiveSkip());
        return; // Garbage collector unactive => to nothing
    }

    try {
        // stop ticker and run GC
        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TGarbageCollectorJobStart());
        StatusNTickerHolder_->StopTicker();

        // note: the order of GC is important
        RemoveContainers();
        UnlinkVolumes();
        RemoveVolumesAndBoxesDirectories();
        for (const auto& it : Places_) {
            RemoveLayers(it.first);
        }
        for (const auto& it : Places_) {
            RemoveStaticResourcesDirectories(PathHolder_->GetStaticResourcesDirectory(it.first), it.first);
        }
        for (const auto& it : Places_) {
            RemoveStorages(it.first);
        }
        RemoveIpAddresses(StatusNTickerHolder_->GetBoxIp6Subnet112Base());
        RemoveHttpRequests();

    } catch (const yexception& e) {
        NLogEvent::TGarbageCollectorJobException ev;
        ev.SetMessage(CurrentExceptionMessage());
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ev);
    }

    try {
        // start ticker
        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TGarbageCollectorJobFinish());
        StatusNTickerHolder_->StartTicker();
    } catch (const yexception& e) {
        NLogEvent::TGarbageCollectorJobException ev;
        ev.SetMessage(CurrentExceptionMessage());
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ev);
    }
}

void TGarbageCollectorJob::SetActive(bool active) {
    AtomicSet(Active_, active);
}

void TGarbageCollectorJob::RemoveContainers() {
    // DEPLOY-470
    // for porto 4.18.* Porto_->List(ContainersPrefix_ + "***") works the same as Porto_->List(ContainersPrefix_ + "*")
    // so we list all containers and filter them by prefix
    const auto listResult = Porto_->List();
    if (!(bool)listResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(listResult.Error()));
        return; // Porto_->List fail => nothing to do
    }

    const auto containers = listResult.Success();

    for (const TPortoContainerName& container : containers) {
        if (!ToString(container).StartsWith(ContainersPrefix_)) {
            continue;
        }

        if (container == PathHolder_->GetResourceGangMetaContainer()) {
            // Skip Resource gang meta container
        } else if (TIdExtractionResult layerId = PathHolder_->ExtractLayerHashFromContainer(container)) {
            if (!StatusNTickerHolder_->HasLayerDownloadHash(layerId.Success())) {
                DestroyContainer(container);
            }
        } else if (TIdExtractionResult staticResourceId = PathHolder_->ExtractStaticResourceHashFromContainer(container)) {
            if (!StatusNTickerHolder_->HasStaticResourceDownloadHash(staticResourceId.Success())) {
                DestroyContainer(container);
            }
        } else if (TIdExtractionResult boxId = PathHolder_->ExtractBoxIdFromContainer(container)) {
            if (StatusNTickerHolder_->HasBox(boxId.Success())) {
                // if it is not root if boxId
                if (!container.IsSimple()) {
                    if (auto workloadId = PathHolder_->ExtractWorkloadIdFromContainer(container)) {
                        // try to parse as workload container
                        if (!StatusNTickerHolder_->HasWorkload(workloadId.Success())) {
                            DestroyContainer(container);
                        }
                    } else if ((bool)PathHolder_->ExtractBoxInitIdFromContainer(container)) {
                        // try to parse as box init container
                    } else {
                        // all other containers are unknown

                        // DEPLOY-528
                        // all unknown box containers are the responsibility of the user
                        NLogEvent::TGarbageCollectorJobUnknownBoxContainer containerEv;
                        containerEv.SetBox(boxId.Success());
                        containerEv.SetContainer(TString(container));
                        LogFrame_->LogEvent(ELogPriority::TLOG_WARNING, containerEv);
                    }
                }
            } else {
                // Remove only root of boxId
                // Other containers will remove with root
                if (container.IsSimple()) {
                    DestroyContainer(container);
                }
            }
        } else if (TIdExtractionResult workloadId = PathHolder_->ExtractWorkloadIdFromContainer(container)) {
            if (!StatusNTickerHolder_->HasWorkload(workloadId.Success())) {
                DestroyContainer(container);
            }
        } else {
            DestroyContainer(container, true);
        }
    }
}


void TGarbageCollectorJob::RemoveDirectories(const TString& storageDir, std::function<bool(const TString&)>&& checkIsActiveDirectory) {
    const auto existsResult = Posix_->ExistsAsync(storageDir).ExtractValueSync();
    if (!(bool)existsResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(existsResult.Error()));
        return; // Posix_->Exists fail => nothing to do
    }

    bool existsPath = existsResult.Success();
    if (existsPath) {
        const auto listResult = Posix_->ListAsync(storageDir).ExtractValueSync();
        if (!(bool)listResult) {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(listResult.Error()));
        } else {
            auto folders = listResult.Success();
            for (const auto& folderName : folders) {
                TFsPath folderPath = TFsPath(storageDir) / folderName;
                if (!checkIsActiveDirectory(folderPath)) {
                    NLogEvent::TGarbageCollectorJobClearDirectory directoryEv;
                    directoryEv.SetDirectory(folderPath);
                    LogFrame_->LogEvent(ELogPriority::TLOG_INFO, directoryEv);

                    const auto result = Posix_->RemoveRecursiveAsync(folderPath).ExtractValueSync();
                    if (!(bool) result) {
                        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(result.Error()));
                    }
                }
            }
        }
    }
}

void TGarbageCollectorJob::RemoveLayers(const TString& place) {
    const auto layerListResult = Porto_->ListLayers(place);
    if (!(bool)layerListResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(layerListResult.Error()));
        return; // Porto_->ListLayers fail => nothing to do
    }
    auto layers = layerListResult.Success();

    for (const auto& layer : layers) {
        if (layer.name().StartsWith(LayerPrefix_)) {
            const TString layerDownloadHash = layer.name().substr(LayerPrefix_.size());

            if (!StatusNTickerHolder_->HasLayerDownloadHash(layerDownloadHash)) {
                NLogEvent::TGarbageCollectorJobClearLayer layerEv;
                layerEv.SetLayer(layer.name());
                layerEv.SetPlace(place);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, layerEv);

                const auto removeResult = Porto_->RemoveLayer(layer.name(), place);
                if (!(bool)removeResult) {
                    LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(removeResult.Error()));
                }
            }
        }
    }

    const TFsPath layersDirectory = PathHolder_->GetLayersDirectory(place);
    const auto existsResult = Posix_->ExistsAsync(layersDirectory).ExtractValueSync();
    if (!(bool)existsResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(existsResult.Error()));
        return; // Posix_->Exists fail => nothing to do
    }

    bool existsPath = existsResult.Success();
    if (existsPath) {
        const auto foldersListResult = Posix_->ListAsync(layersDirectory).ExtractValueSync();
        if (!(bool)foldersListResult) {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(foldersListResult.Error()));
        } else {
            auto folders = foldersListResult.Success();
            for (const auto& folderName : folders) {
                if (!StatusNTickerHolder_->HasLayerDownloadHash(folderName)) {
                    NLogEvent::TGarbageCollectorJobClearDirectory directoryEv;
                    directoryEv.SetDirectory(folderName);
                    LogFrame_->LogEvent(ELogPriority::TLOG_INFO, directoryEv);

                    const auto removeResult = Posix_->RemoveRecursiveAsync(layersDirectory / folderName).ExtractValueSync();
                    if (!(bool)removeResult) {
                        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(removeResult.Error()));
                    }
                }
            }
        }
    }
}

void TGarbageCollectorJob::RemoveStorages(const TString& place) {
    const auto storageListResult = Porto_->ListStorages(place, "");
    if (!(bool)storageListResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(storageListResult.Error()));
        return; // Porto_->ListStorages fail => nothing to do
    }

    auto storages = storageListResult.Success();
    for (const auto& storage : storages) {
        bool needToRemove = false;
        bool hasOnwer = false;
        TIdExtractionResult id = PathHolder_->ExtractBoxIdFromStorage(storage.name());
        if (id) {
            hasOnwer = true;
            if (!StatusNTickerHolder_->HasBox(id.Success())) {
                needToRemove = true;
            }
        }
        id = PathHolder_->ExtractVolumeIdFromStorage(storage.name());
        if (id) {
            hasOnwer = true;
            if (!StatusNTickerHolder_->HasVolume(id.Success())) {
                needToRemove = true;
            }
        }
        if (needToRemove) {
            DestroyStorage(storage.name(), place, false);
        } else {
            if (!hasOnwer && storage.name().StartsWith(PersistentStoragePrefix_)) {
                DestroyStorage(storage.name(), place, true);
            }
        }
    }
}

void TGarbageCollectorJob::UnlinkVolumes() {
    const auto listResult = Porto_->ListVolumes();
    if (!(bool)listResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(listResult.Error()));
        return; // Porto_->ListVolumes fail => nothing to do
    }

    const auto volumes = listResult.Success();
    for (const auto& volume : volumes) {
        const TString& path = volume.path();
        if (volume.path().StartsWith(VolumesStorage_)) {
            if (!IsActiveBoxPath(path) && !IsActiveVolumePath(path)) {
                NLogEvent::TGarbageCollectorJobUnlinkVolume volumeEv;
                volumeEv.SetVolume(path);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, volumeEv);

                auto unlinkResult = Porto_->UnlinkVolume(path, TPortoContainerName::NoEscape("."), path);

                // TODO(DEPLOY-1853, PORTO-644) fix this after fix in porto
                if (!unlinkResult && unlinkResult.Error().Code == EPortoError::VolumeNotLinked) {
                    unlinkResult = Porto_->UnlinkVolume(path, TPortoContainerName::NoEscape("***"));
                }

                if (!unlinkResult && unlinkResult.Error().Code != EPortoError::VolumeNotFound) {
                    LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(unlinkResult.Error()));
                }
            }
        }
    }
}

void TGarbageCollectorJob::RemoveIpAddresses(const TString& ip6Subnet112Base) {
    if (ip6Subnet112Base == "") {
        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TGarbageCollectorJobEmptyIpSubnet());
        return; // No subnet provided => nothing to do
    }

    if (!ip6Subnet112Base.EndsWith(":")) {
        NLogEvent::TTGarbageCollectorJobBadIpSubnet badIpSubnetEv;
        badIpSubnetEv.SetSubnet(ip6Subnet112Base);
        badIpSubnetEv.SetMessage("IpSubnet112Base must ends with ':'");
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, badIpSubnetEv);

        return; // Bad subnet => nothing to do
    }

    const auto listResult = IpClient_->ListAddress(IpDevice_);
    if (!(bool)listResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(listResult.Error()));
        return; // IpClient_->ListAddress fail => nothing to do
    }

    const TString zeroIp = ip6Subnet112Base + "0";

    const auto ips = listResult.Success();
    for (const auto& ip : ips) {
        if (ip.Ip6.StartsWith(ip6Subnet112Base)
            && ip.Ip6 != zeroIp
            && ip.Subnet == 128
            && !StatusNTickerHolder_->HasBoxIp6Address(ip.Ip6)
        ) {
            NLogEvent::TGarbageCollectorJobRemoveIpAddress removeIpAddressEv;
            removeIpAddressEv.SetIp6(ip.Ip6);
            removeIpAddressEv.SetSubnet(ip.Subnet);
            LogFrame_->LogEvent(ELogPriority::TLOG_INFO, removeIpAddressEv);

            auto removeAddressResult = IpClient_->RemoveAddress(IpDevice_, ip);
            if (!removeAddressResult) {
                LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(removeAddressResult.Error()));
            }
        }
    }
}

void TGarbageCollectorJob::RemoveHttpRequests() {
    const auto listResult = NetworkClient_->ListRequests();
    if (!(bool)listResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(listResult.Error()));
        return; // NetworkClient_->ListHttpRequests fail => nothing to do
    }

    const auto httpRequests = listResult.Success();
    for (const auto& httpRequest : httpRequests) {
        // All trees write workloadId to AdditionalInfo
        const TString& workloadId = httpRequest.AdditionalInfo_;
        if (!StatusNTickerHolder_->HasWorkload(workloadId)) {
            NLogEvent::TGarbageCollectorJobRemoveHttpRequest removeHttpRequestEv;
            removeHttpRequestEv.SetKey(httpRequest.Key_);
            removeHttpRequestEv.SetHash(httpRequest.Hash_);
            removeHttpRequestEv.SetAdditionalInfo(httpRequest.AdditionalInfo_);
            LogFrame_->LogEvent(ELogPriority::TLOG_INFO, removeHttpRequestEv);

            auto removeHttpRequestResult = NetworkClient_->RemoveRequest(httpRequest.Key_);
            if (!removeHttpRequestResult) {
                LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(removeHttpRequestResult.Error()));
            }
        }
    }
}

void TGarbageCollectorJob::RemoveVolumesAndBoxesDirectories() {
    RemoveDirectories(VolumesStorage_, [this](const TString& folderPath) {
        return IsActiveVolumePath(folderPath) || IsActiveBoxPath(folderPath);
    });
}

void TGarbageCollectorJob::RemoveStaticResourcesDirectories(const TString& storageDir, const TString& place) {
    RemoveDirectories(storageDir, [this, &place](const TString& folderPath) {
        return IsActiveStaticResourcePath(folderPath, place);
    });
}

bool TGarbageCollectorJob::IsActiveStaticResourcePath(const TString& path, const TString& place) const {
    auto extractionResult = PathHolder_->ExtractStaticResourceHashFromPath(path, place);
    if (extractionResult) {
        return StatusNTickerHolder_->HasStaticResourceDownloadHash(extractionResult.Success());
    }

    return false;
}

bool TGarbageCollectorJob::IsActiveBoxPath(const TString& path) const {
    auto extractionResult = PathHolder_->ExtractBoxIdFromPath(path);
    if (extractionResult) {
        return StatusNTickerHolder_->HasBox(extractionResult.Success());
    }

    return false;
}

bool TGarbageCollectorJob::IsActiveVolumePath(const TString& path) const {
    auto extractionResult = PathHolder_->ExtractVolumeIdFromPath(path);
    if (extractionResult) {
        return StatusNTickerHolder_->HasVolume(extractionResult.Success());
    }

    return false;
}

void TGarbageCollectorJob::DestroyContainer(const TPortoContainerName& container, bool isUnknown) {
    if (isUnknown) {
        NLogEvent::TGarbageCollectorJobUnknownContainer containerEv;
        containerEv.SetContainer(TString(container));
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, containerEv);
    }

    NLogEvent::TGarbageCollectorJobClearContainer containerEv;
    containerEv.SetContainer(TString(container));
    LogFrame_->LogEvent(ELogPriority::TLOG_INFO, containerEv);

    const auto destroyResult = Porto_->Destroy(container);
    if (!(bool)destroyResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(destroyResult.Error()));
    }
}

void TGarbageCollectorJob::DestroyStorage(const TString& storage, const TString& place, bool isUnknown) {
    if (isUnknown) {
        NLogEvent::TGarbageCollectorJobClearUnknownStorage storageEv;
        storageEv.SetStorage(storage);
        storageEv.SetPlace(place);
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, storageEv);
    } else {
        NLogEvent::TGarbageCollectorJobClearStorage storageEv;
        storageEv.SetStorage(storage);
        storageEv.SetPlace(place);
        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, storageEv);
    }
    const auto removeResult = Porto_->RemoveStorage(storage, place);
    if (!(bool) removeResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(removeResult.Error()));
    }
}

NLogEvent::TGarbageCollectorJobException TGarbageCollectorJob::ConstructExceptionEvent(const std::variant<TPortoError, TPosixError, TIpClientError, TNetworkClientError>& error) {
    if (std::holds_alternative<TPortoError>(error)) {
        NLogEvent::TGarbageCollectorJobException ev;
        ev.SetMessage(ToString(std::get<TPortoError>(error)));
        return ev;
    }

    if (std::holds_alternative<TPosixError>(error)) {
        NLogEvent::TGarbageCollectorJobException ev;
        ev.SetMessage(TStringBuilder()
            << "PosixError:"
            << ToString(std::get<TPosixError>(error).Errno)
            << ":" << std::get<TPosixError>(error).Message
        );
        return ev;
    }

    if (std::holds_alternative<TIpClientError>(error)) {
        NLogEvent::TGarbageCollectorJobException ev;
        ev.SetMessage(TStringBuilder()
            << "IpClientError:"
            << ToString(std::get<TIpClientError>(error).Errno)
            << ":" << std::get<TIpClientError>(error).Message
        );
        return ev;
    }

    if (std::holds_alternative<TNetworkClientError>(error)) {
        NLogEvent::TGarbageCollectorJobException ev;
        ev.SetMessage(TStringBuilder()
            << "TNetworkClientError:"
            << ToString(std::get<TNetworkClientError>(error).Errno)
            << ":" << std::get<TNetworkClientError>(error).Message
        );
        return ev;
    }

    ythrow yexception() << "Unable to construct exception event";
}

} // namespace NInfra::NPodAgent
