#pragma once

#include <infra/pod_agent/libs/posix_worker/posix_worker.h>

#include <infra/pod_agent/libs/config/config.pb.h>
#include <infra/pod_agent/libs/ip_client/client.h>
#include <infra/pod_agent/libs/network_client/client.h>
#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/period_job_worker/period_job.h>
#include <infra/pod_agent/libs/pod_agent/status_and_ticker_holder/status_and_ticker_holder.h>
#include <infra/pod_agent/libs/porto_client/client.h>

#include <util/folder/path.h>
#include <util/system/condvar.h>
#include <util/system/fs.h>
#include <util/system/thread.h>

namespace NInfra::NPodAgent {

class TGarbageCollectorJob;
using TGarbageCollectorJobPtr = TIntrusivePtr<TGarbageCollectorJob>;

/**
    Looks at StatusRepository for object ids
    Removes unneeded containers/volumes/layers/dirs

    Run function "stop the world" for work:
        1. stops given PeriodTicker
        2. clears unused resources
        3. starts given PeriodTicker
*/
class TGarbageCollectorJob : public IPeriodJob {
public:
    TGarbageCollectorJob(
        const TDuration& period
        , TLogFramePtr logFrame
        , TStatusNTickerHolderPtr statusNTickerHolder
        , TPortoClientPtr porto
        , TPosixWorkerPtr posix
        , TPathHolderPtr pathHolder
        , TIpClientPtr ipClient
        , TNetworkClientPtr networkClient
        , const TMap<TString, TString>& places
        , const TString& volumesStorage
        , const TString& layerPrefix
        , const TString& persistentStoragePrefix
        , const TString& cacheStorage
        , const TString& resourcesVolumeStoragePrefix
        , const TString& containersPrefix
        , const TString& ipDevice
    )
        : IPeriodJob("GarbageCollectorJob", period, logFrame)
        , LogFrame_(logFrame)
        , StatusNTickerHolder_(statusNTickerHolder)
        , Porto_(porto)
        , Posix_(posix)
        , PathHolder_(pathHolder)
        , IpClient_(ipClient)
        , NetworkClient_(networkClient)
        , Places_(places)
        , VolumesStorage_(volumesStorage.StartsWith('/') ? volumesStorage : NFs::CurrentWorkingDirectory() + "/" + volumesStorage)
        , LayerPrefix_(layerPrefix)
        , PersistentStoragePrefix_(persistentStoragePrefix)
        , ContainersPrefix_(containersPrefix)
        , IpDevice_(ipDevice)
        , Active_(false)
    {
        Y_ENSURE(!cacheStorage.StartsWith(PersistentStoragePrefix_), "cache storage should not share prefix with other persistent storages");
        Y_ENSURE(!cacheStorage.StartsWith(resourcesVolumeStoragePrefix), "cache storage should not share prefix with other persistent storages");
        Y_ENSURE(!resourcesVolumeStoragePrefix.StartsWith(PersistentStoragePrefix_), "resources storage should not share prefix with other persistent storages");
    }

    virtual void Run() override;

    void SetActive(bool active);

private:
    void RemoveContainers();
    void RemoveDirectories(const TString& storageDir, std::function<bool(const TString&)>&& checkIsActiveDirectory);
    void UnlinkVolumes();
    void RemoveLayers(const TString& place);
    void RemoveStorages(const TString& place);
    void RemoveHttpRequests();
    void RemoveVolumesAndBoxesDirectories();
    void RemoveStaticResourcesDirectories(const TString& storageDir, const TString& place);

    // Remove ip from 1 to 2^16 - 1 from 112 subnet which were created with 128 subnet
    // 0 ip will never be deleted because it matches the entire subnet
    // ipSubnet112Base is first 112 bits of subnet in format '2a02:6b8:c08:68a3:0:696:6937:'
    // If ipSubnet112Base is empty, do nothing
    void RemoveIpAddresses(const TString& ip6Subnet112Base);

    bool IsActiveBoxPath(const TString& path) const;
    bool IsActiveVolumePath(const TString& path) const;
    bool IsActiveStaticResourcePath(const TString& path, const TString& place) const;

    // destroy container and write to log
    void DestroyContainer(const TPortoContainerName& container, bool isUnknown = false);
    void DestroyStorage(const TString& storage, const TString& place, bool isUnknown = false);

    static NLogEvent::TGarbageCollectorJobException ConstructExceptionEvent(const std::variant<TPortoError, TPosixError, TIpClientError, TNetworkClientError>& error);

private:
    TLogFramePtr LogFrame_;
    TStatusNTickerHolderPtr StatusNTickerHolder_;

    TPortoClientPtr Porto_;
    TPosixWorkerPtr Posix_;
    TPathHolderPtr PathHolder_;

    TIpClientPtr IpClient_;

    TNetworkClientPtr NetworkClient_;

    const TMap<TString, TString> Places_; // Place -> DownloadVolumePath
    const TString VolumesStorage_;
    const TString LayerPrefix_;
    const TString PersistentStoragePrefix_;
    const TString ContainersPrefix_;
    const TString IpDevice_;

    // Is GarbageCollector active
    // If not do nothing in Run function
    TAtomic Active_;
};

} // namespace NInfra::NPodAgent
