#include "garbage_collector_job.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/test_functions.h>
#include <infra/pod_agent/libs/ip_client/mock_client.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/period_job_worker/period_job_worker.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/system_logs_sender/mock_system_logs_sender.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/dirut.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NGarbageCollectorJobTest {

static TLogger logger({});

const TString CWD = NFs::CurrentWorkingDirectory();
const TString RESOURCES_FOLDER = "resources";
const TString RESOURCES_FOLDER_FAST = "resources_fast";
const TString RESOURCES_FOLDER_BAD = "resources_bad";
const TString CACHE_VOLUME_FOLDER = "cache";
const TString CACHE_VOLUME_PATH = CWD + "/" + CACHE_VOLUME_FOLDER;
const TString LAYER_PREFIX = "Pod_Agent_test_" + ToString(TThread::CurrentThreadId()) + "_";
const TString PERSISTENT_STORAGE_PREFIX = "Pod_Agent_test_storage_" + ToString(TThread::CurrentThreadId()) + "_";
const TString VOLUMES_STORAGE = "volumes_storage";
const TString CONTAINERS_PREFIX = "PodAgent_" + ToString(TThread::CurrentThreadId()) + "_";
const TString RESOURCE_GANG_META_CONTAINER = CONTAINERS_PREFIX + "resource_gang_meta";
const TString PORTO_LOG_FILES_PATH = "porto_log_files";
const TString RBIND_VOLUME_STORAGE_DIR = CWD + "/rbind_volumes";
const TString IP_DEVICE = "ip_device";
const TString IP6_SUBNET112_BASE = "2a02:6b8:c08:68a3:0:696:6937:";

const TMap<TString, TString> GOOD_PLACES = {
    {"///place", CWD + "/" + RESOURCES_FOLDER}
    , {"///fast", CWD + "/" + RESOURCES_FOLDER_FAST}
};
const TMap<TString, TString> BAD_PLACES = {
    {"///unknown", CWD + "/" + RESOURCES_FOLDER_BAD + "_1"}
    , {"///place1", CWD + "/" + RESOURCES_FOLDER_BAD + "_2"}
};
// Must be equal to GOOD_PLACES + BAD_PLACES
const TMap<TString, TString> ALL_PLACES = {
    {"///place", CWD + "/" + RESOURCES_FOLDER}
    , {"///fast", CWD + "/" + RESOURCES_FOLDER_FAST}
    , {"///unknown", CWD + "/" + RESOURCES_FOLDER_BAD + "_1"}
    , {"///place1", CWD + "/" + RESOURCES_FOLDER_BAD + "_2"}
};

const TString DOM0_POD_ROOT = "/root";
const TMap<TString, TString> GOOD_VIRTUAL_DISKS = {
    {"virtual_disk_place", "///place"}
    , {"virtual_disk_fast", "///fast"}
};
const TMap<TString, TString> BAD_VIRTUAL_DISKS = {
    {"virtual_disk_unknown", "///unknown"}
    , {"virtual_disk_place1", "///place1"}
};
// Must be equal to GOOD_VIRTUAL_DISKS + BAD_VIRTUAL_DISKS
const TMap<TString, TString> ALL_VIRTUAL_DISKS = {
    {"virtual_disk_place", "///place"}
    , {"virtual_disk_fast", "///fast"}
    , {"virtual_disk_unknown", "///unknown"}
    , {"virtual_disk_place1", "///place1"}
};

TConfig CreateConfig() {
    TConfig cfg;
    *cfg.MutableVolumes()->MutableStorage() = VOLUMES_STORAGE;
    cfg.MutableGarbageCollector()->SetPeriodMs(100);
    *cfg.MutablePorto()->MutableLayerPrefix() = LAYER_PREFIX;
    *cfg.MutablePorto()->MutablePersistentStoragePrefix() = PERSISTENT_STORAGE_PREFIX;
    *cfg.MutablePorto()->MutableContainersPrefix() = CONTAINERS_PREFIX;
    *cfg.MutableIp()->MutableDevice() = IP_DEVICE;
    return cfg;
}

void EnsureDoesNotExist(const TString& path) {
    if (NFs::Exists(path)) {
        try {
            NFs::RemoveRecursive(path);
        } catch(...) {
            NFs::Remove(path);
        }
    }
}

TPosixWorkerPtr CreatePosixWorker(){
    TAtomicSharedPtr<TFakeThreadPool> queue = new TFakeThreadPool;
    TPosixWorkerPtr posix = new TPosixWorker(queue);
    return posix;
}

TMtpPeriodTickerPtr GetTestMtpPeriodTicker() {
    TMtpPeriodTickerPtr ticker = new TMtpPeriodTicker(TBehaviorTickerConfig{}, logger.SpawnFrame(), logger.SpawnFrame());
    return ticker;
}

struct TTestPortoClient : public TMockPortoClient {
    TExpected<TVector<TPortoContainerName>, TPortoError> List(const TString& mask) override {
        ++ListCalls;
        ListLastMask = mask;
        TVector<TPortoContainerName> result;

        result.push_back(TPortoContainerName("pod_agent"));
        result.push_back(TPortoContainerName({"pod_agent"}, "user_container"));
        result.push_back(TPortoContainerName("random_name"));
        result.push_back(TPortoContainerName(RESOURCE_GANG_META_CONTAINER));

        if (HaveStaticResources) {
            result.push_back(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "static_static_resource1_download_hash_verify"}));
            result.push_back(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "static_static_resource1_download_hash_download"}));
            result.push_back(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "static_static_resource2_download_hash_download"}));
        }

        if (HaveLayers) {
            result.push_back(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "layer_layer1_download_hash_verify"}));
            result.push_back(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "layer_layer1_download_hash_download"}));
            result.push_back(TPortoContainerName({RESOURCE_GANG_META_CONTAINER,  "layer_layer2_download_hash_download"}));
        }

        if (HaveBoxes) {
            result.push_back(TPortoContainerName(CONTAINERS_PREFIX + "box_box1"));
            result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "init1"));
            result.push_back(TPortoContainerName(CONTAINERS_PREFIX + "box_box2"));
        }

        if (HaveWorkloads) {
            UNIT_ASSERT(HaveBoxes);
            result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload1_start"));
            result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload1_stop"));
            result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload2_start"));
        }

        if (HaveOther) {
            result.push_back(TPortoContainerName(CONTAINERS_PREFIX + "some_other"));
            result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "some_other"}, "resource"));
            if (HaveBoxes) {
                result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "some_other"));
            }
            if (HaveWorkloads) {
                UNIT_ASSERT(HaveBoxes);
                result.push_back(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload1_no_name"));
            }
        }

        return result;
    }
    TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
        ++DestroyCalls;
        DestroyLastName = name;
        return TExpected<void, TPortoError>::DefaultSuccess();
    }
    TExpected <TString, TPortoError> CreateVolume(
        const TString& path
        , const TString&
        , const TString&
        , const TVector<TString>&
        , unsigned long long
        , const TString&
        , const EPortoVolumeBackend
        , const TPortoContainerName&
        , const TVector<TPortoVolumeShare>&
        , bool
    ) override {
        Volumes[path] = true;
        return path;
    }
    TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName&, const TString&, bool) override {
        Volumes[path] = false;
        return TExpected<void, TPortoError>::DefaultSuccess();
    }
    TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString&, const TPortoContainerName&) override { // only linked
        TVector<TPortoVolume> res;
        for (const auto& it : Volumes) {
            if (it.second) {
                TPortoVolume new_volume;
                new_volume.set_path(it.first);
                new_volume.mutable_links()->Add();
                res.push_back(new_volume);
            }
        }
        return res;
    }
    TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString&, bool, const TString& place, const TString&) override {
        Layers[place].insert(layer);
        return TExpected<void, TPortoError>::DefaultSuccess();
    }
    TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place) override {
        Layers[place].erase(layer);
        return TExpected<void, TPortoError>::DefaultSuccess();
    }
    TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString&) override {
        TVector<TPortoLayer> res;
        for (const auto& it : Layers[place]) {
           TPortoLayer new_layer;
           new_layer.set_name(it);
           new_layer.set_owner_user("");
           new_layer.set_owner_group("");
           new_layer.set_private_value("");
           new_layer.set_last_usage(0);
           res.push_back(new_layer);
        }
        return res;
    }
    TExpected<void, TPortoError> ImportStorage(const TString& storage, const TString&, const TString& place, const TString&, const TString&) override {
        Storages[place].insert(storage);
        return TExpected<void, TPortoError>::DefaultSuccess();
    }
    TExpected<void, TPortoError> RemoveStorage(const TString& storage, const TString& place) override {
        Storages[place].erase(storage);
        return TExpected<void, TPortoError>::DefaultSuccess();
    }
    TExpected<TVector<TPortoStorage>, TPortoError> ListStorages(const TString& place, const TString&) override {
        TVector<TPortoStorage> res;
        for (const auto& it : Storages[place]) {
            TPortoStorage new_storage;
            new_storage.set_name(it);
            new_storage.set_owner_user("");
            new_storage.set_owner_group("");
            new_storage.set_private_value("");
            new_storage.set_last_usage(0);
            res.push_back(new_storage);
        }
        return res;
    }

    TMap<TString, TSet<TString>> Layers;
    TMap<TString, TSet<TString>> Storages;
    TMap<TString, bool> Volumes; // {key: volumePath, value: isLinked}
    size_t ListCalls = 0;
    TString ListLastMask = "";
    size_t DestroyCalls = 0;
    TPortoContainerName DestroyLastName = {""};

    bool HaveLayers = false;
    bool HaveStaticResources = false;
    bool HaveBoxes = false;
    bool HaveWorkloads = false;
    bool HaveOther = false;
};

struct TIpDescriptionCmp {
    bool operator()(const TIpDescription& a, const TIpDescription& b) const {
        return a.Ip6 < b.Ip6 || (a.Ip6 == b.Ip6 && a.Subnet < b.Subnet);
    }
};

struct TTestIpClient : public TMockIpClient {
    TExpected<void, TIpClientError> AddAddress(const TString& device, const TIpDescription& ip) override {
        Ips[device].insert(ip);
        return TExpected<void, TIpClientError>::DefaultSuccess();
    }

    TExpected<void, TIpClientError> RemoveAddress(const TString& device, const TIpDescription& ip) override {
        ++RemoveCalls;
        Ips[device].erase(ip);
        return TExpected<void, TIpClientError>::DefaultSuccess();
    }

    TExpected<TVector<TIpDescription>, TIpClientError> ListAddress(const TString& device) override {
        ++ListCalls;
        TVector<TIpDescription> res;
        for (const TIpDescription& ip : Ips[device]) {
            res.push_back(ip);
        }
        return res;
    }

    void GenIpAddresses(const TString& device, const TString& ipSubnet112Base, ui32 subnet, ui32 from, ui32 to) {
        for (ui32 i = from; i <= to; ++i) {
            AddAddress(device, TIpDescription(ipSubnet112Base + ToString(i), subnet));
        }
    }

    TMap<TString, TSet<TIpDescription, TIpDescriptionCmp>> Ips;
    size_t RemoveCalls = 0;
    size_t ListCalls = 0;
};

struct TNetworkClientRequestPublicInfoCmp {
    bool operator()(const INetworkClient::TRequestPublicInfo& a, const INetworkClient::TRequestPublicInfo& b) const {
        return std::tuple(a.Key_, a.Hash_, a.AdditionalInfo_) < std::tuple(b.Key_, b.Hash_, b.AdditionalInfo_);
    }
};

struct TTestNetworkClient : public TMockNetworkClient {
    TExpected<void, TNetworkClientError> CheckAndAddHttpRequest(
        const TString& requestKey
        , const TString& requestHash
        , const TString& additionalInfo
        , const TString& /* host */
        , ui32 /* port */
        , const TString& /* path */
        , TDuration /* timeout */
    ) override {
        HttpRequests.insert({requestKey, TRequestPublicInfo(requestKey, requestHash, additionalInfo)});
        return TExpected<void, TNetworkClientError>::DefaultSuccess();
    }

    TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
        ++RemoveCalls;
        HttpRequests.erase(requestKey);
        return TExpected<void, TNetworkClientError>::DefaultSuccess();
    }

    TExpected<TVector<TRequestPublicInfo>, TNetworkClientError> ListRequests() const override {
        ++ListCalls;
        TVector<TRequestPublicInfo> ret;
        for (const auto& [requestKey, requestInfo] : HttpRequests) {
            ret.push_back(requestInfo);
        }
        return ret;
    }

    void GenHttpRequests(const TString& requestKeyPrefix, const TString& requestHashPrefix, const TString& additionalInfoPrefix, ui32 from, ui32 to) {
        for (ui32 i = from; i <= to; ++i) {
            CheckAndAddHttpRequest(
                requestKeyPrefix + ToString(i)
                , requestHashPrefix + ToString(i)
                , additionalInfoPrefix + ToString(i)
                , ""
                , 0
                , ""
                , TDuration::Zero()
            );
        }
    }

    TMap<TString, TRequestPublicInfo> HttpRequests;
    mutable size_t RemoveCalls = 0;
    mutable size_t ListCalls = 0;
};

class ITestGarbageCollectorJobCanon {
public:
    ITestGarbageCollectorJobCanon()
        : Ticker_(new TMtpPeriodTicker(TBehaviorTickerConfig{}, logger.SpawnFrame(), logger.SpawnFrame()))
        , UpdateHolder_(new TUpdateHolder())
        , BoxStatusRepository_(new TBoxStatusRepository())
        , LayerStatusRepository_(new TLayerStatusRepository())
        , StaticResourceStatusRepository_(new TStaticResourceStatusRepository())
        , VolumeStatusRepository_(new TVolumeStatusRepository())
        , WorkloadStatusRepository_(new TWorkloadStatusRepository())
        , WorkloadStatusRepositoryInternal_(new TWorkloadStatusRepositoryInternal())
        , Posix_(CreatePosixWorker())
        , PathHolder_(
            new TPathHolder(
                DOM0_POD_ROOT
                , GOOD_VIRTUAL_DISKS
                , GOOD_PLACES
                , VOLUMES_STORAGE
                , PERSISTENT_STORAGE_PREFIX
                , LAYER_PREFIX
                , CONTAINERS_PREFIX
                , PORTO_LOG_FILES_PATH
                , RBIND_VOLUME_STORAGE_DIR
            )
        )
        , AllPlacesPathHolder_(
            new TPathHolder(
                DOM0_POD_ROOT
                , ALL_VIRTUAL_DISKS
                , ALL_PLACES
                , VOLUMES_STORAGE
                , PERSISTENT_STORAGE_PREFIX
                , LAYER_PREFIX
                , CONTAINERS_PREFIX
                , PORTO_LOG_FILES_PATH
                , RBIND_VOLUME_STORAGE_DIR
            )
        )
        , GCConfig_(CreateConfig())
    {
        StatusRepository_ = new TStatusRepository(
            UpdateHolder_
            , BoxStatusRepository_
            , LayerStatusRepository_
            , StaticResourceStatusRepository_
            , VolumeStatusRepository_
            , WorkloadStatusRepository_
        );
        StatusNTickerHolder_ = new TStatusNTickerHolder(
            Ticker_
            , StatusRepository_
            , WorkloadStatusRepositoryInternal_
            , new TMockSystemLogsSender()
            , new TMockSystemLogsSender()
            , /* isBoxAgentMode = */ false
        );
    }

    virtual ~ITestGarbageCollectorJobCanon() = default;

    void DoTest() {
        Porto_ = GetSpecificPorto(); // must call in derived
        IpClient_ = GetSpecificIpClient(); // must call in derived
        NetworkClient_ = GetSpecificNetworkClient(); // must call in derived
        Test();
    }

protected:
    virtual void Test() = 0;

    TGarbageCollectorJobPtr GetGCJob() {
        return new TGarbageCollectorJob(
            TDuration::MilliSeconds(GCConfig_.GetGarbageCollector().GetPeriodMs())
            , logger.SpawnFrame()
            , StatusNTickerHolder_
            , Porto_
            , Posix_
            , PathHolder_
            , IpClient_
            , NetworkClient_
            , GOOD_PLACES
            , GCConfig_.GetVolumes().GetStorage()
            , GCConfig_.GetPorto().GetLayerPrefix()
            , GCConfig_.GetPorto().GetPersistentStoragePrefix()
            , GCConfig_.GetResources().GetDownloadVolumePath()
            , GCConfig_.GetCache().GetStorage()
            , GCConfig_.GetPorto().GetContainersPrefix()
            , GCConfig_.GetIp().GetDevice()
        );
    }

    void RunGCJob(bool active = true) {
        StatusNTickerHolder_->StartTicker();

        TGarbageCollectorJobPtr job = GetGCJob();
        job->SetActive(active);
        job->Run();

        StatusNTickerHolder_->StopTicker();
    }

    virtual TPortoClientPtr GetSpecificPorto() const { // override for specific porto
        return new TTestPortoClient();
    }

    virtual TIpClientPtr GetSpecificIpClient() const { // override for specific ip client
        return new TTestIpClient();
    }

    virtual TNetworkClientPtr GetSpecificNetworkClient() const { // override for specific network client
        return new TTestNetworkClient();
    }

protected:
    TMtpPeriodTickerPtr Ticker_;
    TUpdateHolderPtr UpdateHolder_;

    TBoxStatusRepositoryPtr BoxStatusRepository_;
    TLayerStatusRepositoryPtr LayerStatusRepository_;
    TStaticResourceStatusRepositoryPtr StaticResourceStatusRepository_;
    TVolumeStatusRepositoryPtr VolumeStatusRepository_;
    TWorkloadStatusRepositoryPtr WorkloadStatusRepository_;

    TWorkloadStatusRepositoryInternalPtr WorkloadStatusRepositoryInternal_;

    TStatusRepositoryPtr StatusRepository_;
    TStatusNTickerHolderPtr StatusNTickerHolder_;

    TPosixWorkerPtr Posix_;
    TPathHolderPtr PathHolder_;
    TPathHolderPtr AllPlacesPathHolder_;
    TPortoClientPtr Porto_;

    TIpClientPtr IpClient_;

    TNetworkClientPtr NetworkClient_;

    TConfig GCConfig_;
};

Y_UNIT_TEST_SUITE(GarbageCollectorJobSuite) {

Y_UNIT_TEST(CheckCorrectPlaces) {
    TSet<TString> goodPaths;
    TSet<TString> badPaths;
    for (const auto& it : GOOD_PLACES) {
        goodPaths.insert(it.second);
    }
    for (const auto& it : BAD_PLACES) {
        badPaths.insert(it.second);
    }

    for (const auto& it : GOOD_PLACES) {
        UNIT_ASSERT(!BAD_PLACES.contains(it.first));
        UNIT_ASSERT(!badPaths.contains(it.second));
        UNIT_ASSERT(ALL_PLACES.contains(it.first));
        UNIT_ASSERT_EQUAL_C(ALL_PLACES.at(it.first), it.second, ALL_PLACES.at(it.first));
    }
    for (const auto& it : BAD_PLACES) {
        UNIT_ASSERT(!GOOD_PLACES.contains(it.first));
        UNIT_ASSERT(!goodPaths.contains(it.second));
        UNIT_ASSERT(ALL_PLACES.contains(it.first));
        UNIT_ASSERT_EQUAL_C(ALL_PLACES.at(it.first), it.second, ALL_PLACES.at(it.first));
    }
}

Y_UNIT_TEST(TestNoRun) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            TGarbageCollectorJobPtr gcJob = GetGCJob();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestListContainersFails) {
    struct TTestPortoClientFails : public TTestPortoClient {
        TExpected<TVector<TPortoContainerName>, TPortoError> List(const TString& mask) override {
            ++ListCalls;
            ListLastMask = mask;
            return TPortoError{EPortoError::Unknown, "List", "", "NO"};
        }
    };

    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            ((TTestPortoClientFails*)Porto_.Get())->HaveLayers = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClientFails*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(0, ((TTestPortoClientFails*)Porto_.Get())->DestroyCalls);
        }

        TPortoClientPtr GetSpecificPorto() const final {
            return new TTestPortoClientFails();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestDestroyContainerFails) {
    struct TTestPortoClientFails : public TTestPortoClient {
        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            ++DestroyCalls;
            DestroyLastName = name;
            if (TString(name).find_first_of("layer2") != TString::npos) {
                return TPortoError{EPortoError::Unknown, "Destroy", name, "NO"};
            }
            return TExpected<void, TPortoError>();
        }
    };

    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));

            ((TTestPortoClientFails*)Porto_.Get())->HaveLayers = true;
            ((TTestPortoClientFails*)Porto_.Get())->HaveBoxes = true;
            ((TTestPortoClientFails*)Porto_.Get())->HaveWorkloads = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClientFails*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(5, ((TTestPortoClientFails*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL_C(
                TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload2_start")
                , ((TTestPortoClient*)Porto_.Get())->DestroyLastName
                , TString(((TTestPortoClient*)Porto_.Get())->DestroyLastName)
            );
        }

        TPortoClientPtr GetSpecificPorto() const final {
            return new TTestPortoClientFails();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearResourceContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            ((TTestPortoClient*)Porto_.Get())->HaveLayers = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "layer_layer2_download_hash_download"}), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearStaticResourceContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            ((TTestPortoClient*)Porto_.Get())->HaveStaticResources = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "static_static_resource2_download_hash_download"}), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearBoxContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));

            ((TTestPortoClient*)Porto_.Get())->HaveBoxes = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName(CONTAINERS_PREFIX + "box_box1"), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearWorkloadContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload2"));

            ((TTestPortoClient*)Porto_.Get())->HaveBoxes = true;
            ((TTestPortoClient*)Porto_.Get())->HaveWorkloads = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(2, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload1_stop"), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearOtherContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            ((TTestPortoClient*)Porto_.Get())->HaveOther = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(2, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({CONTAINERS_PREFIX + "some_other"}, "resource"), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearOtherBoxContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));

            ((TTestPortoClient*)Porto_.Get())->HaveBoxes = true;
            ((TTestPortoClient*)Porto_.Get())->HaveOther = true;
            RunGCJob();

            TPortoContainerName boxOther = TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "some_other");
            const auto listResult = Porto_->List().Success();
            UNIT_ASSERT_C(Find(listResult.begin(), listResult.end(), boxOther) != listResult.end(), "box other container not found");

            // DEPLOY-528
            // all unknown box containers are the responsibility of the user
            // so, in this test, the same containers must be destroyed as in TestClearOtherContainer
            UNIT_ASSERT_EQUAL(2, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(2, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({CONTAINERS_PREFIX + "some_other"}, "resource"), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestClearOtherWorkloadContainer) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload2"));

            ((TTestPortoClient*)Porto_.Get())->HaveBoxes = true;
            ((TTestPortoClient*)Porto_.Get())->HaveWorkloads = true;
            ((TTestPortoClient*)Porto_.Get())->HaveOther = true;
            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL(3, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({CONTAINERS_PREFIX + "box_box1"}, "workload_workload1_no_name"), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveVolumeDirectories) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            EnsureDoesNotExist(VOLUMES_STORAGE);
            NFs::MakeDirectory(VOLUMES_STORAGE);

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("my_box"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("my_volume"));
            NFs::MakeDirectory(PathHolder_->GetBoxRootfsPath("my_box"));
            NFs::MakeDirectory(PathHolder_->GetVolumePath("my_volume"));

            auto foldersBefore = Posix_->ListAsync(VOLUMES_STORAGE).ExtractValueSync().Success();
            UNIT_ASSERT(foldersBefore.size() == 2);

            StatusNTickerHolder_->RemoveVolume("my_volume");

            RunGCJob();

            auto res = Posix_->ListAsync(VOLUMES_STORAGE).ExtractValueSync();
            UNIT_ASSERT((bool)res);
            UNIT_ASSERT_VALUES_EQUAL(res.Success().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(RealPath(VOLUMES_STORAGE) + "/" + res.Success()[0], PathHolder_->GetBoxRootfsPath("my_box"));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveBoxDirectories) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            EnsureDoesNotExist(VOLUMES_STORAGE);
            NFs::MakeDirectory(VOLUMES_STORAGE);

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("my_box"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("my_volume"));
            NFs::MakeDirectory(PathHolder_->GetBoxRootfsPath("my_box"));
            NFs::MakeDirectory(PathHolder_->GetVolumePath("my_volume"));
            auto foldersBefore = Posix_->ListAsync(VOLUMES_STORAGE).ExtractValueSync().Success();
            UNIT_ASSERT(foldersBefore.size() == 2);

            StatusNTickerHolder_->RemoveBox("my_box");

            RunGCJob();

            auto res = Posix_->ListAsync(VOLUMES_STORAGE).ExtractValueSync();
            UNIT_ASSERT((bool)res);
            UNIT_ASSERT_VALUES_EQUAL(res.Success().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(RealPath(VOLUMES_STORAGE) + "/" + res.Success()[0], PathHolder_->GetVolumePath("my_volume"));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveStaticResourceDirectories) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            const TString id = "my_static_resource";
            const TString staticResourceDownloadHash = "MyStaticResourceHash";

            for (const auto& it : ALL_PLACES) {
                EnsureDoesNotExist(AllPlacesPathHolder_->GetStaticResourcesDirectory(it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetStaticResourcesDirectory(it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetStaticResourceDownloadDirectoryFromHash(staticResourceDownloadHash, it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetStaticResourceDownloadDirectoryFromHash(staticResourceDownloadHash + "_1", it.first));
            }

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple(id, staticResourceDownloadHash + "_1"));

            for (const auto& it : ALL_PLACES) {
                auto foldersBefore = Posix_->ListAsync(AllPlacesPathHolder_->GetStaticResourcesDirectory(it.first)).ExtractValueSync().Success();
                UNIT_ASSERT(foldersBefore.size() == 2);
            }

            RunGCJob();

            for (const auto& it : GOOD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(AllPlacesPathHolder_->GetStaticResourcesDirectory(it.first)).ExtractValueSync();
                UNIT_ASSERT((bool)foldersAfter);
                UNIT_ASSERT(foldersAfter.Success().size() == 1);
                UNIT_ASSERT(foldersAfter.Success()[0] == staticResourceDownloadHash + "_1");
            }

            for (const auto& it : BAD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(AllPlacesPathHolder_->GetStaticResourcesDirectory(it.first)).ExtractValueSync().Success();
                UNIT_ASSERT(foldersAfter.size() == 2);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeUnlinkVolumes) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            EnsureDoesNotExist(VOLUMES_STORAGE);
            NFs::MakeDirectory(VOLUMES_STORAGE);

            const TString id = "my_volume";
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(id));
            const TString volumePath = PathHolder_->GetVolumePath(id);
            Porto_->CreateVolume(volumePath, PathHolder_->GetVolumePersistentStorage(id), "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            for (i32 i = 0; i < 10; ++i) {
                const TString curId = id + "_" + ToString(i);
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(curId));
                const TString curPath = PathHolder_->GetVolumePath(curId);
                Porto_->CreateVolume(curPath, PathHolder_->GetVolumePersistentStorage(curId), "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            }

            StatusNTickerHolder_->RemoveVolume(id);

            auto listBefore = Porto_->ListVolumes().Success();
            TSet<TString> setBefore;
            ForEach(listBefore.begin(), listBefore.end(), [&setBefore](const auto& volume){setBefore.insert(volume.path());});

            RunGCJob();

            auto listAfter = Porto_->ListVolumes().Success();
            UNIT_ASSERT_EQUAL(listBefore.size(), listAfter.size() + 1);
            for (const auto& volume : listAfter) {
                UNIT_ASSERT(setBefore.contains(volume.path()));
                UNIT_ASSERT(volume.path() != volumePath);
            }
        }
    };

    TTest test;
    test.DoTest();
}


Y_UNIT_TEST(TestNotUnlinkCacheAndResourceVolumes) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            for (const auto& it : GOOD_PLACES) {
                Porto_->CreateVolume(it.second, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            }
            Porto_->CreateVolume(CACHE_VOLUME_PATH, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);

            const TString id = "my_volume";
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(id));
            const TString volumePath = PathHolder_->GetVolumePath(id);
            Porto_->CreateVolume(volumePath, PathHolder_->GetVolumePersistentStorage(id), "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            StatusNTickerHolder_->RemoveVolume(id);

            auto listBefore = Porto_->ListVolumes().Success();
            TSet<TString> setBefore;
            ForEach(listBefore.begin(), listBefore.end(), [&setBefore](const auto& volume){setBefore.insert(volume.path());});

            RunGCJob();
            auto listAfter = Porto_->ListVolumes().Success();

            TSet<TString> setAfter;
            ForEach(listAfter.begin(), listAfter.end(), [&setAfter](const auto& volume){setAfter.insert(volume.path());});

            for (const auto& it : GOOD_PLACES) {
                UNIT_ASSERT(setBefore.contains(it.second));
                UNIT_ASSERT(setAfter.contains(it.second));
            }
            UNIT_ASSERT(setBefore.contains(CACHE_VOLUME_PATH));
            UNIT_ASSERT(setAfter.contains(CACHE_VOLUME_PATH));
            UNIT_ASSERT(setBefore.contains(volumePath));
            UNIT_ASSERT(!setAfter.contains(volumePath));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxUnlinkVolumes) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            EnsureDoesNotExist(VOLUMES_STORAGE);
            NFs::MakeDirectory(VOLUMES_STORAGE);

            const TString id = "my_box";
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple(id));
            const TString boxPath = PathHolder_->GetBoxRootfsPath(id);
            Porto_->CreateVolume(boxPath, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            Porto_->CreateVolume(boxPath + "/link", "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            for (i32 i = 0; i < 10; ++i) {
                const TString curId = id + "_" + ToString(i);
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple(curId));
                const TString curPath = PathHolder_->GetBoxRootfsPath(curId);
                Porto_->CreateVolume(curPath, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
                Porto_->CreateVolume(curPath + "/link", "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            }

            StatusNTickerHolder_->RemoveBox(id);

            auto listBefore = Porto_->ListVolumes().Success();
            TSet<TString> setBefore;
            ForEach(listBefore.begin(), listBefore.end(), [&setBefore](const auto& volume){setBefore.insert(volume.path());});

            RunGCJob();

            auto listAfter = Porto_->ListVolumes().Success();
            UNIT_ASSERT_EQUAL(listBefore.size(), listAfter.size() + 2);
            for (const auto& volume : listAfter) {
                UNIT_ASSERT(setBefore.contains(volume.path()));
                UNIT_ASSERT(volume.path() != boxPath);
                UNIT_ASSERT(volume.path() != boxPath + "/link");
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLayersPortoAndDirs) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            const TString id = "my_layer";
            const TString layerDownloadHash = "MyLayerHash";

            for (const auto& it : ALL_PLACES) {
                EnsureDoesNotExist(AllPlacesPathHolder_->GetLayersDirectory(it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetLayersDirectory(it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash, it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash + "_1", it.first));

                Porto_->ImportLayer(PathHolder_->GetLayerNameFromHash(layerDownloadHash), AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash, it.first), false, it.first);
                Porto_->ImportLayer(PathHolder_->GetLayerNameFromHash(layerDownloadHash + "_1"), AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash + "_1", it.first), false, it.first);
            }

            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple(id + "_1", layerDownloadHash + "_1"));

            for (const auto& it : ALL_PLACES) {
                auto foldersBefore = Posix_->ListAsync(AllPlacesPathHolder_->GetLayersDirectory(it.first)).ExtractValueSync().Success();
                UNIT_ASSERT(foldersBefore.size() == 2);

                auto layersListBefore = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListBefore.size() == 2);
            }

            RunGCJob();

            for (const auto& it : GOOD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(AllPlacesPathHolder_->GetLayersDirectory(it.first)).ExtractValueSync();
                UNIT_ASSERT((bool)foldersAfter);
                UNIT_ASSERT(foldersAfter.Success().size() == 1);
                UNIT_ASSERT(foldersAfter.Success()[0] == layerDownloadHash + "_1");

                auto layersListAfter = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListAfter.size() == 1);
                UNIT_ASSERT(layersListAfter[0].name() == PathHolder_->GetLayerNameFromHash(layerDownloadHash + "_1"));
            }

            for (const auto& it : BAD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(AllPlacesPathHolder_->GetLayersDirectory(it.first)).ExtractValueSync().Success();
                UNIT_ASSERT(foldersAfter.size() == 2);

                auto layersListAfter = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListAfter.size() == 2);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLayerOnlyPorto) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            const TString id = "my_layer";
            const TString layerDownloadHash = "MyLayerHash";

            for (const auto& it : ALL_PLACES) {
                EnsureDoesNotExist(it.second);
                NFs::MakeDirectoryRecursive(it.second);

                Porto_->ImportLayer(PathHolder_->GetLayerNameFromHash(layerDownloadHash), AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash, it.first), false, it.first);
                Porto_->ImportLayer(PathHolder_->GetLayerNameFromHash(layerDownloadHash + "_1"), AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash + "_1", it.first), false, it.first);
            }

            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple(id + "_1", layerDownloadHash + "_1"));

            for (const auto& it : ALL_PLACES) {
                auto foldersBefore = Posix_->ListAsync(it.second).ExtractValueSync().Success();
                UNIT_ASSERT(foldersBefore.empty());

                auto layersListBefore = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListBefore.size() == 2);
            }

            RunGCJob();

            for (const auto& it : GOOD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(it.second).ExtractValueSync();
                UNIT_ASSERT((bool)foldersAfter);
                UNIT_ASSERT(foldersAfter.Success().empty());

                auto layersListAfter = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListAfter.size() == 1);
                UNIT_ASSERT(layersListAfter[0].name() == PathHolder_->GetLayerNameFromHash(layerDownloadHash + "_1"));
            }

            for (const auto& it : BAD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(it.second).ExtractValueSync();
                UNIT_ASSERT((bool)foldersAfter);
                UNIT_ASSERT(foldersAfter.Success().empty());

                auto layersListAfter = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListAfter.size() == 2);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLayerOnlyDirs) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            const TString id = "my_layer";
            const TString layerDownloadHash = "MyLayerHash";

            for (const auto& it : ALL_PLACES) {
                EnsureDoesNotExist(AllPlacesPathHolder_->GetLayersDirectory(it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetLayersDirectory(it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash, it.first));
                NFs::MakeDirectoryRecursive(AllPlacesPathHolder_->GetFinalLayerPathFromHash(layerDownloadHash + "_1", it.first));
            }

            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple(id + "_1", layerDownloadHash + "_1"));

            for (const auto& it : ALL_PLACES) {
                auto foldersBefore = Posix_->ListAsync(AllPlacesPathHolder_->GetLayersDirectory(it.first)).ExtractValueSync().Success();
                UNIT_ASSERT(foldersBefore.size() == 2);

                auto layersListBefore = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListBefore.empty());
            }

            RunGCJob();

            for (const auto& it : GOOD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(AllPlacesPathHolder_->GetLayersDirectory(it.first)).ExtractValueSync();
                UNIT_ASSERT((bool)foldersAfter);
                UNIT_ASSERT(foldersAfter.Success().size() == 1);
                UNIT_ASSERT(foldersAfter.Success()[0] == layerDownloadHash + "_1");

                auto layersListAfter = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListAfter.empty());
            }

            for (const auto& it : BAD_PLACES) {
                auto foldersAfter = Posix_->ListAsync(AllPlacesPathHolder_->GetLayersDirectory(it.first)).ExtractValueSync().Success();
                UNIT_ASSERT(foldersAfter.size() == 2);

                auto layersListAfter = Porto_->ListLayers(it.first).Success();
                UNIT_ASSERT(layersListAfter.empty());
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveStorage) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            const TString boxId = "my_box";
            const TString volumeId = "my_volume";

            for (const auto& it : ALL_PLACES) {
                Porto_->ImportStorage(PathHolder_->GetBoxRootfsPersistentStorage(boxId), "", it.first, "", "");
                Porto_->ImportStorage(PathHolder_->GetBoxRootfsPersistentStorage(boxId + "_1"), "", it.first, "", "");
                Porto_->ImportStorage(PathHolder_->GetVolumePersistentStorage(volumeId), "", it.first, "", "");
                Porto_->ImportStorage(PathHolder_->GetVolumePersistentStorage(volumeId + "_1"), "", it.first, "", "");
            }

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple(boxId + "_1"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(volumeId + "_1"));

            for (const auto& it : ALL_PLACES) {
                auto storageListBefore = Porto_->ListStorages(it.first, "").Success();
                UNIT_ASSERT(storageListBefore.size() == 4);
            }

            RunGCJob();

            for (const auto& it : GOOD_PLACES) {
                auto storageListAfter = Porto_->ListStorages(it.first, "").Success();
                UNIT_ASSERT(storageListAfter.size() == 2);
                UNIT_ASSERT((TSet<TString>{storageListAfter[0].name(), storageListAfter[1].name()}) == (TSet<TString>{
                    PathHolder_->GetBoxRootfsPersistentStorage(boxId + "_1"), PathHolder_->GetVolumePersistentStorage(volumeId + "_1")
                }));
            }
            for (const auto& it : BAD_PLACES) {
                auto storageListAfter = Porto_->ListStorages(it.first, "").Success();
                UNIT_ASSERT(storageListAfter.size() == 4);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestDontUnlinkAlienVolumes) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest(): ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            const TString AlienDir = CWD + "/" + "aliens_dir";

            for (i32 i = 0; i < 5; ++i) {
                const TString curId = "my_volume_" + ToString(i);
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(curId));
                const TString curPath = PathHolder_->GetVolumePath(curId);
                Porto_->CreateVolume(curPath, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            }

            for (i32 i = 0; i < 5; ++i) {
                const TString curId = "my_box_" + ToString(i);
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple(curId));
                const TString curPath = PathHolder_->GetBoxRootfsPath(curId);
                Porto_->CreateVolume(curPath, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
                Porto_->CreateVolume(curPath + "/link", "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            }

            for (i32 i = 0; i < 5; ++i) {
                const TString curPath = AlienDir + "/alien_volume_" + ToString(i);
                Porto_->CreateVolume(curPath, "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
                Porto_->CreateVolume(curPath + "/link", "", "", {}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false);
            }

            auto listBefore = Porto_->ListVolumes().Success();
            TSet<TString> setBefore;
            ForEach(listBefore.begin(), listBefore.end(), [&setBefore](const auto& volume){setBefore.insert(volume.path());});

            RunGCJob();

            auto listAfter = Porto_->ListVolumes().Success();
            UNIT_ASSERT_EQUAL(listBefore.size(), listAfter.size());
            for (const auto& volume : listAfter) {
                UNIT_ASSERT(setBefore.contains(volume.path()));
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestEmptyIpSubnet) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 112, 0, 0);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 128, 1, 10);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE + "_other", IP6_SUBNET112_BASE, 128, 0, 10);

            RunGCJob();

            UNIT_ASSERT_EQUAL(0, ((TTestIpClient*)IpClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(0, ((TTestIpClient*)IpClient_.Get())->RemoveCalls);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestListIpAddressFails) {
    struct TTestIpClientFails : public TTestIpClient {
        TExpected<TVector<TIpDescription>, TIpClientError> ListAddress(const TString& /* device */) override {
            ++ListCalls;
            return TIpClientError(EIpClientError::Unspecified, "list error");
        }
    };

    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->UpdateBoxIp6Subnet112Base(IP6_SUBNET112_BASE);

            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 112, 0, 0);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 128, 1, 10);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE + "_other", IP6_SUBNET112_BASE, 128, 0, 20);

            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestIpClient*)IpClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(0, ((TTestIpClient*)IpClient_.Get())->RemoveCalls);
        }

        TIpClientPtr GetSpecificIpClient() const final {
            return new TTestIpClientFails();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveIpAddressFails) {
    struct TTestIpClientFails : public TTestIpClient {
        TExpected<void, TIpClientError> RemoveAddress(const TString& device, const TIpDescription& ip) override {
            if (ip.Ip6 == IP6_SUBNET112_BASE + "1") {
                ++RemoveCalls;
                ++RemoveErrors;
                return TIpClientError(EIpClientError::Unspecified, "remove error");
            } else {
                return TTestIpClient::RemoveAddress(device, ip);
            }
        }

        size_t RemoveErrors = 0;
    };

    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->UpdateBoxIp6Subnet112Base(IP6_SUBNET112_BASE);

            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 112, 0, 0);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 128, 1, 10);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE + "_other", IP6_SUBNET112_BASE, 128, 0, 20);

            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestIpClient*)IpClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(10, ((TTestIpClient*)IpClient_.Get())->RemoveCalls);
            UNIT_ASSERT_EQUAL(1, ((TTestIpClientFails*)IpClient_.Get())->RemoveErrors);
        }

        TIpClientPtr GetSpecificIpClient() const final {
            return new TTestIpClientFails();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveIpAddress) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->UpdateBoxIp6Subnet112Base(IP6_SUBNET112_BASE);

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));

            BoxStatusRepository_->UpdateObjectIpAddress("box1", IP6_SUBNET112_BASE + "1");
            BoxStatusRepository_->UpdateObjectIpAddress("box2", IP6_SUBNET112_BASE + "4");

            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 112, 0, 0);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 128, 1, 10);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, IP6_SUBNET112_BASE, 127, 11, 11);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE, "other_" + IP6_SUBNET112_BASE, 128, 11, 11);
            ((TTestIpClient*)IpClient_.Get())->GenIpAddresses(IP_DEVICE + "_other", IP6_SUBNET112_BASE, 128, 0, 20);

            RunGCJob();

            TSet<TIpDescription, TIpDescriptionCmp> expectedListResult = {
                TIpDescription(IP6_SUBNET112_BASE + "0", 112)
                , TIpDescription(IP6_SUBNET112_BASE + "1", 128)
                , TIpDescription(IP6_SUBNET112_BASE + "4", 128)
                , TIpDescription(IP6_SUBNET112_BASE + "11", 127)
                , TIpDescription("other_" + IP6_SUBNET112_BASE + "11", 128)
            };

            UNIT_ASSERT_EQUAL(1, ((TTestIpClient*)IpClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(8, ((TTestIpClient*)IpClient_.Get())->RemoveCalls);

            auto listResult = IpClient_->ListAddress(IP_DEVICE).Success();
            UNIT_ASSERT_EQUAL(expectedListResult.size(), listResult.size());
            for (const auto& ip : listResult) {
                UNIT_ASSERT(expectedListResult.contains(ip));
            }

            UNIT_ASSERT_EQUAL(21, IpClient_->ListAddress(IP_DEVICE + "_other").Success().size());
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestListHttpRequestsFails) {
    struct TTestNetworkClientFails : public TTestNetworkClient {
        TExpected<TVector<TRequestPublicInfo>, TNetworkClientError> ListRequests() const override {
            ++ListCalls;
            return TNetworkClientError(ENetworkClientError::Unspecified, "list error");
        }
    };

    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload"));

            NetworkClient_->CheckAndAddHttpRequest("key", "hash", "workload", "", 0, "", TDuration::Zero());
            ((TTestNetworkClient*)NetworkClient_.Get())->GenHttpRequests("key", "hash", "bad_workload", 1, 10);

            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestNetworkClient*)NetworkClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(0, ((TTestNetworkClient*)NetworkClient_.Get())->RemoveCalls);
        }

        TNetworkClientPtr GetSpecificNetworkClient() const final {
            return new TTestNetworkClientFails();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveHttpRequestFails) {
    struct TTestNetworkClientFails : public TTestNetworkClient {
        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            if (requestKey.EndsWith("4")) {
                ++RemoveCalls;
                ++RemoveErrors;
                return TNetworkClientError(ENetworkClientError::Unspecified, "remove error");
            } else {
                return TTestNetworkClient::RemoveRequest(requestKey);
            }
        }

        size_t RemoveErrors = 0;
    };

    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload"));

            NetworkClient_->CheckAndAddHttpRequest("key", "hash", "workload", "", 0, "", TDuration::Zero());
            ((TTestNetworkClient*)NetworkClient_.Get())->GenHttpRequests("key", "hash", "bad_workload", 1, 10);

            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestNetworkClient*)NetworkClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(10, ((TTestNetworkClient*)NetworkClient_.Get())->RemoveCalls);
            UNIT_ASSERT_EQUAL(1, ((TTestNetworkClientFails*)NetworkClient_.Get())->RemoveErrors);
        }

        TNetworkClientPtr GetSpecificNetworkClient() const final {
            return new TTestNetworkClientFails();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveHttpRequest) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload14"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload19"));

            ((TTestNetworkClient*)NetworkClient_.Get())->GenHttpRequests("key", "hash", "workload", 1, 20);

            RunGCJob();

            UNIT_ASSERT_EQUAL(1, ((TTestNetworkClient*)NetworkClient_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL(17, ((TTestNetworkClient*)NetworkClient_.Get())->RemoveCalls);

            TSet<INetworkClient::TRequestPublicInfo, TNetworkClientRequestPublicInfoCmp> expectedListResult = {
                INetworkClient::TRequestPublicInfo("key1", "hash1", "workload1")
                , INetworkClient::TRequestPublicInfo("key14", "hash14", "workload14")
                , INetworkClient::TRequestPublicInfo("key19", "hash19", "workload19")
            };

            auto listResult = NetworkClient_->ListRequests().Success();
            UNIT_ASSERT_EQUAL(expectedListResult.size(), listResult.size());
            for (const auto& httpRequest : listResult) {
                UNIT_ASSERT(expectedListResult.contains(httpRequest));
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestUnactiveGCJob) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            ((TTestPortoClient*)Porto_.Get())->HaveLayers = true;

            RunGCJob(false);

            UNIT_ASSERT_EQUAL_C(((TTestPortoClient*)Porto_.Get())->ListCalls, 0, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_EQUAL_C(((TTestPortoClient*)Porto_.Get())->DestroyCalls, 0, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName(""), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestWithPeriodJobWorker) {
    class TTest : public ITestGarbageCollectorJobCanon {
    public:
        TTest() : ITestGarbageCollectorJobCanon()
        {}

    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            ((TTestPortoClient*)Porto_.Get())->HaveLayers = true;
            TGarbageCollectorJobPtr gcJob =  GetGCJob();
            gcJob->SetActive(true);

            StatusNTickerHolder_->StartTicker();
            TPeriodJobWorker periodJobWorker(TDuration::Seconds(5), logger.SpawnFrame());
            periodJobWorker.AddJob(gcJob);

            periodJobWorker.Start();
            Sleep(TDuration::MilliSeconds(350));
            periodJobWorker.Stop();

            StatusNTickerHolder_->StopTicker();

            UNIT_ASSERT_C(((TTestPortoClient*)Porto_.Get())->ListCalls > 0, ((TTestPortoClient*)Porto_.Get())->ListCalls);
            UNIT_ASSERT_EQUAL("", ((TTestPortoClient*)Porto_.Get())->ListLastMask);
            UNIT_ASSERT_C(((TTestPortoClient*)Porto_.Get())->DestroyCalls > 0, ((TTestPortoClient*)Porto_.Get())->DestroyCalls);
            UNIT_ASSERT_EQUAL(TPortoContainerName({RESOURCE_GANG_META_CONTAINER, "layer_layer2_download_hash_download"}), ((TTestPortoClient*)Porto_.Get())->DestroyLastName);
        }
    };

    TTest test;
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NGarbageCollectorJobTest
