#include "file_stream_rotated_impl.h"

#include <util/datetime/base.h>
#include <util/string/builder.h>

namespace NInfra::NPodAgent {

TFileStreamRotatedImpl::TFileStreamRotatedImpl(
    const TString& filePath
    , double minPortionToCut
    , ui32 maxLogsFileSize
    , blksize_t fileSystemBlockSize
    , TFileSystemUtilsPtr fileSystemUtils
)
    : MarkedInvalid_(false)
    , FilePath_(filePath)
    , File_(FilePath_, OpenAlways)
    , Stream_(File_)
    , MinPortionToCut_(minPortionToCut)
    , MaxLogsFileSize_(maxLogsFileSize)
    , FileSystemBlockSize_(fileSystemBlockSize)
    , LastCachedModificationTime_(TInstant::Now().Seconds())
    , FileSystemUtils_(fileSystemUtils)
    , InitialInodeGetResult_(FileSystemUtils_->GetFileInode(FilePath_))
{ }

TExpected<TString, TPushClientError> TFileStreamRotatedImpl::Read(ui64 offset, ui64 size) {
    if (!FileSystemUtils_->Exists(FilePath_)) {
        return TPushClientError{EPushClientError::FileNotExists, TStringBuilder() << "Read failed: file not exists. Path:" << FilePath_};
    }
    try {
        File_.Seek(offset, SeekDir::sSet);
        TString result = TString(size, ' ');
        ui64 loadedLength = Stream_.Load(const_cast<char*>(result.data()), size);
        result.erase(loadedLength);
        return result;
    } catch (...) {
        return TPushClientError{EPushClientError::FileReadingError, CurrentExceptionMessage()};
    }
}

TExpected<ui64, TPushClientError> TFileStreamRotatedImpl::Rotate() {
    if (!FileSystemUtils_->Exists(FilePath_)) {
        return TPushClientError{EPushClientError::FileNotExists, TStringBuilder() << "Rotate failed: file not exists. Path:" << FilePath_};
    }

    auto fileSize = OUTCOME_TRYX(GetFileSize());
    if (fileSize < MaxLogsFileSize_) {
        return TPushClientError{EPushClientError::FileRotateError, TStringBuilder() << "Rotate failed: fileSize < maxFileSize. Path: " << FilePath_};
    }
    i64 lengthToCut = fileSize - (i64)(MaxLogsFileSize_ * (1 - MinPortionToCut_));
    if (lengthToCut % FileSystemBlockSize_ != 0) {
        lengthToCut -= lengthToCut % FileSystemBlockSize_;
        lengthToCut += FileSystemBlockSize_;
    }

    if (lengthToCut == 0) {
        return 0;
    }

    OUTCOME_TRYV(FileSystemUtils_->CutHeadOfFile(File_, lengthToCut));

    return lengthToCut;
}

TExpected<ui64, TPushClientError> TFileStreamRotatedImpl::GetFileSize() const {
    auto size = File_.GetLength();
    if (size == -1) {
        return TPushClientError{EPushClientError::FileStatError, TStringBuilder() << "Get length returns error. Path: " << FilePath_};
    }
    return size;
}

TExpected<ui32, TPushClientError> TFileStreamRotatedImpl::TryClearFile() {
    if (!FileSystemUtils_->Exists(FilePath_)) {
        return TPushClientError{EPushClientError::FileNotExists, TStringBuilder() << "TryClear failed: file not exists. Path:" << FilePath_};
    }

    auto fileSize = OUTCOME_TRYX(GetFileSize());
    fileSize = fileSize - fileSize % FileSystemBlockSize_;
    if (!fileSize) {
        return 0;
    }
    OUTCOME_TRYV(FileSystemUtils_->CutHeadOfFile(File_, fileSize));
    return fileSize;
}

TString TFileStreamRotatedImpl::GetFilePath() const {
    return FilePath_;
}

TMaybe<TPushClientError> TFileStreamRotatedImpl::HasBecomeInvalid() {
    if (!MarkedInvalid_) {
        auto isInvalid = IsInvalid();
        if (isInvalid.Defined()) {
            MarkedInvalid_ = true;
        }
        return isInvalid;
    }

    return Nothing();
}

TMaybe<TPushClientError> TFileStreamRotatedImpl::IsInvalid() {
    if (!FileSystemUtils_->Exists(FilePath_)) {
        return TPushClientError{EPushClientError::FileNotExists, TStringBuilder() << "IsInvalid: file not exists. Path:" << FilePath_};
    }

    ino_t initInode = OUTCOME_TRYX(InitialInodeGetResult_);
    ino_t currentInode = OUTCOME_TRYX(FileSystemUtils_->GetFileInode(FilePath_));

    if (initInode != currentInode) {
        return TPushClientError{EPushClientError::FileInodeChanged, TStringBuilder() << "IsInvalid: file inode changed, init inode: " << initInode << " current inode: " << currentInode << " Path: " << FilePath_};
    }
    return Nothing();
}


}
