#include "file_stream_rotated_impl.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/file_system_utils_mock.h>
#include <library/cpp/testing/unittest/registar.h>
#include <sys/stat.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NFileStreamRotatedTest {

static constexpr const double defaultMinPortionToCut = 0.2;
static const ui32 defaultMaxLogsFileSize = 1000;
static const blksize_t defaultFileSystemBlockSize = 100;
static const TString defaultFileName = "filename";

struct TTestFileSystemUtils: public TMockFileSystemUtils {
    bool Exists(const TString&) const override {
        ++IsFileExistsCalls;
        return true;
    }

    TExpected<void, TPushClientError> CutHeadOfFile(const TFile&, off64_t) const override {
        ++CutHeadOfFileCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ino_t, TPushClientError> GetFileInode(const TString&) const override {
        ++GetFileInodeCalls;
        return TExpected<ino_t, TPushClientError>::DefaultSuccess();
    }

    mutable size_t IsFileExistsCalls = 0;
    mutable size_t CutHeadOfFileCalls = 0;
    mutable size_t GetFileInodeCalls = 0;
};

Y_UNIT_TEST_SUITE(FileStreamRotatedSuite) {

    Y_UNIT_TEST(SimpleTest) {
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, new TTestFileSystemUtils);
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(ReadTest) {
        TUnbufferedFileOutput output(defaultFileName);
        TString inputLine = "Hello World";
        output << inputLine;
        output.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtils;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        TString result = fileStream->Read(0, inputLine.Size()).Success();
        UNIT_ASSERT_EQUAL(result, inputLine);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(ReadFailWhenFileNotExistTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            bool Exists(const TString&) const override {
                ++IsFileExistsCalls;
                return false;
            }
        };

        TUnbufferedFileOutput output(defaultFileName);
        TString inputLine = "Hello World";
        output << inputLine;
        output.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtilsFails;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto readResult = fileStream->Read(0, inputLine.Size());
        UNIT_ASSERT(!readResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileNotExists, readResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("Read failed: file not exists. Path:" + defaultFileName, readResult.Error().Message);

        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(ReadWithOffsetTest) {
        TUnbufferedFileOutput output(defaultFileName);
        TString inputLineHello = "Hello";
        TString inputLineWorld = "World";
        output << inputLineHello << inputLineWorld;
        output.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtils;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        TString result = fileStream->Read(inputLineHello.Size(), inputLineWorld.Size()).Success();
        UNIT_ASSERT_EQUAL(result, inputLineWorld);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(RealRotateTest) {
        TUnbufferedFileOutput output(defaultFileName);
        TString inputLine = TString(20000, 'a');
        output << inputLine;
        output.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TFileSystemUtils;
        blksize_t realFileSystemBlockSize = fileStreamUtils->GetFileSystemBlockSize().Success();
        ui32 realMaxFileSize = 10000;

        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, realMaxFileSize, realFileSystemBlockSize, fileStreamUtils);
        auto rotateResult = fileStream->Rotate();
        UNIT_ASSERT(rotateResult);

        ui64 ans = inputLine.Size() - realMaxFileSize * (1- defaultMinPortionToCut);
        ans -= ans % realFileSystemBlockSize;
        ans += realFileSystemBlockSize;

        UNIT_ASSERT_EQUAL(ans, rotateResult.Success());

        TFile file1(defaultFileName, OpenAlways);
        UNIT_ASSERT_EQUAL((size_t)(inputLine.Size() - ans), (size_t)file1.GetLength());
        file1.Close();
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(RotateFailWhenFileNotExistTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            bool Exists(const TString&) const override {
                ++IsFileExistsCalls;
                return false;
            }
        };

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtilsFails;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto rotateResult = fileStream->Rotate();
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileNotExists, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("Rotate failed: file not exists. Path:" + defaultFileName, rotateResult.Error().Message);

        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileSystemUtils*)fileStreamUtils.Get())->CutHeadOfFileCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(RotateFailWhenCutHeadReturnErrorTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            TExpected<void, TPushClientError> CutHeadOfFile(const TFile&, off64_t) const override {
                ++CutHeadOfFileCalls;
                return TPushClientError{EPushClientError::FileRotateError, "fallocate error"};
            }
        };
        TUnbufferedFileOutput output(defaultFileName);
        TString inputLine = TString(20000, 'a');
        output << inputLine;
        output.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtilsFails;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto rotateResult = fileStream->Rotate();
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileRotateError, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("fallocate error", rotateResult.Error().Message);

        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->CutHeadOfFileCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(RotateFailWhenFileSizeLessThanMaxFileSizeTest) {
        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtils;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto rotateResult = fileStream->Rotate();
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileRotateError, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS(TStringBuilder() << "Rotate failed: fileSize < maxFileSize. Path: " << defaultFileName, rotateResult.Error().Message);

        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileSystemUtils*)fileStreamUtils.Get())->CutHeadOfFileCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(TryClearFailWhenFileNotExistsTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            bool Exists(const TString&) const override {
                ++IsFileExistsCalls;
                return false;
            }
        };

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtilsFails;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto tryClearResult = fileStream->TryClearFile();
        UNIT_ASSERT(!tryClearResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileNotExists, tryClearResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("TryClear failed: file not exists. Path:" + defaultFileName, tryClearResult.Error().Message);

        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileSystemUtils*)fileStreamUtils.Get())->CutHeadOfFileCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(TryClearFailWhenCutHeadReturnsErrorTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            TExpected<void, TPushClientError> CutHeadOfFile(const TFile&, off64_t) const override {
                ++CutHeadOfFileCalls;
                return TPushClientError{EPushClientError::FileRotateError, "fallocate error"};
            }
        };
        TUnbufferedFileOutput file(defaultFileName);
        TString inputLine = TString(20000, 'a');
        file << inputLine;
        file.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtilsFails;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto tryClearResult = fileStream->TryClearFile();

        UNIT_ASSERT(!tryClearResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileRotateError, tryClearResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("fallocate error", tryClearResult.Error().Message);

        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->CutHeadOfFileCalls);

        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(TryClearOkTest) {
        TUnbufferedFileOutput file(defaultFileName);
        TString inputLine = TString(20000, 'a');
        file << inputLine;
        file.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtils;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto tryClearResult = fileStream->TryClearFile();
        UNIT_ASSERT(tryClearResult);
        UNIT_ASSERT(tryClearResult.Success());
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileStreamUtils.Get())->CutHeadOfFileCalls);
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(GetFileSizeOkTest) {
        size_t fileSize = 20000;
        TUnbufferedFileOutput file(defaultFileName);
        TString inputLine = TString(fileSize, 'a');
        file << inputLine;
        file.Finish();

        TFileSystemUtilsPtr fileStreamUtils = new TTestFileSystemUtils;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileStreamUtils);

        auto getFileSizeResult = fileStream->GetFileSize();
        UNIT_ASSERT(getFileSizeResult);
        UNIT_ASSERT_EQUAL(fileSize, getFileSizeResult.Success());
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(HasBecomeInvalidWhenFileIsOkTest) {
        TFileSystemUtilsPtr fileSystemUtils = new TTestFileSystemUtils;
        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileSystemUtils);

        auto hasBecomeInvalidResult = fileStream->HasBecomeInvalid();
        UNIT_ASSERT(!hasBecomeInvalidResult.Defined());
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileSystemUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(2, ((TTestFileSystemUtils*)fileSystemUtils.Get())->GetFileInodeCalls);
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(HasBecomeInvalidWhenFileNotExistsTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            bool Exists(const TString&) const override {
                ++IsFileExistsCalls;
                return false;
            }
        };

        TFileSystemUtilsPtr fileSystemUtils = new TTestFileSystemUtilsFails;

        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileSystemUtils);

        auto hasBecomeInvalidResult = fileStream->HasBecomeInvalid();
        UNIT_ASSERT(hasBecomeInvalidResult.Defined());
        UNIT_ASSERT_EQUAL(EPushClientError::FileNotExists, hasBecomeInvalidResult.Get()->Errno);
        UNIT_ASSERT_STRING_CONTAINS("IsInvalid: file not exists. Path:" + defaultFileName, hasBecomeInvalidResult.Get()->Message);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileSystemUtils.Get())->IsFileExistsCalls);
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(HasBecomeInvalidWhenGetInodeFailsTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            TExpected<ino_t, TPushClientError> GetFileInode(const TString&) const override {
                ++GetFileInodeCalls;

                return TPushClientError{EPushClientError::FileStatError, "filestat error"};
            }
        };

        TFileSystemUtilsPtr fileSystemUtils = new TTestFileSystemUtilsFails;

        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileSystemUtils);

        auto hasBecomeInvalidResult = fileStream->HasBecomeInvalid();
        UNIT_ASSERT(hasBecomeInvalidResult.Defined());
        UNIT_ASSERT_EQUAL(EPushClientError::FileStatError, hasBecomeInvalidResult.Get()->Errno);
        UNIT_ASSERT_STRING_CONTAINS("filestat error" + defaultFileName, hasBecomeInvalidResult.Get()->Message);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileSystemUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileSystemUtils.Get())->GetFileInodeCalls);
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }

    Y_UNIT_TEST(HasBecomeInvalidWhenFileInodeChangesTest) {
        struct TTestFileSystemUtilsInodeChanges: public TTestFileSystemUtils {
            TExpected<ino_t, TPushClientError> GetFileInode(const TString&) const override {
                ++GetFileInodeCalls;

                return Inode;
            }

            ino_t Inode = 0;
        };

        ino_t firstInode = 100;
        ino_t secondInode = 200;

        TFileSystemUtilsPtr fileSystemUtils = new TTestFileSystemUtilsInodeChanges;
        ((TTestFileSystemUtilsInodeChanges*)fileSystemUtils.Get())->Inode = firstInode;

        TFileStreamRotatedPtr fileStream = new TFileStreamRotatedImpl(defaultFileName, defaultMinPortionToCut, defaultMaxLogsFileSize, defaultFileSystemBlockSize, fileSystemUtils);
        ((TTestFileSystemUtilsInodeChanges*)fileSystemUtils.Get())->Inode = secondInode;

        auto hasBecomeInvalidResult = fileStream->HasBecomeInvalid();
        UNIT_ASSERT(hasBecomeInvalidResult.Defined());
        UNIT_ASSERT_EQUAL(EPushClientError::FileInodeChanged, hasBecomeInvalidResult.Get()->Errno);
        UNIT_ASSERT_STRING_CONTAINS(TStringBuilder() << "IsInvalid: file inode changed, init inode: " << firstInode << " current inode: " << secondInode << " Path: " << defaultFileName, hasBecomeInvalidResult.Get()->Message);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileSystemUtils.Get())->IsFileExistsCalls);
        UNIT_ASSERT_EQUAL(2, ((TTestFileSystemUtils*)fileSystemUtils.Get())->GetFileInodeCalls);
        fileStream.Reset();
        NFs::Remove(defaultFileName);
    }
}

}
