#include "processed_containers_filter.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>

namespace NInfra::NPodAgent {

TProcessedContainersFilter::TProcessedContainersFilter(
    THoldersContextPtr stdoutHoldersContext
    , THoldersContextPtr stderrHoldersContext
    , TLogFramePtr logFrame
)
    : StdoutHoldersContext_(stdoutHoldersContext)
    , StderrHoldersContext_(stderrHoldersContext)
    , LogFrame_(logFrame)
{}

THashSet<TPushContainer> TProcessedContainersFilter::Filter(THashSet<TPushContainer>&& pushContainers) {
    for (auto it = pushContainers.begin(); it != pushContainers.end();) {
        bool isStdoutOldProcessed = true, isStderrOldProcessed = true;

        if (auto stdoutCheckResult = IsProcessedContainer(*it, StdoutHoldersContext_)) {
            isStdoutOldProcessed = stdoutCheckResult.Success();
        } else {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(stdoutCheckResult.Error()));
        }

        if (auto stderrCheckResult = IsProcessedContainer(*it, StderrHoldersContext_)) {
            isStderrOldProcessed = stderrCheckResult.Success();
        } else {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(stderrCheckResult.Error()));
        }

        auto itToErase = it++;
        if (isStdoutOldProcessed && isStderrOldProcessed) {
            pushContainers.erase(itToErase);
        }
    }

    return pushContainers;
}

TExpected<i32, TPushClientError> TProcessedContainersFilter::IsProcessedContainer(const TPushContainer& pushContainer, THoldersContextPtr holdersContext) {
    auto fileStream = OUTCOME_TRYX(holdersContext->GetFileStreamHolder()->GetFileStream(pushContainer));
    auto offset = OUTCOME_TRYX(holdersContext->GetLogsFileOffsetHolder()->GetOffset(pushContainer));
    auto fileSize = OUTCOME_TRYX(fileStream->GetFileSize());

    if (offset != fileSize) {
        return false;
    }

    auto clearedSize = OUTCOME_TRYX(fileStream->TryClearFile());
    if (clearedSize) {
        OUTCOME_TRYX(holdersContext->GetLogsFileOffsetHolder()->DecrementOffset(pushContainer, clearedSize));
    }

    return true;
}

} //namespace NInfra::NPodAgent

