#include "processed_containers_filter.h"

#include <infra/libs/logger/logger.h>

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/file_stream/file_stream_rotated_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/file_stream_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/holders_updater.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/holders_context.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/logs_file_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/serializable_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NProcessedContainersFilterTest {

static TLogger logger({});

static THashSet<TPushContainer> pushContainers = {
    TPushContainer({TPortoContainerName({"box_1"}, "workload_1"), {"1"}, {"1"}}),
    TPushContainer({TPortoContainerName({"box_2"}, "workload_2"), {"2"}, {"2"}}),
};
static const ui32 defaultClearingSize = 1;
static const ui32 defaultFileSize = 3;
static const ui32 defaultOffset = 3;

struct TTestLogsFileOffsetHolder : public TMockLogsFileOffsetHolder {
    TExpected<void, TPushClientError> DecrementOffset(const TPushContainer&, ui64) override {
        ++DecrementOffsetCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
        ++GetOffsetCalls;
        return defaultOffset;
    }

    size_t DecrementOffsetCalls = 0;
    mutable size_t GetOffsetCalls = 0;
};

struct TTestFileStreamHolder: public TMockFileStreamHolder {
    TTestFileStreamHolder(TFileStreamRotatedPtr fileStream)
        : FileStream(fileStream)
    {}

    TExpected<TFileStreamRotatedPtr, TPushClientError> GetFileStream(const TPushContainer&) const override {
        ++GetFileStreamCalls;
        return FileStream;
    }

    mutable size_t GetFileStreamCalls = 0;
    TFileStreamRotatedPtr FileStream;
};

struct TTestFileStreamRotated: public TMockFileStreamRotated {
    TExpected<ui32, TPushClientError> TryClearFile() override {
        ++TryClearCalls;
        return defaultClearingSize;
    }

    TExpected<ui64, TPushClientError> GetFileSize() const override {
        ++GetFileSizeCalls;
        return defaultFileSize;
    }

    size_t TryClearCalls = 0;
    mutable size_t GetFileSizeCalls = 0;
};

Y_UNIT_TEST_SUITE(ProcessedContainersFilterSuite) {

    Y_UNIT_TEST(FilterWhenGetStreamReturnsError) {
        struct TTestFileStreamHolderFails: public TTestFileStreamHolder {
            TTestFileStreamHolderFails(TFileStreamRotatedPtr fileStream)
                : TTestFileStreamHolder(fileStream)
            {}

            TExpected<TFileStreamRotatedPtr, TPushClientError> GetFileStream(const TPushContainer&) const override {
                ++GetFileStreamCalls;
                return TPushClientError{EPushClientError::NoLogsFileStreamForWorkload, "GetFileStream fails"};
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TFileStreamHolderPtr fileStreamHolder = new TTestFileStreamHolderFails(fileStream);
        TLogsFileOffsetHolderPtr offsetHolder = new TTestLogsFileOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(fileStreamHolder, offsetHolder, new TMockSerializableOffsetHolder);
        TFilterPtr processedContainersFilter = new TProcessedContainersFilter(holdersContext, holdersContext, logger.SpawnFrame());
        auto tmpContainers = pushContainers;

        processedContainersFilter->Filter(std::move(tmpContainers));

        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->DecrementOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->GetOffsetCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamHolder*)fileStreamHolder.Get())->GetFileStreamCalls);

        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->GetFileSizeCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->TryClearCalls);
    }

    Y_UNIT_TEST(FilterWhenGetOffsetReturnsError) {
        struct TTestOffsetHolderFails: public TTestLogsFileOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{EPushClientError::NoOffsetForWorkload, "GetOffset error"};
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TFileStreamHolderPtr fileStreamHolder = new TTestFileStreamHolder(fileStream);
        TLogsFileOffsetHolderPtr offsetHolder = new TTestOffsetHolderFails;
        THoldersContextPtr holdersContext = new THoldersContext(fileStreamHolder, offsetHolder, new TMockSerializableOffsetHolder);
        TFilterPtr processedContainersFilter = new TProcessedContainersFilter(holdersContext, holdersContext, logger.SpawnFrame());
        auto tmpContainers = pushContainers;

        processedContainersFilter->Filter(std::move(tmpContainers));

        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->DecrementOffsetCalls);
        UNIT_ASSERT_EQUAL(4, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->GetOffsetCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamHolder*)fileStreamHolder.Get())->GetFileStreamCalls);

        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->GetFileSizeCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->TryClearCalls);
    }

    Y_UNIT_TEST(FilterWhenGetFileSizeReturnsError) {
        struct TTestFileStreamRotatedFails: public TTestFileStreamRotated {
            TExpected<ui64, TPushClientError> GetFileSize() const override {
                ++GetFileSizeCalls;
                return TPushClientError{EPushClientError::FileStatError, "getFileSizeError"};
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotatedFails;
        TFileStreamHolderPtr fileStreamHolder = new TTestFileStreamHolder(fileStream);
        TLogsFileOffsetHolderPtr offsetHolder = new TTestLogsFileOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(fileStreamHolder, offsetHolder, new TMockSerializableOffsetHolder);
        TFilterPtr processedContainersFilter = new TProcessedContainersFilter(holdersContext, holdersContext, logger.SpawnFrame());
        auto tmpContainers = pushContainers;

        processedContainersFilter->Filter(std::move(tmpContainers));

        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->DecrementOffsetCalls);
        UNIT_ASSERT_EQUAL(4, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->GetOffsetCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamHolder*)fileStreamHolder.Get())->GetFileStreamCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamRotated*)fileStream.Get())->GetFileSizeCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->TryClearCalls);
    }

    Y_UNIT_TEST(FilterWhenTryClearReturnsError) {
        struct TTestFileStreamRotatedFails: public TTestFileStreamRotated {
            TExpected<ui32, TPushClientError> TryClearFile() override {
                ++TryClearCalls;
                return TPushClientError{EPushClientError::FileNotExists, "TryClear error"};
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotatedFails;
        TFileStreamHolderPtr fileStreamHolder = new TTestFileStreamHolder(fileStream);
        TLogsFileOffsetHolderPtr offsetHolder = new TTestLogsFileOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(fileStreamHolder, offsetHolder, new TMockSerializableOffsetHolder);
        TFilterPtr processedContainersFilter = new TProcessedContainersFilter(holdersContext, holdersContext, logger.SpawnFrame());
        auto tmpContainers = pushContainers;

        processedContainersFilter->Filter(std::move(tmpContainers));

        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->DecrementOffsetCalls);
        UNIT_ASSERT_EQUAL(4, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->GetOffsetCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamHolder*)fileStreamHolder.Get())->GetFileStreamCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamRotated*)fileStream.Get())->GetFileSizeCalls);
        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamRotated*)fileStream.Get())->TryClearCalls);
    }

    Y_UNIT_TEST(FilterWhenDecrementOffsetReturnsError) {
        struct TTestOffsetHolderFails: public TTestLogsFileOffsetHolder {
            TExpected<void, TPushClientError> DecrementOffset(const TPushContainer&, ui64) override {
                ++DecrementOffsetCalls;
                return TExpected<void, TPushClientError>::DefaultSuccess();
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TFileStreamHolderPtr fileStreamHolder = new TTestFileStreamHolder(fileStream);
        TLogsFileOffsetHolderPtr offsetHolder = new TTestOffsetHolderFails;
        THoldersContextPtr holdersContext = new THoldersContext(fileStreamHolder, offsetHolder, new TMockSerializableOffsetHolder);
        TFilterPtr processedContainersFilter = new TProcessedContainersFilter(holdersContext, holdersContext, logger.SpawnFrame());
        auto tmpContainers = pushContainers;

        processedContainersFilter->Filter(std::move(tmpContainers));

        UNIT_ASSERT_EQUAL(4, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->DecrementOffsetCalls);
        UNIT_ASSERT_EQUAL(4, ((TTestLogsFileOffsetHolder*)offsetHolder.Get())->GetOffsetCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamHolder*)fileStreamHolder.Get())->GetFileStreamCalls);

        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamRotated*)fileStream.Get())->GetFileSizeCalls);
        UNIT_ASSERT_EQUAL(4, ((TTestFileStreamRotated*)fileStream.Get())->TryClearCalls);
    }

}

} //namespace NInfra::NPodAgent::NProcessedContainersFilterTest
