#include "file_stream_holder_impl.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/file_stream/file_stream_rotated_impl.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/rotation_params_validator.h>

#include <util/generic/hash_set.h>
#include <sys/stat.h>

namespace NInfra::NPodAgent {

TFileStreamHolderImpl::TFileStreamHolderImpl(
    ui32 maxLogsFileSize
    , double minPortionToCut
    , blksize_t fileSystemBlockSize
    , const ELogType& typeOfLog
    , TPathHolderPtr pathHolder
    , TFileStreamRotatedFactoryPtr fileStreamFactory
    , const bool isBoxAgentMode
)
    : MaxLogsFileSize_(maxLogsFileSize)
    , MinPortionToCut_(minPortionToCut)
    , FileSystemBlockSize_(fileSystemBlockSize)
    , TypeOfLog_(typeOfLog)
    , PathHolder_(pathHolder)
    , FileStreamFactory_(fileStreamFactory)
    , IsBoxAgentMode_(isBoxAgentMode)
{}

void TFileStreamHolderImpl::Update(const THashSet<TPushContainer>& pushContainers) {
    THashMap<TPushContainer, TFileStreamRotatedPtr> pushContainerToFileStreamUpdated;

    for (const auto& pushContainer: pushContainers) {

        if (PushContainerToFileStream_.contains(pushContainer)) {
            pushContainerToFileStreamUpdated[pushContainer] = PushContainerToFileStream_[pushContainer];
        } else {
            TString path;
            if (IsBoxAgentMode_) {
                path = PathHolder_->GetWorkloadLogsFilePathInBox(pushContainer.WorkloadId, ToString(TypeOfLog_));
            } else {
                path = PathHolder_->GetWorkloadLogsFilePathAtLogsVolumePath(pushContainer.WorkloadId, ToString(TypeOfLog_));
            }
            pushContainerToFileStreamUpdated[pushContainer] = FileStreamFactory_->Create(path);
        }
    }

    PushContainerToFileStream_ = pushContainerToFileStreamUpdated;
}

TExpected<ui64, TPushClientError> TFileStreamHolderImpl::TryRotate(const TPushContainer& pushContainer) {
    if (!PushContainerToFileStream_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoLogsFileStreamForWorkload, TStringBuilder() << "No logs file stream for workload: " << pushContainer.Container};
    }

    if (!OUTCOME_TRYX(NRotationParamsValidator::AreRotationParamsValid(MaxLogsFileSize_, MinPortionToCut_, FileSystemBlockSize_))) {
        return TPushClientError{EPushClientError::FileRotateError, TStringBuilder() << "Rotate params are invalid: maxLogsFileSize = " << MaxLogsFileSize_ << ", MinPortionToCut = " << MinPortionToCut_ << ", FileSystem block size = " << FileSystemBlockSize_};
    }

    if (OUTCOME_TRYX(PushContainerToFileStream_[pushContainer]->GetFileSize()) < MaxLogsFileSize_) {
        return 0;
    }

    return PushContainerToFileStream_[pushContainer]->Rotate();
}

TExpected<TFileStreamRotatedPtr, TPushClientError> TFileStreamHolderImpl::GetFileStream(const TPushContainer& pushContainer) const {
    if (!PushContainerToFileStream_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoLogsFileStreamForWorkload, TStringBuilder() << "No logs file stream for workload: " << pushContainer.Container};
    }

    return PushContainerToFileStream_.at(pushContainer);
}

TVector<TPushClientError> TFileStreamHolderImpl::FilesBecameInvalid() {
    TVector<TPushClientError> result;

    for (auto& containerToStream: PushContainerToFileStream_) {
        auto hasBecomeInvalidResult = containerToStream.second->HasBecomeInvalid();

        if (hasBecomeInvalidResult.Defined())
            result.push_back(*hasBecomeInvalidResult.Get());
    }

    return result;
}

}
