#include "file_stream_holder_impl.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/file_stream/file_stream_rotated_mock.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NFileStreamHolderTest {

static THashSet<TPushContainer> pushContainers = {
    TPushContainer{{"box_1/workload_1"}, {"1"}, {"1"}}
    , TPushContainer{{"box_2/workload_2"}, {"2"}, {"2"}}
    , TPushContainer{{"box_3/workload_3"}, {"3"}, {"3"}}
};

static const TPushContainer defaultContainer{{"box_1/workload_1"}, {"1"}, {"1"}};
static const ui32 defaultMaxLogsFileSize = 1000;
static constexpr const double defaultMinPortionToCut = 0.2;
static const blksize_t defaultFileSystemBlockSize = 100;
static const size_t defaultFileSize = 15000;
static const size_t defaultRotateSize = 10;
static const bool isBoxAgentMode = false;

static TPathHolderPtr pathHolder = new TPathHolder(
    ""
    , {{"", ""}}
    , {{"", ""}}
    , ""
    , ""
    , ""
    , ""
    , ""//NFs::CurrentWorkingDirectory() + "/porto_log_files"
    , ""
);

struct TTestFileStreamRotated: public TMockFileStreamRotated {
    TTestFileStreamRotated(size_t fileSize, size_t rotateSize)
        : FileSize(fileSize)
        , RotateSize(rotateSize)
    {}

    TExpected<ui64, TPushClientError> Rotate() override {
        ++RotateCalls;
        return RotateSize;
    }

    TExpected<ui64, TPushClientError> GetFileSize() const override {
        ++GetFileSizeCalls;
        return FileSize;
    }

    TMaybe<TPushClientError> HasBecomeInvalid() override {
        ++HasBecomeInvalidCalls;
        return Nothing();
    }

    size_t RotateCalls = 0;
    mutable size_t GetFileSizeCalls = 0;
    mutable size_t HasBecomeInvalidCalls = 0;

private:
    size_t FileSize;
    size_t RotateSize;
};

struct TTestFileStreamRotatedFactory: public TFileStreamRotatedFactory {
    TTestFileStreamRotatedFactory(TFileStreamRotatedPtr fileStream)
        : FileStream(fileStream)
    {}

    TFileStreamRotatedPtr Create(const TString& path) override {
        FileStreamPath = path;
        return FileStream;
    }

private:
    TFileStreamRotatedPtr FileStream;

public:
    TString FileStreamPath = "";
};

Y_UNIT_TEST_SUITE(FileStreamHolderSuite) {

    Y_UNIT_TEST(NoContainerInStreamHolderTest) {
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated(defaultFileSize, defaultRotateSize);
        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStream), isBoxAgentMode);

        auto rotateResult = fileStreamHolder->TryRotate(defaultContainer);
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::NoLogsFileStreamForWorkload, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS(TStringBuilder() << "No logs file stream for workload: " << defaultContainer.Container, rotateResult.Error().Message);

        auto getFileStreamResult = fileStreamHolder->GetFileStream(defaultContainer);
        UNIT_ASSERT(!getFileStreamResult);
        UNIT_ASSERT_EQUAL(EPushClientError::NoLogsFileStreamForWorkload, getFileStreamResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS(TStringBuilder() << "No logs file stream for workload: " << defaultContainer.Container, getFileStreamResult.Error().Message);
    }

    Y_UNIT_TEST(UpdateOkTest) {
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated(defaultFileSize, defaultRotateSize);
        TFileStreamRotatedFactoryPtr fileStreamFactory = new TTestFileStreamRotatedFactory(fileStream);
        ELogType logType = ELogType::Stdout;

        for (const bool curIsBoxAgentMode : {false, true}) {
            TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, logType, pathHolder, fileStreamFactory, curIsBoxAgentMode);
            fileStreamHolder->Update(pushContainers);

            TPushContainer pushContainer1 = {{"box_1/workload_1"}, {"1"}, {"1"}};
            UNIT_ASSERT(fileStreamHolder->GetFileStream(pushContainer1));

            TPushContainer pushContainer2 = {{"box_2/workload_2"}, {"2"}, {"2"}};
            UNIT_ASSERT(fileStreamHolder->GetFileStream(pushContainer2));

            TPushContainer pushContainer3 = {{"box_3/workload_3"}, {"3"}, {"3"}};
            UNIT_ASSERT(fileStreamHolder->GetFileStream(pushContainer3));

            pushContainers.erase(pushContainer3);
            TPushContainer pushContainer4 = {{"box_4/workload_4"}, {"4"}, {"4"}};
            pushContainers.insert(pushContainer4);

            fileStreamHolder->Update(pushContainers);

            auto getResult = fileStreamHolder->GetFileStream(pushContainer3);
            UNIT_ASSERT(!getResult);
            UNIT_ASSERT_EQUAL(EPushClientError::NoLogsFileStreamForWorkload, getResult.Error().Errno);

            UNIT_ASSERT(fileStreamHolder->GetFileStream(pushContainer4));

            if (curIsBoxAgentMode) {
                UNIT_ASSERT_EQUAL(
                    ((TTestFileStreamRotatedFactory*)fileStreamFactory.Get())->FileStreamPath
                    , pathHolder->GetWorkloadLogsFilePathInBox(pushContainer4.WorkloadId, ToString(logType))
                );
            } else {
                UNIT_ASSERT_EQUAL(
                    ((TTestFileStreamRotatedFactory*)fileStreamFactory.Get())->FileStreamPath
                    , pathHolder->GetWorkloadLogsFilePathAtLogsVolumePath(pushContainer4.WorkloadId, ToString(logType))
                );
            }

            pushContainers.erase(pushContainer4);
            pushContainers.insert(pushContainer3);
        }
    }

    Y_UNIT_TEST(TryRotateOkTest) {
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated(defaultFileSize, defaultRotateSize);
        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStream), isBoxAgentMode);
        fileStreamHolder->Update(pushContainers);

        auto rotateResult = fileStreamHolder->TryRotate(defaultContainer);
        UNIT_ASSERT(rotateResult);
        UNIT_ASSERT_EQUAL(defaultRotateSize, rotateResult.Success());

        auto fileStream1 = fileStreamHolder->GetFileStream(defaultContainer).Success();
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->RotateCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->GetFileSizeCalls);
    }

    Y_UNIT_TEST(TryRotateWhenGetFileSizeReturnErrorTest) {
        struct TTestFileStreamRotatedFail: public TTestFileStreamRotated {
            TTestFileStreamRotatedFail(size_t fileSize, size_t rotateSize)
                    : TTestFileStreamRotated(fileSize, rotateSize)
            {}

            TExpected<ui64, TPushClientError> GetFileSize() const override {
                ++GetFileSizeCalls;
                return TPushClientError{EPushClientError::FileNotExists, "GetFileSize error"};
            }
        };

        TFileStreamRotatedPtr fileStreamFail = new TTestFileStreamRotatedFail(defaultFileSize, defaultRotateSize);
        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStreamFail), isBoxAgentMode);
        fileStreamHolder->Update(pushContainers);

        auto rotateResult = fileStreamHolder->TryRotate(defaultContainer);
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileNotExists, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("GetFileSize error", rotateResult.Error().Message);

        auto fileStream1 = fileStreamHolder->GetFileStream(defaultContainer).Success();
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream1.Get())->RotateCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->GetFileSizeCalls);
    }

    Y_UNIT_TEST(TryRotateWhenSmallFileTest) {
        size_t smallFileSize = 0;
        TFileStreamRotatedPtr fileStreamSmallFileSize = new TTestFileStreamRotated(smallFileSize, defaultRotateSize);
        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStreamSmallFileSize), isBoxAgentMode);
        fileStreamHolder->Update(pushContainers);

        auto rotateResult = fileStreamHolder->TryRotate(defaultContainer);
        UNIT_ASSERT(rotateResult);
        UNIT_ASSERT_EQUAL(0, rotateResult.Success());

        auto fileStream1 = fileStreamHolder->GetFileStream(defaultContainer).Success();
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream1.Get())->RotateCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->GetFileSizeCalls);
    }

    Y_UNIT_TEST(TryRotateWhenRotateReturnErrorTest) {
        struct TTestFileStreamRotatedFail: public TTestFileStreamRotated {
            TTestFileStreamRotatedFail(size_t fileSize, size_t rotateSize)
                : TTestFileStreamRotated(fileSize, rotateSize)
            {}

            TExpected<ui64, TPushClientError> Rotate() override {
                ++RotateCalls;
                return TPushClientError{EPushClientError::FileRotateError, "Rotate error"};
            }
        };

        TFileStreamRotatedPtr fileStreamFail = new TTestFileStreamRotatedFail(defaultFileSize, defaultRotateSize);
        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStreamFail), isBoxAgentMode);
        fileStreamHolder->Update(pushContainers);

        auto rotateResult = fileStreamHolder->TryRotate(defaultContainer);
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileRotateError, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("Rotate error", rotateResult.Error().Message);

        auto fileStream1 = fileStreamHolder->GetFileStream(defaultContainer).Success();
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->RotateCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->GetFileSizeCalls);
    }

    Y_UNIT_TEST(TryRotateWhenParamsAreInvalidTest) {
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated(defaultFileSize, defaultRotateSize);
        size_t invalidMaxLogsFileSize = 1000;
        double invalidMinPortionToCut = 0.8;
        blksize_t invalidFileSystemBlockSize = 700;

        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(invalidMaxLogsFileSize, invalidMinPortionToCut, invalidFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStream), isBoxAgentMode);
        fileStreamHolder->Update(pushContainers);

        auto rotateResult = fileStreamHolder->TryRotate(defaultContainer);
        UNIT_ASSERT(!rotateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileRotateError, rotateResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS(
            rotateResult.Error().Message
            , TStringBuilder()
                << "Rotate params are invalid: maxLogsFileSize = " << invalidMaxLogsFileSize
                << ", MinPortionToCut = " << invalidMinPortionToCut
                << ", FileSystem block size = " << invalidFileSystemBlockSize
        );

        auto fileStream1 = fileStreamHolder->GetFileStream(defaultContainer).Success();
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream1.Get())->RotateCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream1.Get())->GetFileSizeCalls);
    }

    Y_UNIT_TEST(FilesBecomeInvalidWhenAllFilesAreValidTest) {
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated(defaultFileSize, defaultRotateSize);

        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStream), isBoxAgentMode);
        fileStreamHolder->Update(pushContainers);

        auto filesBecameInvalid = fileStreamHolder->FilesBecameInvalid();
        UNIT_ASSERT(filesBecameInvalid.empty());

        UNIT_ASSERT_EQUAL(pushContainers.size(), ((TTestFileStreamRotated*)fileStream.Get())->HasBecomeInvalidCalls);
    }

    Y_UNIT_TEST(FilesBecomeInvalidWhenAllFilesAreNotValidTest) {
        struct TTestFileStreamRotatedInvalid: public TTestFileStreamRotated {
            TTestFileStreamRotatedInvalid(size_t fileSize, size_t rotateSize)
                    : TTestFileStreamRotated(fileSize, rotateSize)
            {}

            TMaybe<TPushClientError> HasBecomeInvalid() override {
                ++HasBecomeInvalidCalls;
                return TPushClientError{EPushClientError::FileInodeChanged, "file inode changed"};
            }
        };

        TFileStreamRotatedPtr fileStreamFail = new TTestFileStreamRotatedInvalid(defaultFileSize, defaultRotateSize);
        TFileStreamHolderPtr fileStreamHolder = new TFileStreamHolderImpl(defaultMaxLogsFileSize, defaultMinPortionToCut, defaultFileSystemBlockSize, ELogType::Stdout, pathHolder, new TTestFileStreamRotatedFactory(fileStreamFail), isBoxAgentMode);
        fileStreamHolder->Update({defaultContainer});

        auto filesBecameInvalid = fileStreamHolder->FilesBecameInvalid();
        UNIT_ASSERT_EQUAL(filesBecameInvalid.size(), 1);
        UNIT_ASSERT_EQUAL(EPushClientError::FileInodeChanged, filesBecameInvalid.at(0).Errno);
        UNIT_ASSERT_STRING_CONTAINS("file inode changed", filesBecameInvalid.at(0).Message);

        auto fileStream1 = fileStreamHolder->GetFileStream(defaultContainer).Success();
        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream1.Get())->HasBecomeInvalidCalls);
    }
}

}
