#include "logs_file_offset_holder_impl.h"
#include "serializable_offset_holder_mock.h"

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_impl.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NOffsetHolderTest {

static TLogger logger({});

static TPushContainer pushContainer1 = TPushContainer{{"box_1/workload_1"}, {"1"}, {"1"}};
static const ui64 defaultOffset = 100;

struct TTestSerializableOffsetHolder: public TMockSerializableOffsetHolder {
    TExpected<void, TPushClientError> UpdateOffset(const TPushContainer& container, ui64 offset) override {
        ++UpdateOffsetCalls;
        ContainerToOffset[container] = offset;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer& container) const override {
        ++GetOffsetCalls;
        return ContainerToOffset[container];
    }

    size_t UpdateOffsetCalls = 0;
    mutable size_t GetOffsetCalls = 0;
    mutable THashMap<TPushContainer, ui64> ContainerToOffset;
};

Y_UNIT_TEST_SUITE(LogsFileOffsetHolderSuite) {

    Y_UNIT_TEST(DecrementOffsetOkTest) {
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSerializableOffsetHolderPtr offsetHolderImpl = new TTestSerializableOffsetHolder;
        TLogsFileOffsetHolderImplPtr logsFileOffsetHolder = new TLogsFileOffsetHolder(logger.SpawnFrame(), statistics, offsetHolderImpl);

        UNIT_ASSERT(logsFileOffsetHolder->UpdateOffset(pushContainer1, defaultOffset));

        UNIT_ASSERT_EQUAL(defaultOffset, logsFileOffsetHolder->GetOffset(pushContainer1).Success());

        ui64 delta = 10;
        auto decrementResult = logsFileOffsetHolder->DecrementOffset(pushContainer1, delta);
        UNIT_ASSERT(decrementResult);

        UNIT_ASSERT_EQUAL(2, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(2, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->UpdateOffsetCalls);

        UNIT_ASSERT_EQUAL(defaultOffset - delta, logsFileOffsetHolder->GetOffset(pushContainer1).Success());
    }

    Y_UNIT_TEST(DecrementOffsetWhenDeltaMoreThanOffsetTest) {
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSerializableOffsetHolderPtr offsetHolderImpl = new TTestSerializableOffsetHolder;
        TLogsFileOffsetHolderImplPtr logsFileOffsetHolder = new TLogsFileOffsetHolder(logger.SpawnFrame(), statistics, offsetHolderImpl);

        UNIT_ASSERT(logsFileOffsetHolder->UpdateOffset(pushContainer1, defaultOffset));

        UNIT_ASSERT_EQUAL(defaultOffset, logsFileOffsetHolder->GetOffset(pushContainer1).Success());

        ui64 bigDelta = 500;
        ui32 pushValue = bigDelta - defaultOffset;

        auto decrementResult = logsFileOffsetHolder->DecrementOffset(pushContainer1, bigDelta);
        UNIT_ASSERT(decrementResult);

        UNIT_ASSERT_EQUAL(2, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(2, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->UpdateOffsetCalls);

        UNIT_ASSERT_EQUAL(0, logsFileOffsetHolder->GetOffset(pushContainer1).Success());

        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotated(), pushValue);
    }

    Y_UNIT_TEST(DecrementOffsetFailsWhenNoContainerInHolderTest) {
        struct TTestSerializableOffsetHolderFails: public TTestSerializableOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{.Errno = EPushClientError::NoOffsetForWorkload};
            }
        };

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSerializableOffsetHolderPtr offsetHolderImpl = new TTestSerializableOffsetHolderFails;
        TLogsFileOffsetHolderImplPtr logsFileOffsetHolder = new TLogsFileOffsetHolder(logger.SpawnFrame(), statistics, offsetHolderImpl);

        ui64 delta = 10;
        auto decrementResult = logsFileOffsetHolder->DecrementOffset(pushContainer1, delta);

        UNIT_ASSERT(!decrementResult);
        UNIT_ASSERT_EQUAL(EPushClientError::NoOffsetForWorkload, decrementResult.Error().Errno);

        UNIT_ASSERT_EQUAL(1, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->UpdateOffsetCalls);
    }

}

}
