#include "porto_offset_holder_impl.h"
#include "serializable_offset_holder_impl.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>
#include <infra/pod_agent/libs/porto_client/client.h>

#include <util/generic/hash_set.h>

namespace NInfra::NPodAgent {

TPortoOffsetHolderImpl::TPortoOffsetHolderImpl(
    TLogFramePtr logFrame
    , TLogsTransmitterStatisticsPtr statistics
    , TSerializableOffsetHolderPtr serializableOffsetHolderImpl
)
    : LogFrame_(logFrame)
    , Statistics_(statistics)
    , SerializableOffsetHolderImpl_(serializableOffsetHolderImpl)
{}

void TPortoOffsetHolderImpl::Update(const THashSet<TPushContainer>& pushContainers) {
    SerializableOffsetHolderImpl_->Update(pushContainers);
}

TExpected<ui64, TPushClientError> TPortoOffsetHolderImpl::GetOffset(const TPushContainer& container) const {
    return SerializableOffsetHolderImpl_->GetOffset(container);
}

TExpected<void, TPushClientError> TPortoOffsetHolderImpl::UpdateOffset(const TPushContainer& container, ui64 offset) {
    auto cachedOffset = OUTCOME_TRYX(SerializableOffsetHolderImpl_->GetOffset(container));

    //1. offset > cachedOffset
       //a. container didn't restart - porto rotated data size = offset - cachedOffset (we can detect this case)
       //b. container restarted - porto rotated data size = offset (we can't detect this case)
    //2. offset == cachedOffset
       //a. container didn't restart - porto didn't rotate data
       //b. container restarted - porto rotated data size = offset (we can't detect this case)
    //3. (offset < cachedOffset) && (offset > 0)
       //a. container didn't restart - impossible case
       //b. container restarted - porto rotated data size = offset (we can detect this case, not implemented)
    //4. offset < cachedOffset && (offset == 0)
       //a. container didn't restart - impossible case
       //b. container restarted - porto didn't rotate data

    if (offset > cachedOffset) {
        Statistics_->AddNumBytesRotatedByPorto(offset - cachedOffset);
    }

    return SerializableOffsetHolderImpl_->UpdateOffset(container, offset);
}

ui64 TPortoOffsetHolderImpl::NumOfOffsets() const {
    return SerializableOffsetHolderImpl_->NumOfOffsets();
}

void TPortoOffsetHolderImpl::SerializeAll(TSerializerPtr serializer) const {
    SerializableOffsetHolderImpl_->SerializeAll(serializer);
}

void TPortoOffsetHolderImpl::Serialize(const THashSet<TPushContainer>& pushContainers, TSerializerPtr serializer) const {
    SerializableOffsetHolderImpl_->Serialize(pushContainers, serializer);
}

void TPortoOffsetHolderImpl::Deserialize(const THashSet<TPushContainer>& pushContainers, TDeserializerPtr deserializer) {
    SerializableOffsetHolderImpl_->Deserialize(pushContainers, deserializer);
}

}

