#include "porto_offset_holder_impl.h"
#include "serializable_offset_holder_mock.h"

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_impl.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NPortoOffsetHolderTest {

static TLogger logger({});

static TPushContainer pushContainer1 = TPushContainer{{"box_1/workload_1"}, {"1"}, {"1"}};
static THashSet<TPushContainer> pushContainers = {
    TPushContainer{{"box_1/workload_1"}, {"1"}, {"1"}}
    , TPushContainer{{"box_2/workload_2"}, {"2"}, {"2"}}
    , TPushContainer{{"box_3/workload_3"}, {"3"}, {"3"}}
};

static const ui64 defaultOffset = 100;

struct TTestSerializableOffsetHolder: public TMockSerializableOffsetHolder {
    TExpected<void, TPushClientError> UpdateOffset(const TPushContainer& container, ui64 offset) override {
        ++UpdateOffsetCalls;
        ContainerToOffset[container] = offset;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer& container) const override {
        ++GetOffsetCalls;
        return ContainerToOffset[container];
    }

    size_t UpdateOffsetCalls = 0;
    mutable size_t GetOffsetCalls = 0;
    mutable THashMap<TPushContainer, ui64> ContainerToOffset;
};

Y_UNIT_TEST_SUITE(PortoOffsetHolderSuite) {

    Y_UNIT_TEST(NoStatistsicsUpdateIfNewOffsetEqualTo0Test) {
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSerializableOffsetHolderPtr offsetHolderImpl = new TTestSerializableOffsetHolder;
        TPortoOffsetHolderPtr portoOffsetHolder = new TPortoOffsetHolderImpl(logger.SpawnFrame(), statistics, offsetHolderImpl);

        for (const TPushContainer& pushContainer : pushContainers) {
            portoOffsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotatedByPorto(), defaultOffset * pushContainers.size());

        ui64 newOffset = 0;
        auto updateResult = portoOffsetHolder->UpdateOffset(pushContainer1, newOffset);
        UNIT_ASSERT(updateResult);

        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotatedByPorto(), defaultOffset * pushContainers.size());

        UNIT_ASSERT_EQUAL(pushContainers.size() + 1, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(pushContainers.size() + 1, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->UpdateOffsetCalls);
    }

    Y_UNIT_TEST(UpdateStatistsicsUpdateIfNewOffsetIsGreaterThenPrevTest) {
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSerializableOffsetHolderPtr offsetHolderImpl = new TTestSerializableOffsetHolder;
        TPortoOffsetHolderPtr portoOffsetHolder = new TPortoOffsetHolderImpl(logger.SpawnFrame(), statistics, offsetHolderImpl);

        for (const TPushContainer& pushContainer : pushContainers) {
            portoOffsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotatedByPorto(), defaultOffset * pushContainers.size());

        ui64 newOffset = defaultOffset + 1;
        auto updateResult = portoOffsetHolder->UpdateOffset(pushContainer1, newOffset);
        UNIT_ASSERT(updateResult);

        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotatedByPorto(), defaultOffset * pushContainers.size() + 1);

        UNIT_ASSERT_EQUAL(pushContainers.size() + 1, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(pushContainers.size() + 1, ((TTestSerializableOffsetHolder*)offsetHolderImpl.Get())->UpdateOffsetCalls);
    }

    Y_UNIT_TEST(UpdateOffsetFailWhenHolderImplReturnsErrorTest) {
        struct TTestOffsetHolderFails: public TTestSerializableOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{.Errno = EPushClientError::NoOffsetForWorkload};
            }
        };

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSerializableOffsetHolderPtr offsetHolderImplFails = new TTestOffsetHolderFails;
        TPortoOffsetHolderPtr portoOffsetHolder = new TPortoOffsetHolderImpl(logger.SpawnFrame(), statistics, offsetHolderImplFails);

        TPushContainer unknownContainer = TPushContainer{{"box_5/workload_5"}, {"5"}, {"5"}};
        auto updateResult = portoOffsetHolder->UpdateOffset(unknownContainer, defaultOffset);

        UNIT_ASSERT(!updateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::NoOffsetForWorkload, updateResult.Error().Errno);

        UNIT_ASSERT_EQUAL(1, ((TTestSerializableOffsetHolder*)offsetHolderImplFails.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSerializableOffsetHolder*)offsetHolderImplFails.Get())->UpdateOffsetCalls);
    }

}

}
