#include "serializable_offset_holder_impl.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>

namespace NInfra::NPodAgent {

TSerializableOffsetHolder::TSerializableOffsetHolder(TLogFramePtr logFrame)
    : LogFrame_(logFrame)
{}

void TSerializableOffsetHolder::Update(const THashSet<TPushContainer>& pushContainers) {
    THashMap<TPushContainer, TOffsetDescription> containerToOffsetUpdated;

    for (const auto& pushContainer : pushContainers) {
        if (ContainerToOffsetDescription_.contains(pushContainer)) {
            containerToOffsetUpdated[pushContainer] = ContainerToOffsetDescription_[pushContainer];
        } else {
            containerToOffsetUpdated[pushContainer] = {0, 0};
        }
    }

    ContainerToOffsetDescription_ = containerToOffsetUpdated;
}

TExpected<ui64, TPushClientError> TSerializableOffsetHolder::GetOffset(const TPushContainer& container) const {
    if (!ContainerToOffsetDescription_.contains(container)) {
        return TPushClientError{EPushClientError::NoOffsetForWorkload, TStringBuilder() << "No offset for workload: " << container.Container};
    }

    return ContainerToOffsetDescription_.at(container).Offset_;
}

TExpected<void, TPushClientError> TSerializableOffsetHolder::UpdateOffset(const TPushContainer& container, ui64 offset) {
    if (!ContainerToOffsetDescription_.contains(container)) {
        return TPushClientError{EPushClientError::NoOffsetForWorkload, TStringBuilder() << "No offset for workload: " << container.Container};
    }

    ContainerToOffsetDescription_[container].Offset_ = offset;
    return TExpected<void, TPushClientError>::DefaultSuccess();
}

ui64 TSerializableOffsetHolder::NumOfOffsets() const {
    return ContainerToOffsetDescription_.size();
}

void TSerializableOffsetHolder::SerializeAll(TSerializerPtr serializer) const {
    for (const auto& entry: ContainerToOffsetDescription_) {
        SerializeIfUpdated(entry.first, entry.second, serializer);
    }
}

void TSerializableOffsetHolder::Serialize(const THashSet<TPushContainer>& pushContainers, TSerializerPtr serializer) const {
    for (const auto& pushContainer: pushContainers) {
        auto offsetIt = ContainerToOffsetDescription_.find(pushContainer);
        if (offsetIt == ContainerToOffsetDescription_.end()) {
            auto error = TPushClientError{EPushClientError::NoOffsetForWorkload, TStringBuilder() << "No offset for workload while serialization: " << pushContainer.Container};
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(error));
            return;
        }

        TOffsetDescription offsetDescription = offsetIt->second;
        this->SerializeIfUpdated(pushContainer, offsetDescription, serializer);
    }
}

void TSerializableOffsetHolder::SerializeIfUpdated(const TPushContainer& pushContainer, const TOffsetDescription& offsetDescription, TSerializerPtr serializer) const {
    if (offsetDescription.LastSerializedOffset_ < offsetDescription.Offset_) {
        serializer->Serialize(pushContainer, offsetDescription.Offset_);
        ContainerToOffsetDescription_[pushContainer].LastSerializedOffset_ = offsetDescription.Offset_;
    }
}

void TSerializableOffsetHolder::Deserialize(const THashSet<TPushContainer>& pushContainers, TDeserializerPtr deserializer) {

    for (const auto& pushContainer: pushContainers) {
        ui64 offset = 0;
        deserializer->Deserialize(pushContainer, offset);

        if (offset) {
            auto updateResult = UpdateOffset(pushContainer, offset);
            if (!updateResult) {
                LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(updateResult.Error()));
            }
            ContainerToOffsetDescription_[pushContainer].LastSerializedOffset_ = offset;
        }
    }
}

} //namespace NInfra::NPodAgent
