#include "serializable_offset_holder_impl.h"

#include <infra/libs/logger/logger.h>

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/serializer_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/deserializer_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/containers_set_builder.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NOffsetHolderTest {

static TLogger logger({});

static TPushContainer pushContainer1 = TPushContainer{{"box_1/workload_1"}, {"1"}, {"1"}};
static TPushContainer pushContainer2 = TPushContainer{{"box_2/workload_2"}, {"2"}, {"2"}};
static TPushContainer pushContainer3 = TPushContainer{{"box_3/workload_3"}, {"3"}, {"3"}};
static THashSet<TPushContainer> pushContainers = {
    TPushContainer{{"box_1/workload_1"}, {"1"}, {"1"}}
    , TPushContainer{{"box_2/workload_2"}, {"2"}, {"2"}}
    , TPushContainer{{"box_3/workload_3"}, {"3"}, {"3"}}
};

static const ui64 defaultOffset = 100;
static const ui64 defaultDeserializeValue = 200;

struct TTestSerializer: public TMockSerializer {
public:
    void Serialize(const TPushContainer&, ui64) override {
        ++SerializeCalls;
    }

    size_t SerializeCalls = 0;
};

struct TTestDeserializer: public TMockDeserializer {
public:
    void Deserialize(const TPushContainer&, ui64& value) override {
        ++DeserializeCalls;
        value = defaultDeserializeValue;
    }

    size_t DeserializeCalls = 0;
};

Y_UNIT_TEST_SUITE(OffsetHolderSuite) {

    Y_UNIT_TEST(UpdateOffsetsWithoutCacheOkTest) {
        TSerializableOffsetHolderPtr offsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        offsetHolder->Update(pushContainers);
        UNIT_ASSERT_EQUAL(3, offsetHolder->NumOfOffsets());

        for (const TPushContainer& pushContainer : pushContainers) {
            offsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer1));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer1).Success());

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer2));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer2).Success());

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer3));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer3).Success());
    }

    Y_UNIT_TEST(GetOffsetFailTest) {
        TSerializableOffsetHolderPtr offsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        TPushContainer unknownContainer = TPushContainer{{"box_5/workload_5"}, {"5"}, {"5"}};
        offsetHolder->Update(pushContainers);

        auto getOffsetResult = offsetHolder->GetOffset(unknownContainer);
        UNIT_ASSERT(!getOffsetResult);
        UNIT_ASSERT_EQUAL(EPushClientError::NoOffsetForWorkload, getOffsetResult.Error().Errno);
    }

    Y_UNIT_TEST(UpdateOffsetOkTest) {
        TSerializableOffsetHolderPtr offsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        offsetHolder->Update(pushContainers);

        for (const TPushContainer& pushContainer : pushContainers) {
            offsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer1).Success());
        UNIT_ASSERT_EQUAL(3, offsetHolder->NumOfOffsets());

        ui64 newOffset = 10;
        auto updateResult = offsetHolder->UpdateOffset(pushContainer1, newOffset);
        UNIT_ASSERT(updateResult);

        UNIT_ASSERT_EQUAL(newOffset, offsetHolder->GetOffset(pushContainer1).Success());
        UNIT_ASSERT_EQUAL(3, offsetHolder->NumOfOffsets());
    }

    Y_UNIT_TEST(UpdateOffsetFailTest) {
        TSerializableOffsetHolderPtr offsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        TPushContainer unknownContainer = TPushContainer{{"box_5/workload_5"}, {"5"}, {"5"}};
        offsetHolder->Update(pushContainers);
        auto updateResult = offsetHolder->UpdateOffset(unknownContainer, defaultOffset);

        UNIT_ASSERT(!updateResult);
        UNIT_ASSERT_EQUAL(EPushClientError::NoOffsetForWorkload, updateResult.Error().Errno);

        UNIT_ASSERT_EQUAL(3, offsetHolder->NumOfOffsets());
    }

    Y_UNIT_TEST(UpdateOffsetsWithCacheOkTest) {
        TSerializableOffsetHolderPtr offsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        offsetHolder->Update(pushContainers);
        UNIT_ASSERT_EQUAL(offsetHolder->NumOfOffsets(), 3);

        for (const TPushContainer& pushContainer : pushContainers) {
            offsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer1));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer1).Success());

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer2));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer2).Success());

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer3));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer3).Success());

        TPushContainer pushContainer4 = TPushContainer{{"box_4/workload_4"}, {"4"}, {"4"}};
        THashSet<TPushContainer> pushContainers2 = {
            pushContainer2
            , pushContainer4
        };

        offsetHolder->Update(pushContainers2);
        UNIT_ASSERT_EQUAL(offsetHolder->NumOfOffsets(), 2);

        for (const TPushContainer& pushContainer : pushContainers2) {
            offsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        UNIT_ASSERT(!offsetHolder->GetOffset(pushContainer1));
        UNIT_ASSERT(!offsetHolder->GetOffset(pushContainer3));

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer2));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer2).Success());

        UNIT_ASSERT(offsetHolder->GetOffset(pushContainer4));
        UNIT_ASSERT_EQUAL(defaultOffset, offsetHolder->GetOffset(pushContainer4).Success());
    }

    Y_UNIT_TEST(SerializeAllTest) {
        TSerializerPtr serializer = new TTestSerializer;
        TSerializableOffsetHolderPtr serializableOffsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        serializableOffsetHolder->Update(pushContainers);
        for (const TPushContainer& pushContainer : pushContainers) {
            serializableOffsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        serializableOffsetHolder->SerializeAll(serializer);

        size_t serializeCalls = pushContainers.size();
        UNIT_ASSERT_EQUAL(serializeCalls, ((TTestSerializer*)serializer.Get())->SerializeCalls);

        serializableOffsetHolder->UpdateOffset(*pushContainers.begin(), defaultOffset + 100);

        serializableOffsetHolder->SerializeAll(serializer);
        // Containers without updated offsets were not serialized.
        UNIT_ASSERT_EQUAL(serializeCalls + 1, ((TTestSerializer*)serializer.Get())->SerializeCalls);
    }

    Y_UNIT_TEST(SerializeTest) {
        TSerializerPtr serializer = new TTestSerializer;
        TSerializableOffsetHolderPtr serializableOffsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        THashSet<TPushContainer> pushContainersToSerialize = {
            TPushContainer{{"box_4/workload_4"}, {"4"}, {"4"}}
            , TPushContainer{{"box_5/workload_5"}, {"5"}, {"5"}}
        };

        THashSet<TPushContainer> allPushContainers = TContainersSetBuilder::Init(pushContainers)
            ->Union(pushContainersToSerialize)
            ->Result();

        serializableOffsetHolder->Update(allPushContainers);
        for (const TPushContainer& pushContainer : allPushContainers) {
            serializableOffsetHolder->UpdateOffset(pushContainer, defaultOffset);
        }

        serializableOffsetHolder->Serialize(pushContainersToSerialize, serializer);

        size_t serializeCalls = pushContainersToSerialize.size();
        UNIT_ASSERT_EQUAL(serializeCalls, ((TTestSerializer*)serializer.Get())->SerializeCalls);

        serializableOffsetHolder->UpdateOffset(*pushContainersToSerialize.begin(), defaultOffset + 100);
        serializableOffsetHolder->Serialize(pushContainersToSerialize, serializer);
        // Containers without updated offsets were not serialized.
        UNIT_ASSERT_EQUAL(serializeCalls + 1, ((TTestSerializer*)serializer.Get())->SerializeCalls);
    }

    Y_UNIT_TEST(DeserializeTest) {
        TDeserializerPtr deserializer = new TTestDeserializer;
        TSerializableOffsetHolderPtr serializableOffsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        serializableOffsetHolder->Update(pushContainers);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer1).Success(), 0);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer2).Success(), 0);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer3).Success(), 0);

        serializableOffsetHolder->Deserialize(pushContainers, deserializer);

        UNIT_ASSERT_EQUAL(pushContainers.size(), ((TTestDeserializer*)deserializer.Get())->DeserializeCalls);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer1).Success(), defaultDeserializeValue);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer2).Success(), defaultDeserializeValue);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer3).Success(), defaultDeserializeValue);

        TSerializerPtr serializer = new TTestSerializer;
        serializableOffsetHolder->Serialize(pushContainers, serializer);
        // Serializing after deserializing does nothing
        UNIT_ASSERT_EQUAL(0, ((TTestSerializer*)serializer.Get())->SerializeCalls);
    }

    Y_UNIT_TEST(DeserializeWhenDeserializerNotChangeValueTest) {
        struct TTestDeserializerFail: public TTestDeserializer {
            void Deserialize(const TPushContainer&, ui64&) override {
                ++DeserializeCalls;
            }
        };

        TDeserializerPtr deserializer = new TTestDeserializerFail;
        TSerializableOffsetHolderPtr serializableOffsetHolder = new TSerializableOffsetHolder(logger.SpawnFrame());

        serializableOffsetHolder->Update(pushContainers);
        serializableOffsetHolder->Deserialize(pushContainers, deserializer);

        UNIT_ASSERT_EQUAL(pushContainers.size(), ((TTestDeserializerFail*)deserializer.Get())->DeserializeCalls);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer1).Success(), 0);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer2).Success(), 0);
        UNIT_ASSERT_EQUAL(serializableOffsetHolder->GetOffset(pushContainer3).Success(), 0);
    }

}

}
