#include "session_holder_impl.h"

#include <util/generic/hash_set.h>
#include <util/string/vector.h>

namespace NInfra::NPodAgent {

TSessionHolderImpl::TSessionHolderImpl(const TString& logName, const ELogType& typeOfLog, TPushSessionFactoryPtr pushSessionFactory, TLogsTransmitterStatisticsPtr statistics)
    : LogName_(logName)
    , TypeOflog_(typeOfLog)
    , Statistics_(statistics)
    , PushSessionFactory_(pushSessionFactory)
{
}

size_t TSessionHolderImpl::NumOfSessions() const {
    return PushContainerToSession_.size();
}

void TSessionHolderImpl::Update(const THashSet<TPushContainer>& pushContainers) {
    THashMap<TPushContainer, TPushSessionPtr> updatedSessionIdToSession;

    for (const auto& pushContainer: pushContainers) {
        if (PushContainerToSession_.contains(pushContainer)) {
            updatedSessionIdToSession[pushContainer] = PushContainerToSession_.at(pushContainer);
        } else {
            TPushSessionPtr session = PushSessionFactory_->Create(pushContainer);
            updatedSessionIdToSession[pushContainer] = session;
        }
    }
    PushContainerToSession_ = updatedSessionIdToSession;
}

TExpected<TPushSessionPtr, TPushClientError> TSessionHolderImpl::GetSession(const TPushContainer& pushContainer) const {
    if (!PushContainerToSession_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoSessionForWorkload, TStringBuilder() << "no " << TypeOflog_ << " push session for workload: " << pushContainer.Container};
    }

    return PushContainerToSession_.at(pushContainer);
}

size_t TSessionHolderImpl::NumOfActiveSessions() const {
    size_t numOfActiveSessions = 0;

    for (const auto& session: PushContainerToSession_) {
        if (session.second->IsActive()) {
            numOfActiveSessions++;
        }
    }

    return numOfActiveSessions;
}


TExpected<void, TPushClientError> TSessionHolderImpl::DeactivateSession(const TPushContainer& pushContainer) {
    if (!PushContainerToSession_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoSessionForWorkload, TStringBuilder() << "no " << TypeOflog_ << " push session for workload: " << pushContainer.Container};
    }

    PushContainerToSession_[pushContainer]->Deactivate();

    return TExpected<void, TPushClientError>::DefaultSuccess();
}

TExpected<TPushSessionPtr, TPushClientError> TSessionHolderImpl::ActivateSession(const TPushContainer& pushContainer) {
    if (!PushContainerToSession_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoSessionForWorkload, TStringBuilder() << "no " << TypeOflog_ << " push session for workload: " << pushContainer.Container};
    }

    auto session = PushContainerToSession_[pushContainer];
    TString sessionId = session->GetId();

    auto activationResult = session->Activate(LogName_, sessionId);

    if (!activationResult) {
        Statistics_->SessionActivationErrorOccured();
        Statistics_->ErrorFromPushAgentOccured();

        return activationResult.Error();
    }

    return session;
}

TExpected<void, TPushClientError> TSessionHolderImpl::IncreaseLogSeqNumber(ui32 increaseFactor, const TPushContainer& pushContainer) {
    if (!PushContainerToSession_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoSessionForWorkload, TStringBuilder() << "no " << TypeOflog_ << " push session for workload: " << pushContainer.Container};
    }

    PushContainerToSession_[pushContainer]->IncreaseLogSeqNumber(increaseFactor);

    return TExpected<void, TPushClientError>::DefaultSuccess();
}

TExpected<int, TPushClientError> TSessionHolderImpl::IsActive(const TPushContainer& pushContainer) const {
    if (!PushContainerToSession_.contains(pushContainer)) {
        return TPushClientError{EPushClientError::NoSessionForWorkload, TStringBuilder() << "no " << TypeOflog_ << " push session for workload: " << pushContainer.Container};
    }

    return PushContainerToSession_.at(pushContainer)->IsActive();
}

const THashMap<TPushContainer, TPushSessionPtr>& TSessionHolderImpl::GetSessions() const {
    return PushContainerToSession_;
}


} //namespace NInfra::NPodAgent
