#include "session_holder_impl.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_impl.h>
#include <infra/pod_agent/libs/push_client/sessions/push_session_impl.h>
#include <infra/pod_agent/libs/push_client/sessions/push_session_factory_impl.h>
#include <infra/pod_agent/libs/push_client/unified_agent_mock.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NSessionHolderTest {

Y_UNIT_TEST_SUITE(SessionHolderSuite) {

    Y_UNIT_TEST(TestUpdateSessionsOk) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderImplPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);

        TString boxId1 = "box1";
        TString workloadId1 = "workload1";

        TString boxId2 = "box2";
        TString workloadId2 = "workload2";

        TString boxId3 = "box3";
        TString workloadId3 = "workload3";

        TPushContainer container1 = TPushContainer{{boxId1, workloadId1}, boxId1, workloadId1};
        TPushContainer container2 = TPushContainer{{boxId2, workloadId2}, boxId2, workloadId2};
        TPushContainer container3 = TPushContainer{{boxId3, workloadId3}, boxId3, workloadId3};

        THashSet<TPushContainer> pushContainers1;
        pushContainers1.insert(container1);

        pushContainers1.insert(container2);

        sessionHolder->Update(pushContainers1);

        UNIT_ASSERT_EQUAL(sessionHolder->NumOfSessions(), 2);

        TPushSessionImpl refSession1(grpcChannel, ELogType::Stdout, container1, "");
        TPushSessionImpl refSession2(grpcChannel, ELogType::Stdout, container2, "");

        UNIT_ASSERT_EQUAL(*((TPushSessionImpl*)(sessionHolder->GetSession(container1).Success().Get())), refSession1);
        UNIT_ASSERT_EQUAL(*((TPushSessionImpl*)(sessionHolder->GetSession(container2).Success().Get())), refSession2);

        THashSet<TPushContainer> pushContainers2;
        pushContainers2.insert(container2);
        pushContainers2.insert(container3);

        sessionHolder->Update(pushContainers2);

        UNIT_ASSERT_EQUAL(sessionHolder->NumOfSessions(), 2);

        TPushSessionImpl refSession3(grpcChannel, ELogType::Stdout, container3, "");

        UNIT_ASSERT_EQUAL(*((TPushSessionImpl*)(sessionHolder->GetSession(container2).Success().Get())), refSession2);
        UNIT_ASSERT_EQUAL(*((TPushSessionImpl*)(sessionHolder->GetSession(container3).Success().Get())), refSession3);
    }

    Y_UNIT_TEST(TestIncreaseLogSeqNumberOk) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSessionHolderImplPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);

        TString boxId1 = "box1";
        TString workloadId1 = "workload1";

        THashSet<TPushContainer> pushContainers;
        TPushContainer pushContainer = {{boxId1, workloadId1}, boxId1, workloadId1};
        pushContainers.insert(pushContainer);

        sessionHolder->Update(pushContainers);
        UNIT_ASSERT_EQUAL(sessionHolder->GetSession(pushContainer).Success()->GetSeqNumber(), 1);

        auto increaseResult = sessionHolder->IncreaseLogSeqNumber(2, pushContainer);
        UNIT_ASSERT(increaseResult);

        UNIT_ASSERT_EQUAL(sessionHolder->GetSession(pushContainer).Success()->GetSeqNumber(), 3);
    }

    Y_UNIT_TEST(TestActivateAndDeactivateSessionOk) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSessionHolderImplPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);

        unifiedAgentModel.AddResponse(validSessionActivationResponse);

        TString boxId1 = "box1";
        TString workloadId1 = "workload1";

        THashSet<TPushContainer> pushContainers;
        TPushContainer pushContainer = {{boxId1, workloadId1}, boxId1, workloadId1};;
        pushContainers.insert(pushContainer);

        sessionHolder->Update(pushContainers);
        UNIT_ASSERT(!sessionHolder->IsActive(pushContainer).Success());

        auto activateResult = sessionHolder->ActivateSession(pushContainer);
        UNIT_ASSERT(activateResult);

        UNIT_ASSERT(sessionHolder->IsActive(pushContainer).Success());
        UNIT_ASSERT_EQUAL(sessionHolder->NumOfActiveSessions(), 1);
        UNIT_ASSERT_EQUAL(sessionHolder->NumOfSessions(), 1);

        auto deactivateResult = sessionHolder->DeactivateSession(pushContainer);
        UNIT_ASSERT(deactivateResult);

        UNIT_ASSERT(!sessionHolder->IsActive(pushContainer).Success());
        UNIT_ASSERT_EQUAL(sessionHolder->NumOfActiveSessions(), 0);
        UNIT_ASSERT_EQUAL(sessionHolder->NumOfSessions(), 1);
    }

    Y_UNIT_TEST(TestNoSessionInHolder) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSessionHolderImplPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);

        TString boxId = "box";
        TString workloadId = "workload";
        TPushContainer pushContainer = {{boxId, workloadId}, boxId, workloadId};

        const auto getSessionResult = sessionHolder->GetSession(pushContainer);

        UNIT_ASSERT(!getSessionResult);
        UNIT_ASSERT_EQUAL(getSessionResult.Error().Errno, EPushClientError::NoSessionForWorkload);
        UNIT_ASSERT_STRING_CONTAINS(getSessionResult.Error().Message, "no stdout push session for workload: box/workload");

        const auto activationResult = sessionHolder->ActivateSession(pushContainer);

        UNIT_ASSERT(!activationResult);
        UNIT_ASSERT_EQUAL(activationResult.Error().Errno, EPushClientError::NoSessionForWorkload);
        UNIT_ASSERT_STRING_CONTAINS(activationResult.Error().Message, "no stdout push session for workload: box/workload");

        const auto deactivationResult = sessionHolder->DeactivateSession(pushContainer);

        UNIT_ASSERT(!deactivationResult);
        UNIT_ASSERT_EQUAL(deactivationResult.Error().Errno, EPushClientError::NoSessionForWorkload);
        UNIT_ASSERT_STRING_CONTAINS(deactivationResult.Error().Message, "no stdout push session for workload: box/workload");

        ui32 increaseFactor = 1;
        const auto increaseResult = sessionHolder->IncreaseLogSeqNumber(increaseFactor, pushContainer);

        UNIT_ASSERT(!increaseResult);
        UNIT_ASSERT_EQUAL(increaseResult.Error().Errno, EPushClientError::NoSessionForWorkload);
        UNIT_ASSERT_STRING_CONTAINS(increaseResult.Error().Message, "no stdout push session for workload: box/workload");

        const auto checkResult = sessionHolder->IsActive(pushContainer);

        UNIT_ASSERT(!checkResult);
        UNIT_ASSERT_EQUAL(checkResult.Error().Errno, EPushClientError::NoSessionForWorkload);
        UNIT_ASSERT_STRING_CONTAINS(checkResult.Error().Message, "no stdout push session for workload: box/workload");
    }

    Y_UNIT_TEST(TestActivateSessionFailsWhenNoInitResponse) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());

        TString staticSecret = "static_secret";
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSessionHolderImplPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, staticSecret), statistics);

        TString boxId = "boxId";
        TString workloadId = "workloadId";
        TPushContainer pushContainer = {{boxId, workloadId}, boxId, workloadId};

        THashSet<TPushContainer> pushContainers;
        pushContainers.insert(pushContainer);

        sessionHolder->Update(pushContainers);

        TPushLogResponse invalidResponse;
        unifiedAgentModel.AddResponse(invalidResponse);

        auto activateResult = sessionHolder->ActivateSession(pushContainer);

        UNIT_ASSERT(!(bool)activateResult);
        UNIT_ASSERT_EQUAL(activateResult.Error().Errno, EPushClientError::InvalidResponse);
        UNIT_ASSERT_STRING_CONTAINS(activateResult.Error().Message, "not init response while stdout session init. logname: logName, sessionId: ");

        auto requests = unifiedAgentModel.GetRequests();
        UNIT_ASSERT_EQUAL(requests.size(), 1);

        auto activationRequest = requests.at(0);
        UNIT_ASSERT(activationRequest.has_initialize());

        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(0).GetName(), TPushSessionImpl::DEPLOY_BOX_KEY);
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(0).GetValue(), boxId);
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(1).GetName(), TPushSessionImpl::DEPLOY_WORKLOAD_KEY);
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(1).GetValue(), workloadId);
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(2).GetName(), TPushSessionImpl::DEPLOY_CONTAINER_ID_KEY);
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(2).GetValue(), ToString(pushContainer.Container));
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(3).GetName(), TPushSessionImpl::DEPLOY_LOGGER_NAME_KEY);
        UNIT_ASSERT_EQUAL(activationRequest.initialize().meta(3).GetValue(), ToString(ELogType::Stdout));
        UNIT_ASSERT_EQUAL(activationRequest.initialize().shared_secret_key(), staticSecret);

        UNIT_ASSERT(activationRequest.initialize().session_id().empty());

        UNIT_ASSERT_EQUAL(statistics->GetNumOfSessionActivationErrors(), 1);
        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 1);
    }

    Y_UNIT_TEST(TestActivateSessionFailsWhenEmptySessionIdInResponseTest) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());

        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();
        TSessionHolderImplPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);

        TString boxId = "boxId";
        TString workloadId = "workloadId";
        TPushContainer pushContainer = {{boxId, workloadId}, boxId, workloadId};

        THashSet<TPushContainer> pushContainers;
        pushContainers.insert(pushContainer);

        sessionHolder->Update(pushContainers);

        TPushLogResponse invalidResponse;
        invalidResponse.mutable_initialized()->set_last_seq_no(1);
        unifiedAgentModel.AddResponse(invalidResponse);

        auto activationResult = sessionHolder->ActivateSession(pushContainer);

        UNIT_ASSERT(!(bool)activationResult);
        UNIT_ASSERT_EQUAL(activationResult.Error().Errno, EPushClientError::InvalidResponse);
        UNIT_ASSERT_STRING_CONTAINS(activationResult.Error().Message, "session id is empty while stdout session init. logname: logName, sessionId: ");

        auto requests = unifiedAgentModel.GetRequests();
        UNIT_ASSERT_EQUAL(requests.size(), 1);

        auto activationRequest = requests.at(0);
        UNIT_ASSERT(activationRequest.has_initialize());
        UNIT_ASSERT(activationRequest.initialize().session_id().empty());

        UNIT_ASSERT_EQUAL(statistics->GetNumOfSessionActivationErrors(), 1);
        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 1);
    }
}

} // namespace NInfra::NPodAgent::SessionHolderTest
