#include "logs_transmitter_impl.h"

#include <infra/pod_agent/libs/push_client/push_event_impl.h>

#include <util/generic/hash_set.h>

namespace NInfra::NPodAgent {

TLogsTransmitterImpl::TLogsTransmitterImpl(
    THoldersContextPtr holdersContext
    , TPushClientPtr simpleClient
    , TLogFramePtr logFrame
    , TLogsTransmitterStatisticsPtr statistics
)
    : HoldersContext_(holdersContext)
    , SimpleClient_(simpleClient)
    , LogFrame_(logFrame)
    , Statistics_(statistics)
{}

void TLogsTransmitterImpl::TransmitLogs(const THashSet<TPushContainer>& pushContainers) {
    TPushEvents events = GetEvents(pushContainers);
    SendEvents(events);
}

TPushEvents TLogsTransmitterImpl::GetEvents(const THashSet<TPushContainer>& pushContainers) {
    TPushEvents events;
    events.reserve(pushContainers.size());

    for (const auto& pushContainer : pushContainers) {
        auto getEventResult = GetEvent(pushContainer);
        if (getEventResult) {
            events.push_back(getEventResult.Success());
        } else if (EPushClientError::EndOfFileWarning != getEventResult.Error().Errno) {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(getEventResult.Error()));
        }
    }

    return events;
}

TExpected<TPushEventConstPtr, TPushClientError> TLogsTransmitterImpl::GetEvent(const TPushContainer& pushContainer) {
    ui64 offset = OUTCOME_TRYX(HoldersContext_->GetLogsFileOffsetHolder()->GetOffset(pushContainer));
    TFileStreamRotatedPtr stream = OUTCOME_TRYX(HoldersContext_->GetFileStreamHolder()->GetFileStream(pushContainer));

    auto fileSize = OUTCOME_TRYX(stream->GetFileSize());
    if (fileSize < offset) {
        HoldersContext_->GetLogsFileOffsetHolder()->UpdateOffset(pushContainer, fileSize);
        return TPushClientError{EPushClientError::InvalidPushLogEvent, TStringBuilder() << "File size less than offset. Container: " << pushContainer.Container};
    }

    if (fileSize == offset) {
        return TPushClientError{EPushClientError::EndOfFileWarning, TStringBuilder() << "End of file porto log file reached. Container: " << pushContainer.Container};
    }

    return TPushEventConstPtr(
        new TPushEventImpl(
            stream
            , offset
            , pushContainer
        )
    );
}

void TLogsTransmitterImpl::SendEvents(const TPushEvents& pushEvents) {
    if (pushEvents.empty()) return;

    ui64 rawBytesSent = 0;
    ui64 wrappedBytesSent = 0;

    for (auto pushEvent: pushEvents) {
        const auto sendResult = SimpleClient_->Send(pushEvent);
        if (!(bool) sendResult) {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(sendResult.Error()));
        } else {
            HoldersContext_->GetLogsFileOffsetHolder()->UpdateOffset(pushEvent->GetPushContainer(), pushEvent->GetOffsetToReadFrom() + sendResult.Success().NumBytesBeforeWrapping);

            rawBytesSent += sendResult.Success().NumBytesBeforeWrapping;
            wrappedBytesSent += sendResult.Success().NumBytesAfterWrapping;
        }
    }

    Statistics_->AddNumRawBytesSent(rawBytesSent);
    Statistics_->AddNumWrappedBytesSent(wrappedBytesSent);
}

} //namespace NInfra::NPodAgent
