#include "logs_transmitter_job.h"

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/file_stream/file_stream_rotated_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/file_stream_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/logs_file_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/serializable_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_impl.h>
#include <infra/pod_agent/libs/porto_client/porto_types.h>
#include <infra/pod_agent/libs/porto_client/property.h>
#include <infra/pod_agent/libs/push_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NLogsTransmitterTest {

static TLogger logger({});

static const uint32_t defaultClearingSize = 1;
static const uint32_t defaultFileSize = 3;
static const ui64 defaultSentRawSize = 2;
static const ui64 defaultSentWrappedSize = 5;

static const THashSet<TPushContainer> pushContainers = {
    TPushContainer({TPortoContainerName({"box_1"}, "workload_1"), {"1"}, {"1"}}),
    TPushContainer({TPortoContainerName({"box_2"}, "workload_2"), {"2"}, {"2"}}),
    TPushContainer({TPortoContainerName({"box_3"}, "workload_3"), {"3"}, {"3"}})
};

struct TTestLogsFileOffsetHolder: public TMockLogsFileOffsetHolder {

    TExpected<void, TPushClientError> UpdateOffset(const TPushContainer&, ui64 offset) override {
        LastOffsetWhileUpdate = offset;
        ++UpdateOffsetCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
        ++GetOffsetCalls;
        return TExpected<ui64, TPushClientError>::DefaultSuccess();
    }

    TExpected<void, TPushClientError> DecrementOffset(const TPushContainer&, ui64) override {
        ++DecrementOffsetCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }


    mutable size_t GetOffsetCalls = 0;
    size_t UpdateOffsetCalls = 0;
    size_t DecrementOffsetCalls = 0;
    ui64 LastOffsetWhileUpdate = 0;
};

struct TTestPushClient: public TMockPushClient {

    TExpected<TSendLogResult, TPushClientError> Send(TPushEventConstPtr event) override {
        ++SendCalls;
        Event.push_back(event);
        return TSendLogResult{defaultSentRawSize, defaultSentWrappedSize};
    }

    size_t SendCalls = 0;
    TPushEvents Event;
};

struct TTestFileStreamHolder: public TMockFileStreamHolder {
    TTestFileStreamHolder(TFileStreamRotatedPtr fileStream)
        : FileStream(fileStream)
    {}

    TExpected<TFileStreamRotatedPtr, TPushClientError> GetFileStream(const TPushContainer&) const override {
        ++GetFileStreamCalls;
        return FileStream;
    }

    mutable size_t GetFileStreamCalls = 0;
    TFileStreamRotatedPtr FileStream;
};

struct TTestFileStreamRotated: public TMockFileStreamRotated {
    TExpected<ui32, TPushClientError> TryClearFile() override {
        ++TryClearCalls;
        return defaultClearingSize;
    }

    TExpected<ui64, TPushClientError> GetFileSize() const override {
        ++GetFileSizeCalls;
        return defaultFileSize;
    }

    size_t TryClearCalls = 0;
    mutable size_t GetFileSizeCalls = 0;
};

class ITestLogsTransmitterImplCanon {
public:
    ITestLogsTransmitterImplCanon(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
        : SimpleClient_(pushClient)
        , OffsetHolder_(offsetHolder)
        , FileStreamHolder_(fileStreamHolder)
        , FileStream_(fileStream)
        , Statistics_(statistics)
    {}

    virtual ~ITestLogsTransmitterImplCanon() = default;

    TLogsTransmitterPtr GetLogsTransmitterImpl() {
        return new TLogsTransmitterImpl(
            new THoldersContext(
                FileStreamHolder_
                , OffsetHolder_
                , new TMockSerializableOffsetHolder
            ),
            SimpleClient_,
            logger.SpawnFrame(),
            Statistics_
        );
    }

    void RunLogsTransmitterImpl(const THashSet<TPushContainer>& pushContainers) {
        GetLogsTransmitterImpl()->TransmitLogs(pushContainers);
    }

    void DoTest() {
        Test();
    }

protected:
    virtual void Test() = 0;

protected:
    TPushClientPtr SimpleClient_;
    TLogsFileOffsetHolderPtr OffsetHolder_;
    TFileStreamHolderPtr FileStreamHolder_;
    TFileStreamRotatedPtr FileStream_;
    TLogsTransmitterStatisticsPtr Statistics_;
};

Y_UNIT_TEST_SUITE(LogsTransmitterSuite) {

    Y_UNIT_TEST(TestNoOffsetForWorkloadInHolder) {
        struct TTestOffsetHolderFails: public TTestLogsFileOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{ EPushClientError::NoOffsetForWorkload, "No offset for workload" };
            }
        };

        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterImpl(pushContainers);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TTest test(new TTestPushClient(), new TTestOffsetHolderFails(), new TTestFileStreamHolder(fileStream), fileStream, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }

    Y_UNIT_TEST(TestNoFileStreamForWorkloadInHolder) {
        struct TTestFileStreamHolderFails: public TTestFileStreamHolder {
            TTestFileStreamHolderFails(TFileStreamRotatedPtr fileStream)
                :TTestFileStreamHolder(fileStream)
            {}

            TExpected<TFileStreamRotatedPtr, TPushClientError> GetFileStream(const TPushContainer&) const override {
                ++GetFileStreamCalls;
                return TPushClientError{ EPushClientError::NoLogsFileStreamForWorkload, "No file stream for workload" };
            }
        };

        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterImpl(pushContainers);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TTest test(new TTestPushClient(), new TTestLogsFileOffsetHolder(), new TTestFileStreamHolderFails(fileStream), fileStream, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }

    Y_UNIT_TEST(TestGetFileSizeReturnsError) {
        struct TTestFileStreamRotatedFails: public TTestFileStreamRotated {
            TExpected<ui64, TPushClientError> GetFileSize() const override {
                ++GetFileSizeCalls;
                return TPushClientError{EPushClientError::FileNotExists, "GetFileSize error"};
            }
        };

        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterImpl(pushContainers);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);
            }
        };

        TFileStreamRotatedPtr fileStreamFails = new TTestFileStreamRotatedFails;
        TTest test(new TTestPushClient(), new TTestLogsFileOffsetHolder(), new TTestFileStreamHolder(fileStreamFails), fileStreamFails, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }

    Y_UNIT_TEST(TestOffsetMoreThanFileSize) {
        struct TTestOffsetHolderReturnsOffsetMoreThanSize: public TTestLogsFileOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TTestFileStreamRotated().GetFileSize().Success() + 1;
            }
        };

        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterImpl(pushContainers);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(defaultFileSize, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->LastOffsetWhileUpdate);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(0, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TTest test(new TTestPushClient(), new TTestOffsetHolderReturnsOffsetMoreThanSize(), new TTestFileStreamHolder(fileStream), fileStream, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }

    Y_UNIT_TEST(TestSendLogsFails) {
        struct TTestPushClientFails: public TTestPushClient {
            TExpected<TSendLogResult, TPushClientError> Send(TPushEventConstPtr) override {
                ++SendCalls;
                return TPushClientError{EPushClientError::GrpcStreamBroken, ""};
            }
        };

        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterImpl(pushContainers);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestPushClient*)SimpleClient_.Get())->Event.size());
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TTest test(new TTestPushClientFails(), new TTestLogsFileOffsetHolder(), new TTestFileStreamHolder(fileStream), fileStream, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }

    Y_UNIT_TEST(TestUpdateOffsetIfSendOk) {
        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                    : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                TLogsTransmitterPtr logsTransmitterImpl = GetLogsTransmitterImpl();

                logsTransmitterImpl->TransmitLogs({TPushContainer({TPortoContainerName({"box_1"}, "workload_1"), {"1"}, {"1"}})});

                UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(defaultSentRawSize, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->LastOffsetWhileUpdate);
                UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);

                UNIT_ASSERT_EQUAL(1, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)FileStream_.Get())->TryClearCalls);

                UNIT_ASSERT_EQUAL(1, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);

                UNIT_ASSERT_EQUAL(defaultSentRawSize, Statistics_->GetNumRawBytesSent());
                UNIT_ASSERT_EQUAL(defaultSentWrappedSize, Statistics_->GetNumWrappedBytesSent());
            }
        };


        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TTest test(new TTestPushClient(), new TTestLogsFileOffsetHolder(), new TTestFileStreamHolder(fileStream), fileStream, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }

    Y_UNIT_TEST(TestTwoCallsTransmitLogsOk) {
        class TTest: public ITestLogsTransmitterImplCanon {
        public:
            TTest(TPushClientPtr pushClient, TLogsFileOffsetHolderPtr offsetHolder, TFileStreamHolderPtr fileStreamHolder, TFileStreamRotatedPtr fileStream, TLogsTransmitterStatisticsPtr statistics)
                : ITestLogsTransmitterImplCanon(pushClient, offsetHolder, fileStreamHolder, fileStream, statistics)
            {}

        protected:
            void Test() override {
                TLogsTransmitterPtr logsTransmitterImpl = GetLogsTransmitterImpl();

                logsTransmitterImpl->TransmitLogs(pushContainers);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(3, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(3, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);

                logsTransmitterImpl->TransmitLogs(pushContainers);
                UNIT_ASSERT_EQUAL(6, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->UpdateOffsetCalls);
                UNIT_ASSERT_EQUAL(6, ((TTestLogsFileOffsetHolder*)OffsetHolder_.Get())->GetOffsetCalls);

                UNIT_ASSERT_EQUAL(6, ((TTestFileStreamHolder*)FileStreamHolder_.Get())->GetFileStreamCalls);

                UNIT_ASSERT_EQUAL(6, ((TTestFileStreamRotated*)FileStream_.Get())->GetFileSizeCalls);

                UNIT_ASSERT_EQUAL(6, ((TTestPushClient*)SimpleClient_.Get())->SendCalls);
            }
        };

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        TTest test(new TTestPushClient(), new TTestLogsFileOffsetHolder(), new TTestFileStreamHolder(fileStream), fileStream, new TLogsTransmitterStatisticsImpl);
        test.DoTest();
    }
}

} //namespace NInfra::NPodAgent::NLogsTransmitterTest
