#include "logs_transmitter_job.h"
#include "logs_transmitter_porto_offsets_checker_impl.h"
#include "porto_rotated_handlers_context.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/file_deserializer.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/file_serializer.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/porto_deserializer.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/porto_serializer.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/serialize_utils/protos/container_metadata.pb.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/containers_set_builder.h>

#include <util/generic/hash_set.h>
#include <util/string/vector.h>

namespace NInfra::NPodAgent {

TLogsTransmitterJob::TLogsTransmitterJob(
    const TDuration& period
    , THoldersContextPtr stdoutHoldersContext
    , THoldersContextPtr stderrHoldersContext
    , TLogsTransmitterPtr stdoutLogsTransmitter
    , TLogsTransmitterPtr stderrLogsTransmitter
    , TLogFramePtr logFrame
    , TLogFramePtr periodWorkerEventsLogFrame
    , TPathHolderPtr pathHolder
    , THoldersUpdaterPtr holdersUpdater
    , TFilterPtr processedContainersFilter
    , TFilterPtr portoExistsContainersFilter
    , TNeedTransmitLogsDetectorPtr needTransmitLogsDetector
    , TActiveContainersGetterPtr activeContainersGetter
    , TLogsTransmitterStatisticsPrinterPtr statisticsPrinter
    , TLogsTransmitterPortoOffsetsCheckerPtr portoStdoutOffsetsChecker
    , TLogsTransmitterPortoOffsetsCheckerPtr portoStderrOffsetsChecker
    , TPortoClientPtr porto
    , TFileSystemUtilsPtr fileSystemUtils
    , ui32 jobRunsCountBetweenOffsetsSerialization
    , const THashSet<TPushContainer>& initPushContainers
)
    : IPeriodJob("LogsTransmitterJob", period, periodWorkerEventsLogFrame)
    , StdoutHoldersContext_(stdoutHoldersContext)
    , StderrHoldersContext_(stderrHoldersContext)
    , StdoutLogsTransmitter_(stdoutLogsTransmitter)
    , StderrLogsTransmitter_(stderrLogsTransmitter)
    , LogFrame_(logFrame)
    , PathHolder_(pathHolder)
    , HoldersUpdater_(holdersUpdater)
    , ProcessedContainersFilter_(processedContainersFilter)
    , PortoExistsContainersFilter_(portoExistsContainersFilter)
    , NeedTransmitLogsDetector_(needTransmitLogsDetector)
    , ActiveContainersGetter_(activeContainersGetter)
    , StatisticsPrinter_(statisticsPrinter)
    , PortoStdoutOffsetsChecker_(portoStdoutOffsetsChecker)
    , PortoStderrOffsetsChecker_(portoStderrOffsetsChecker)
    , Porto_(porto)
    , FileSystemUtils_(fileSystemUtils)
    , JobRunsCountBetweenOffsetsSerialization_(jobRunsCountBetweenOffsetsSerialization)
    , PushContainers_(initPushContainers)
{}

void TLogsTransmitterJob::Run() {
    THashSet<TPushContainer> freshPushContainers = ActiveContainersGetter_->GetActivePushContainers();

    auto newPushContainers = TContainersSetBuilder::
        Init(freshPushContainers)
        ->Minus(PushContainers_)
        ->Result();

    auto oldPushContainers = TContainersSetBuilder::
        Init(freshPushContainers)
        ->Minus(newPushContainers)
        ->Result();

    newPushContainers = PortoExistsContainersFilter_->Filter(std::move(newPushContainers));
    freshPushContainers = TContainersSetBuilder::
        Init({})
        ->Union(oldPushContainers)
        ->Union(newPushContainers)
        ->Result();

    auto currentPushContainers = TContainersSetBuilder::
        Init(PushContainers_)
        ->Minus(freshPushContainers)
        ->Filter(ProcessedContainersFilter_)
        ->Union(freshPushContainers)
        ->Result();

    auto processedPushContainers = TContainersSetBuilder::
        Init(PushContainers_)
        ->Minus(currentPushContainers)
        ->Result();

    PushContainers_ = currentPushContainers;

    if (!processedPushContainers.empty()) {
        SerializeContainersLogFilesOffsets(processedPushContainers);
    }

    HoldersUpdater_->Update(PushContainers_);

    if (!newPushContainers.empty()) {
        DeserializeContainersLogFilesOffsets(newPushContainers);
    }

    TryRotateContainersLogFiles(PushContainers_);

    if (NeedTransmitLogsDetector_->NeedTransmitLogs()) {
        StdoutLogsTransmitter_->TransmitLogs(PushContainers_);
        StderrLogsTransmitter_->TransmitLogs(PushContainers_);
    }

    if (JobRunsCountBetweenOffsetsSerialization_ > 0) {
        ++RunsAfterLastOffsetSerialization_;

        if (RunsAfterLastOffsetSerialization_ == JobRunsCountBetweenOffsetsSerialization_) {
            RunsAfterLastOffsetSerialization_ = 0;
            if (!PushContainers_.empty()) {
                SerializeContainersLogFilesOffsets(PushContainers_);
            }
        }
    }

    PortoStdoutOffsetsChecker_->Check(PushContainers_);
    PortoStderrOffsetsChecker_->Check(PushContainers_);

    StatisticsPrinter_->Print();
}

void TLogsTransmitterJob::TryRotateContainersLogFiles(const THashSet<TPushContainer>& pushContainers) {
    for (const auto& container : pushContainers) {
        TryRotateContainerLogFiles(StdoutHoldersContext_, container);
        TryRotateContainerLogFiles(StderrHoldersContext_, container);
    }
}

void TLogsTransmitterJob::TryRotateContainerLogFiles(THoldersContextPtr holdersContext, const TPushContainer& pushContainer) {
    auto tryRotateResult = holdersContext->GetFileStreamHolder()->TryRotate(pushContainer);
    if (!tryRotateResult) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(tryRotateResult.Error()));
    } else {
        ui64 rotateSize = tryRotateResult.Success();

        if (!rotateSize) {
            return;
        }

        auto decrementResult = holdersContext->GetLogsFileOffsetHolder()->DecrementOffset(pushContainer, rotateSize);
        if (!decrementResult) {
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(decrementResult.Error()));
        }
    }
}

void TLogsTransmitterJob::Shutdown() {
    StdoutHoldersContext_->GetPortoOffsetHolder()->SerializeAll(new TPortoSerializer(Porto_, EPortoContainerProperty::CachedStdoutPortoOffset, LogFrame_));
    StderrHoldersContext_->GetPortoOffsetHolder()->SerializeAll(new TPortoSerializer(Porto_, EPortoContainerProperty::CachedStderrPortoOffset, LogFrame_));

    std::function<TContainerMetadata(TContainerMetadata, ui64)> setFileOffsetToMetadata =
        [](TContainerMetadata meta, ui64 value) {
            meta.set_file_offset(value);
            return meta;
        };

    StdoutHoldersContext_->GetLogsFileOffsetHolder()->SerializeAll(new TFileSerializer(PathHolder_, ELogType::Stdout, LogFrame_, FileSystemUtils_, setFileOffsetToMetadata));
    StderrHoldersContext_->GetLogsFileOffsetHolder()->SerializeAll(new TFileSerializer(PathHolder_, ELogType::Stderr, LogFrame_, FileSystemUtils_, setFileOffsetToMetadata));
}

void TLogsTransmitterJob::Initialize() {
    THashSet<TPushContainer> pushContainers = ActiveContainersGetter_->GetActivePushContainers();
    HoldersUpdater_->Update(pushContainers);

    StdoutHoldersContext_->GetPortoOffsetHolder()->Deserialize(pushContainers, new TPortoDeserializer(Porto_, EPortoContainerProperty::CachedStdoutPortoOffset, LogFrame_));
    StderrHoldersContext_->GetPortoOffsetHolder()->Deserialize(pushContainers, new TPortoDeserializer(Porto_, EPortoContainerProperty::CachedStderrPortoOffset, LogFrame_));

    DeserializeContainersLogFilesOffsets(pushContainers);

    PushContainers_ = pushContainers;

    auto portoRotatedDataOnStartCheckHandler = TPortoRotatedHandlersContext::GetPortoRotatedOnStartHandler();

    ui32 nullCheckPeriodSec = 0;

    //these checkers run to detect case when porto rotated logs files while pod_agent restarting
    TLogsTransmitterPortoOffsetsCheckerImpl(Porto_, StdoutHoldersContext_, LogFrame_, nullCheckPeriodSec, EPortoContainerProperty::StdoutOffset, EPortoContainerProperty::StdOutPath, portoRotatedDataOnStartCheckHandler)
        .Check(pushContainers);
    TLogsTransmitterPortoOffsetsCheckerImpl(Porto_, StderrHoldersContext_, LogFrame_, nullCheckPeriodSec, EPortoContainerProperty::StderrOffset, EPortoContainerProperty::StdErrPath, portoRotatedDataOnStartCheckHandler)
        .Check(pushContainers);
}

void TLogsTransmitterJob::SerializeContainersLogFilesOffsets(const THashSet<TPushContainer>& pushContainers) {
    std::function<TContainerMetadata(TContainerMetadata, ui64)> setFileOffsetToMetadata =
        [](TContainerMetadata meta, ui64 value) {
            meta.set_file_offset(value);
            return meta;
        };

    StdoutHoldersContext_->GetLogsFileOffsetHolder()->Serialize(pushContainers, new TFileSerializer(PathHolder_, ELogType::Stdout, LogFrame_, FileSystemUtils_, setFileOffsetToMetadata));
    StderrHoldersContext_->GetLogsFileOffsetHolder()->Serialize(pushContainers, new TFileSerializer(PathHolder_, ELogType::Stderr, LogFrame_, FileSystemUtils_, setFileOffsetToMetadata));
}

void TLogsTransmitterJob::DeserializeContainersLogFilesOffsets(const THashSet<TPushContainer>& pushContainers) {
    std::function<ui64(TContainerMetadata)> getFileOffsetFromMetadata =
        [](TContainerMetadata meta) {
            return meta.file_offset();
        };

    StdoutHoldersContext_->GetLogsFileOffsetHolder()->Deserialize(pushContainers, new TFileDeserializer(PathHolder_, ELogType::Stdout, LogFrame_, FileSystemUtils_, getFileOffsetFromMetadata));
    StderrHoldersContext_->GetLogsFileOffsetHolder()->Deserialize(pushContainers, new TFileDeserializer(PathHolder_, ELogType::Stderr, LogFrame_, FileSystemUtils_, getFileOffsetFromMetadata));
}

} // namespace NInfra::NPodAgent
