#include "logs_transmitter_job_factory.h"
#include "logs_transmitter_job.h"
#include "logs_transmitter_porto_offsets_checker_impl.h"
#include "porto_rotated_handlers_context.h"

#include <contrib/libs/grpc/include/grpc++/create_channel.h>

#include <infra/libs/logger/log_frame.h>

#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/serializable_offset_holder_impl.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/porto_offset_holder_impl.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/filter/porto_exists_containers_filter.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/filter/processed_containers_filter.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_impl.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_printer_impl.h>
#include <infra/pod_agent/libs/porto_client/property.h>
#include <infra/pod_agent/libs/push_client/push_log_splitter.h>
#include <infra/pod_agent/libs/push_client/sessions/push_session_factory_impl.h>
#include <infra/pod_agent/libs/push_client/simple_client.h>

#include <util/system/env.h>

namespace NInfra::NPodAgent {

TLogsTransmitterJobFactory::TLogsTransmitterJobFactory(
    const TConfig& config
    , TLogFramePtr logFrame
    , TLogFramePtr jobPeriodWorkerEventsLogFrame
    , TPathHolderPtr pathHolder
    , const TString& hostName
    , TStatusRepositoryPtr statusRepository
    , TStatusNTickerHolderPtr statusNTickerHolder
    , TPortoClientPtr portoClient
    , const bool isBoxAgentMode
)
    : Config_(config)
    , LogFrame_(logFrame)
    , JobPeriodWorkerEventsLogFrame_(jobPeriodWorkerEventsLogFrame)
    , PathHolder_(pathHolder)
    , HostName_(hostName)
    , StatusRepository_(statusRepository)
    , StatusNTickerHolder_(statusNTickerHolder)
    , PortoClient_(portoClient)
    , IsBoxAgentMode_(isBoxAgentMode)
{}

TLogsTransmitterJobPtr TLogsTransmitterJobFactory::Create() {
    ui32 maxLogsFileSize = Config_.GetLogsTransmitter().GetMaxLogsFileSize();
    double minPortionToCut = Config_.GetLogsTransmitter().GetMinPortionToCut();

    const TString logName = Config_.GetLogsTransmitter().GetLogName();

    TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

    TPortoOffsetHolderPtr portoStdoutOffsetHolder = new TPortoOffsetHolderImpl(LogFrame_, statistics, new TSerializableOffsetHolder(LogFrame_));
    TPortoOffsetHolderPtr portoStderrOffsetHolder = new TPortoOffsetHolderImpl(LogFrame_, statistics, new TSerializableOffsetHolder(LogFrame_));

    auto stdoutSessionFactory = CreatePushSessionFactory(ELogType::Stdout);
    auto stderrSessionFactory = CreatePushSessionFactory(ELogType::Stderr);

    TSessionHolderPtr stdoutSessionHolder = new TSessionHolderImpl(logName, ELogType::Stdout, stdoutSessionFactory, statistics);
    TSessionHolderPtr stderrSessionHolder = new TSessionHolderImpl(logName, ELogType::Stderr, stderrSessionFactory, statistics);

    TLogsFileOffsetHolderPtr stdoutOffsetHolder = new TLogsFileOffsetHolder(LogFrame_, statistics, new TSerializableOffsetHolder(LogFrame_));
    TLogsFileOffsetHolderPtr stderrOffsetHolder = new TLogsFileOffsetHolder(LogFrame_, statistics, new TSerializableOffsetHolder(LogFrame_));

    TFileSystemUtilsPtr fileSystemUtils = new TFileSystemUtils();

    blksize_t fileSystemBlockSize = fileSystemUtils->GetFileSystemBlockSize().Success();
    TFileStreamRotatedFactoryPtr fileStreamFactory = new TFileStreamRotatedFactory(maxLogsFileSize, minPortionToCut, fileSystemBlockSize, fileSystemUtils);

    TFileStreamHolderPtr stdoutFileStreamHolder = new TFileStreamHolderImpl(maxLogsFileSize, minPortionToCut, fileSystemBlockSize, ELogType::Stdout, PathHolder_, fileStreamFactory, IsBoxAgentMode_);
    TFileStreamHolderPtr stderrFileStreamHolder = new TFileStreamHolderImpl(maxLogsFileSize, minPortionToCut, fileSystemBlockSize, ELogType::Stderr, PathHolder_, fileStreamFactory, IsBoxAgentMode_);

    THoldersContextPtr stdoutHoldersContext = new THoldersContext(stdoutFileStreamHolder, stdoutOffsetHolder, portoStdoutOffsetHolder);
    THoldersContextPtr stderrHoldersContext = new THoldersContext(stderrFileStreamHolder, stderrOffsetHolder, portoStderrOffsetHolder);
    THoldersUpdaterPtr holdersUpdater = new THoldersUpdater();

    holdersUpdater->AddHolder(stdoutFileStreamHolder);
    holdersUpdater->AddHolder(stderrFileStreamHolder);
    holdersUpdater->AddHolder(stdoutOffsetHolder);
    holdersUpdater->AddHolder(stderrOffsetHolder);
    holdersUpdater->AddHolder(stdoutSessionHolder);
    holdersUpdater->AddHolder(stderrSessionHolder);
    holdersUpdater->AddHolder(portoStdoutOffsetHolder);
    holdersUpdater->AddHolder(portoStderrOffsetHolder);


    TFilterPtr processedContainersFilter = new TProcessedContainersFilter(stdoutHoldersContext, stderrHoldersContext, LogFrame_);
    TFilterPtr portoExistsContainersFilter = new TPortoExistsContainersFilter(PortoClient_, LogFrame_);

    const ui32 numOfThreads = Config_.GetLogsTransmitter().GetNumOfTransmitThreads();

    auto maxFormattedLogsSize = Config_.GetLogsTransmitter().GetMaxFormattedLogSize();
    auto maxFormattedRawLogsSize = Config_.GetLogsTransmitter().GetMaxFormattedRawLogSize();

    TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogsSize, maxFormattedRawLogsSize);

    TPushClientPtr simplePushClientStdout = new TSimplePushClient(logsFormatter, stdoutSessionHolder, Config_.GetLogsTransmitter().GetMaxStdoutLogPerOneIteration(), statistics);

    TPushClientPtr simplePushClientStderr = new TSimplePushClient(logsFormatter, stderrSessionHolder, Config_.GetLogsTransmitter().GetMaxStderrLogPerOneIteration(), statistics);

    TLogsTransmitterPtr stdoutLogsTransmitter = new TLogsTransmitterImpl(
        stdoutHoldersContext
        , simplePushClientStdout
        , LogFrame_
        , statistics
    );

    TLogsTransmitterPtr stderrLogsTransmitter = new TLogsTransmitterImpl(
        stderrHoldersContext
        , simplePushClientStderr
        , LogFrame_
        , statistics
    );

    MtpQueue_ = new TThreadPool();
    MtpQueue_->Start(numOfThreads * NUM_OF_LOG_TYPES);

    TLogsTransmitterMultithreadPtr stdoutLogsTransmitterMultithread = new TLogsTransmitterMultithread(numOfThreads, stdoutLogsTransmitter, MtpQueue_, LogFrame_);
    TLogsTransmitterMultithreadPtr stderrLogsTransmitterMultithread = new TLogsTransmitterMultithread(numOfThreads, stderrLogsTransmitter, MtpQueue_, LogFrame_);

    TLogsTransmitterStatisticsPrinterPtr statisticsPrinter = new TLogsTransmitterStatisticsPrinterImpl(statistics, stdoutSessionHolder, stderrSessionHolder, stdoutFileStreamHolder, stderrFileStreamHolder, LogFrame_, STATISTICS_PRINTER_INVOKATION_PERIOD_SEC);

    TActiveContainersGetterPtr activeContainersGetter = new TActiveContainersGetter(StatusNTickerHolder_->GetWorkloadStatusRepository(), PathHolder_, LogFrame_, IsBoxAgentMode_);
    TNeedTransmitLogsDetectorPtr needTransmitLogsDetector = new TNeedTransmitLogsDetector(StatusRepository_);

    auto portoRotatedDataOnPeriodicCheckHandler = TPortoRotatedHandlersContext::GetPortoRotatedOnPeriodicCheckHandler();

    TLogsTransmitterPortoOffsetsCheckerPtr portoStdoutOffsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(PortoClient_, stdoutHoldersContext, LogFrame_, PORTO_OFFSET_CHECKER_INVOKATION_PERIOD_SEC, EPortoContainerProperty::StdoutOffset, EPortoContainerProperty::StdOutPath, portoRotatedDataOnPeriodicCheckHandler);

    TLogsTransmitterPortoOffsetsCheckerPtr portoStderrOffsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(PortoClient_, stderrHoldersContext, LogFrame_, PORTO_OFFSET_CHECKER_INVOKATION_PERIOD_SEC, EPortoContainerProperty::StderrOffset, EPortoContainerProperty::StdErrPath, portoRotatedDataOnPeriodicCheckHandler);

    ui32 periodMs = Config_.GetLogsTransmitter().GetPeriodMs();
    ui32 offsetSerializationPeriodMs = Config_.GetLogsTransmitter().GetLogsOffsetsSerializationPeriodMs();
    if (offsetSerializationPeriodMs > 0) {
        Y_ENSURE(offsetSerializationPeriodMs > MIN_LOGS_TRANSMITTER_JOB_RUNS_BETWEEN_OFFSETS_SERIALIZATIONS * periodMs,
            "Offsets serialization period must be at least "
                << MIN_LOGS_TRANSMITTER_JOB_RUNS_BETWEEN_OFFSETS_SERIALIZATIONS
                << " times more than logs transmitter job period"
        );
    }

    ui32 jobRunsCountBetweenOffsetsSerialization = offsetSerializationPeriodMs / periodMs;

    TLogsTransmitterJobPtr job = new TLogsTransmitterJob(
        TDuration::MilliSeconds(periodMs)
        , stdoutHoldersContext
        , stderrHoldersContext
        , stdoutLogsTransmitterMultithread
        , stderrLogsTransmitterMultithread
        , LogFrame_
        , JobPeriodWorkerEventsLogFrame_
        , PathHolder_
        , holdersUpdater
        , processedContainersFilter
        , portoExistsContainersFilter
        , needTransmitLogsDetector
        , activeContainersGetter
        , statisticsPrinter
        , portoStdoutOffsetChecker
        , portoStderrOffsetChecker
        , PortoClient_
        , fileSystemUtils
        , jobRunsCountBetweenOffsetsSerialization
    );

    job->Initialize();
    return job;
}

TPushSessionFactoryPtr TLogsTransmitterJobFactory::CreatePushSessionFactory(const ELogType& logType) {
    const auto unifiedAgentNewProtocolPort = Config_.GetLogsTransmitter().GetUnifiedAgentNewProtocolPort();
    TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << unifiedAgentNewProtocolPort, grpc::InsecureChannelCredentials());
    const auto unifiedAgentStaticSecret = Config_.GetLogsTransmitter().GetUnifiedAgentStaticSecret();
    return new TPushSessionFactoryImpl(logType, grpcChannel, unifiedAgentStaticSecret);
}

} // namespace NInfra::NPodAgent
