#include "logs_transmitter_job.h"
#include "logs_transmitter_mock.h"
#include "logs_transmitter_porto_offsets_checker_mock.h"

#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/file_stream_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/holders_updater_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/logs_file_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/serializable_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_printer_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/active_containers_getter_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/file_system_utils_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/need_transmit_logs_detector_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/filter/filter_mock.h>
#include <infra/pod_agent/libs/pod_agent/period_job_worker/period_job_worker.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/push_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NLogsTransmitterCollectorJobTest {

static TLogger logger({});
static TString defaultBox("box_box1");
static TPushContainer pushContainer1 = TPushContainer({TPortoContainerName({"box_box1"}, "workload_workload1"), {"box1"}, {"workload1"}});
static TPushContainer pushContainer2 = TPushContainer({TPortoContainerName({"box_box1"}, "workload_workload2"), {"box1"}, {"workload2"}});
static TPushContainer pushContainer3 = TPushContainer({TPortoContainerName({"box_box1"}, "workload_workload3"), {"box1"}, {"workload3"}});
static const TVector<TPushContainer> pushContainers = {pushContainer1, pushContainer2};
static const ui32 defaultRotatedSize = 1;
static const TDuration defaultPeriod = TDuration::MilliSeconds(100);
static const ui32 defaultJobRunsCountBetweenOffsetsSerialization = 5;

struct TTestPortoClient: public TMockPortoClient {
    TExpected<TString, TPortoError> GetProperty(const TPortoContainerName&, EPortoContainerProperty, int) override {
        return TString("0");
    }

};

struct TTestLogsTransmitter : public TMockLogstransmitter {
    void TransmitLogs(const THashSet<TPushContainer>& pushContainers) override {
        ++TransmitCalls;
        SizeOfPushContainers = pushContainers.size();
    }

    size_t TransmitCalls = 0;
    size_t SizeOfPushContainers = 0;
};

struct TTestLogsFileOffsetHolder: public TMockLogsFileOffsetHolder {
    TExpected<void, TPushClientError> DecrementOffset(const TPushContainer&, ui64) override {
        ++DecrementOffsetCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    void Serialize(const THashSet<TPushContainer>& containersToSerialize, TSerializerPtr) const override {
        ++SerializeOffsetCalls;
        SerializedContainers_ = containersToSerialize;
    }

    void Deserialize(const THashSet<TPushContainer>& containersToDeserialize, TDeserializerPtr) override {
        ++DeserializeOffsetCalls;
        DeserializedContainers_ = containersToDeserialize;
    }

    size_t DecrementOffsetCalls = 0;
    mutable size_t SerializeOffsetCalls = 0;
    size_t DeserializeOffsetCalls = 0;
    mutable THashSet<TPushContainer> SerializedContainers_;
    THashSet<TPushContainer> DeserializedContainers_;
};

struct TTestFileStreamHolder: public TMockFileStreamHolder {
    TExpected<ui64, TPushClientError> TryRotate(const TPushContainer&) override {
        ++TryRotateCalls;
        return defaultRotatedSize;
    }

    size_t TryRotateCalls = 0;
};

struct TTestHoldersUpdater: public TMockHoldersUpdater {
    void Update(const THashSet<TPushContainer>&) override {
        ++UpdateHoldersContextCalls;
    }

    size_t UpdateHoldersContextCalls = 0;
};

struct TTestFilter: public TMockFilter {
    THashSet<TPushContainer> Filter(THashSet<TPushContainer>&&) override {
        return {pushContainers.begin(), pushContainers.end()};
    }
};

struct TTestActiveContainersGetter: public TMockActiveContainersGetter {
    THashSet<TPushContainer> GetActivePushContainers() override {
        return {pushContainers.begin(), pushContainers.end()};
    }
};

struct TTestPortoOffsetHolder:  public TMockSerializableOffsetHolder {
    void SerializeAll(TSerializerPtr) const override {
        ++SerializeCalls;
    }

    void Deserialize(const THashSet<TPushContainer>&, TDeserializerPtr) override {
        ++DeserializeCalls;
    }

    mutable size_t SerializeCalls = 0;
    size_t DeserializeCalls = 0;
};

class ITestLogsTransmitterJobCanon {
public:
    ITestLogsTransmitterJobCanon(
        TFileStreamHolderPtr stdoutFileStreamHolder
        , TFileStreamHolderPtr stderrFileStreamHolder
        , TNeedTransmitLogsDetectorPtr needTransmitLogsDetector
        , TActiveContainersGetterPtr activeContainerGetter
        , TLogsTransmitterStatisticsPrinterPtr statisticsPrinter
        , TFilterPtr processedContainersFilter = new TTestFilter
        , TFilterPtr portoExistsContainersFilter = new TTestFilter
        , const THashSet<TPushContainer>& initPushContainers = {}
        , ui32 jobRunsCountBetweenOffsetsSerialization = defaultJobRunsCountBetweenOffsetsSerialization
    )
        : StdoutHolderContext_(
            new THoldersContext(
                stdoutFileStreamHolder
                , new TTestLogsFileOffsetHolder
                , new TTestPortoOffsetHolder
            )
        )
        , StderrHolderContext_(
            new THoldersContext(
                stderrFileStreamHolder
                , new TTestLogsFileOffsetHolder
                , new TTestPortoOffsetHolder
            )
        )
        , StdoutLogsTransmitter_(new TTestLogsTransmitter())
        , StderrLogsTransmitter_(new TTestLogsTransmitter())
        , PathHolder_(
            new TPathHolder(
                ""
                , {{"", ""}}
                , {{"", ""}}
                , ""
                , ""
                , ""
                , ""
                , ""
                , ""
            )
        )
        , HoldersUpdater_(new TTestHoldersUpdater)
        , NeedTransmitLogsDetector_(needTransmitLogsDetector)
        , ActiveContainerGetter_(activeContainerGetter)
        , StatisticsPrinter_(statisticsPrinter)
        , PortoStdoutOffsetsChecker_(new TLogsTransmitterPortoOffsetsCheckerMock)
        , PortoStderrOffsetsChecker_(new TLogsTransmitterPortoOffsetsCheckerMock)
        , ProcessedContainersFilter_(processedContainersFilter)
        , PortoExistsContainersFilter_(portoExistsContainersFilter)
        , InitPushContainers_(initPushContainers)
        , JobRunsCountBetweenOffsetsSerialization_(jobRunsCountBetweenOffsetsSerialization)
    {
    }

    virtual ~ITestLogsTransmitterJobCanon() = default;

    TLogsTransmitterJobPtr GetLogsTransmitterJob() {
        return new TLogsTransmitterJob(
            defaultPeriod
            , StdoutHolderContext_
            , StderrHolderContext_
            , StdoutLogsTransmitter_
            , StderrLogsTransmitter_
            , logger.SpawnFrame()
            , logger.SpawnFrame()
            , PathHolder_
            , HoldersUpdater_
            , ProcessedContainersFilter_
            , PortoExistsContainersFilter_
            , NeedTransmitLogsDetector_
            , ActiveContainerGetter_
            , StatisticsPrinter_
            , PortoStdoutOffsetsChecker_
            , PortoStderrOffsetsChecker_
            , new TTestPortoClient
            , new TMockFileSystemUtils
            , JobRunsCountBetweenOffsetsSerialization_
            , InitPushContainers_
        );
    }

    void RunLogsTransmitterJob() {
        TString boxPath = PathHolder_->GetBoxRootfsPath(defaultBox);
        NFs::MakeDirectory(boxPath);
        TFile(TStringBuilder() << boxPath << "/" << pushContainers[0].WorkloadId << "_stdout.portolog", OpenAlways);
        TFile(TStringBuilder() << boxPath << "/" << pushContainers[0].WorkloadId << "_stderr.portolog", OpenAlways);
        TFile(TStringBuilder() << boxPath << "/" << pushContainers[1].WorkloadId << "_stdout.portolog", OpenAlways);
        TFile(TStringBuilder() << boxPath << "/" << pushContainers[1].WorkloadId << "_stderr.portolog", OpenAlways);
        GetLogsTransmitterJob()->Run();
        NFs::RemoveRecursive(boxPath);
    }

    void DoTest() {
        Test();
    }

protected:
    virtual void Test() = 0;

protected:
    THoldersContextPtr StdoutHolderContext_;
    THoldersContextPtr StderrHolderContext_;
    TLogsTransmitterPtr StdoutLogsTransmitter_;
    TLogsTransmitterPtr StderrLogsTransmitter_;
    TPathHolderPtr PathHolder_;
    THoldersUpdaterPtr HoldersUpdater_;
    TNeedTransmitLogsDetectorPtr NeedTransmitLogsDetector_;
    TActiveContainersGetterPtr ActiveContainerGetter_;
    TLogsTransmitterStatisticsPrinterPtr StatisticsPrinter_;
    TLogsTransmitterPortoOffsetsCheckerPtr PortoStdoutOffsetsChecker_;
    TLogsTransmitterPortoOffsetsCheckerPtr PortoStderrOffsetsChecker_;
    TFilterPtr ProcessedContainersFilter_;
    TFilterPtr PortoExistsContainersFilter_;
    THashSet<TPushContainer> InitPushContainers_;
    ui32 JobRunsCountBetweenOffsetsSerialization_;
};

Y_UNIT_TEST_SUITE(LogsTransmitterJobSuite) {

    Y_UNIT_TEST(TestNoRun) {
        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter)
            {}

        protected:
            void Test() override {
                TLogsTransmitterJobPtr gcJob = GetLogsTransmitterJob();
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock);
        test.DoTest();
    }

    Y_UNIT_TEST(TestNoTransmitLogsWhenNeedLogsTransmitReturnsFalse) {

        struct TTestNeedTransmitLogsDetector: public TMockNeedTransmitLogsDetector {
            bool NeedTransmitLogs() const override {
                ++NeedTransmitLogsCalls;
                return false;
            }
            mutable size_t NeedTransmitLogsCalls = 0;
        };

        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter)
            {}
        protected:
            void Test() override {
                RunLogsTransmitterJob();

                UNIT_ASSERT_EQUAL(1, ((TTestNeedTransmitLogsDetector*)NeedTransmitLogsDetector_.Get())->NeedTransmitLogsCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsTransmitter*)StdoutLogsTransmitter_.Get())->TransmitCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsTransmitter*)StdoutLogsTransmitter_.Get())->SizeOfPushContainers);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsTransmitter*)StderrLogsTransmitter_.Get())->TransmitCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsTransmitter*)StderrLogsTransmitter_.Get())->SizeOfPushContainers);

                //decrement offset is called while rotation
                UNIT_ASSERT_EQUAL(2, ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->DecrementOffsetCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(2, ((TTestFileStreamHolder*)StdoutHolderContext_->GetFileStreamHolder().Get())->TryRotateCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestFileStreamHolder*)StderrHolderContext_->GetFileStreamHolder().Get())->TryRotateCalls);
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TTestNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock);
        test.DoTest();
    }

    Y_UNIT_TEST(TestTryRotateReturnsError) {

        struct TTestFileStreamHolderFails: public TTestFileStreamHolder {
            TExpected<ui64, TPushClientError> TryRotate(const TPushContainer&) override {
                ++TryRotateCalls;
                return TPushClientError{EPushClientError::Unspecified, "Try rotate fails"};
            }
        };

        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter)
            {}
        protected:
            void Test() override {
                RunLogsTransmitterJob();
                UNIT_ASSERT_EQUAL(1, ((TTestLogsTransmitter*)StdoutLogsTransmitter_.Get())->TransmitCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestLogsTransmitter*)StdoutLogsTransmitter_.Get())->SizeOfPushContainers);
                UNIT_ASSERT_EQUAL(1, ((TTestLogsTransmitter*)StderrLogsTransmitter_.Get())->TransmitCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestLogsTransmitter*)StderrLogsTransmitter_.Get())->SizeOfPushContainers);

                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->DecrementOffsetCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(2, ((TTestFileStreamHolder*)StdoutHolderContext_->GetFileStreamHolder().Get())->TryRotateCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestFileStreamHolder*)StderrHolderContext_->GetFileStreamHolder().Get())->TryRotateCalls);
            }
        };

        TTest test(new TTestFileStreamHolderFails, new TTestFileStreamHolderFails, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock);
        test.DoTest();
    }

    Y_UNIT_TEST(TestTryRotateReturnsZero) {

        struct TTestFileStreamHolderZero: public TTestFileStreamHolder {
            TExpected<ui64, TPushClientError> TryRotate(const TPushContainer&) override {
                ++TryRotateCalls;
                return 0;
            }
        };

        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterJob();
                UNIT_ASSERT_EQUAL(1, ((TTestLogsTransmitter*)StdoutLogsTransmitter_.Get())->TransmitCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestLogsTransmitter*)StdoutLogsTransmitter_.Get())->SizeOfPushContainers);
                UNIT_ASSERT_EQUAL(1, ((TTestLogsTransmitter*)StderrLogsTransmitter_.Get())->TransmitCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestLogsTransmitter*)StderrLogsTransmitter_.Get())->SizeOfPushContainers);

                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->DecrementOffsetCalls);
                UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->DecrementOffsetCalls);

                UNIT_ASSERT_EQUAL(2, ((TTestFileStreamHolder*)StdoutHolderContext_->GetFileStreamHolder().Get())->TryRotateCalls);
                UNIT_ASSERT_EQUAL(2, ((TTestFileStreamHolder*)StderrHolderContext_->GetFileStreamHolder().Get())->TryRotateCalls);
            }
        };

        TTest test(new TTestFileStreamHolderZero, new TTestFileStreamHolderZero, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock);
        test.DoTest();
    }

    Y_UNIT_TEST(TestSerializeOffsetsForProcessedContainersAndDeserializeOffsetsForNewContainers) {

        //emulate situation when pushContainer1 is not exists anymnore and processed,
        //pushContainer3 - new appeared container

        THashSet<TPushContainer> initContainers = {pushContainer1, pushContainer2};

        struct TTestProcessedContainersFilter: public TMockFilter {
            THashSet<TPushContainer> Filter(THashSet<TPushContainer>&&) override {
                //pushContainer1 processed
                return {pushContainer2};
            }
        };

        struct TTestPortoExistsFilter: public TMockFilter {
            THashSet<TPushContainer> Filter(THashSet<TPushContainer>&&) override {
                return {pushContainer3};
            }
        };

        struct TTestActiveContainersGetter: public TMockActiveContainersGetter {
            THashSet<TPushContainer> GetActivePushContainers() override {
                return {pushContainer2, pushContainer3};
            }
        };

        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter, TFilterPtr processedContainersFilter, TFilterPtr portoExistsFilter, const THashSet<TPushContainer>& initContainers)
                    : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter, processedContainersFilter, portoExistsFilter, initContainers)
            {}

        protected:
            void Test() override {
                RunLogsTransmitterJob();

                UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->SerializeOffsetCalls);
                UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->SerializeOffsetCalls);

                UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->DeserializeOffsetCalls);
                UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->DeserializeOffsetCalls);

                auto serializedStdoutContainers = ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->SerializedContainers_;
                auto serializedStderrContainers = ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->SerializedContainers_;

                UNIT_ASSERT_EQUAL(1, serializedStdoutContainers.size());
                UNIT_ASSERT_EQUAL(true, serializedStdoutContainers.contains(pushContainer1));
                UNIT_ASSERT_EQUAL(1, serializedStderrContainers.size());
                UNIT_ASSERT_EQUAL(true, serializedStderrContainers.contains(pushContainer1));

                auto deserializedStdoutContainers = ((TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get())->DeserializedContainers_;
                auto deserializedStderrContainers = ((TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get())->DeserializedContainers_;

                UNIT_ASSERT_EQUAL(1, deserializedStdoutContainers.size());
                UNIT_ASSERT_EQUAL(true, deserializedStdoutContainers.contains(pushContainer3));
                UNIT_ASSERT_EQUAL(1, deserializedStderrContainers.size());
                UNIT_ASSERT_EQUAL(true, deserializedStderrContainers.contains(pushContainer3));
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock, new TTestProcessedContainersFilter, new TTestPortoExistsFilter, initContainers);
        test.DoTest();
    }

    Y_UNIT_TEST(TestPeriodicallyOffsetsSerialization) {

        THashSet<TPushContainer> initContainers = {pushContainer1, pushContainer2};

        struct TTestPortoExistsFilter: public TMockFilter {
            THashSet<TPushContainer> Filter(THashSet<TPushContainer>&&) override {
                return {pushContainer1, pushContainer2};
            }
        };

        struct TTestActiveContainersGetter: public TMockActiveContainersGetter {
            THashSet<TPushContainer> GetActivePushContainers() override {
                return {pushContainer1, pushContainer2};
            }
        };

        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter, TFilterPtr processedContainersFilter, TFilterPtr portoExistsFilter, const THashSet<TPushContainer>& initContainers)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter, processedContainersFilter, portoExistsFilter, initContainers)
            {}

        protected:
            void Test() override {
                auto logsTransmitterJob = GetLogsTransmitterJob();
                for (ui32 bigLoop = 1; bigLoop < 3; ++bigLoop) {
                    auto* stdoutTestOffsetHolder = (TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get();
                    auto* stderrTestOffsetHolder = (TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get();

                    for (ui32 smallLoop = 1; smallLoop < JobRunsCountBetweenOffsetsSerialization_; ++smallLoop) {
                        logsTransmitterJob->Run();
                        UNIT_ASSERT_EQUAL(0, stdoutTestOffsetHolder->SerializeOffsetCalls);
                        UNIT_ASSERT_EQUAL(0, stderrTestOffsetHolder->SerializeOffsetCalls);

                        auto serializedStdoutContainers = stdoutTestOffsetHolder->SerializedContainers_;
                        auto serializedStderrContainers = stderrTestOffsetHolder->SerializedContainers_;

                        UNIT_ASSERT(serializedStdoutContainers.empty());
                        UNIT_ASSERT(serializedStderrContainers.empty());
                    }

                    logsTransmitterJob->Run();
                    UNIT_ASSERT_EQUAL(1, stdoutTestOffsetHolder->SerializeOffsetCalls);
                    UNIT_ASSERT_EQUAL(1, stderrTestOffsetHolder->SerializeOffsetCalls);

                    auto serializedStdoutContainers = stdoutTestOffsetHolder->SerializedContainers_;
                    auto serializedStderrContainers = stderrTestOffsetHolder->SerializedContainers_;

                    UNIT_ASSERT_EQUAL(2, serializedStdoutContainers.size());
                    UNIT_ASSERT_EQUAL(true, serializedStdoutContainers.contains(pushContainer1));
                    UNIT_ASSERT_EQUAL(true, serializedStdoutContainers.contains(pushContainer2));
                    UNIT_ASSERT_EQUAL(2, serializedStderrContainers.size());
                    UNIT_ASSERT_EQUAL(true, serializedStderrContainers.contains(pushContainer1));
                    UNIT_ASSERT_EQUAL(true, serializedStderrContainers.contains(pushContainer2));

                    stdoutTestOffsetHolder->SerializedContainers_ = {};
                    stdoutTestOffsetHolder->SerializeOffsetCalls = 0;
                    stderrTestOffsetHolder->SerializedContainers_ = {};
                    stderrTestOffsetHolder->SerializeOffsetCalls = 0;
                }
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock, new TTestFilter, new TTestPortoExistsFilter, initContainers);
        test.DoTest();
    }

    Y_UNIT_TEST(TestPeriodicallyOffsetsSerializationDisabled) {

        THashSet<TPushContainer> initContainers = {pushContainer1, pushContainer2};

        struct TTestPortoExistsFilter: public TMockFilter {
            THashSet<TPushContainer> Filter(THashSet<TPushContainer>&&) override {
                return {pushContainer1, pushContainer2};
            }
        };

        struct TTestActiveContainersGetter: public TMockActiveContainersGetter {
            THashSet<TPushContainer> GetActivePushContainers() override {
                return {pushContainer1, pushContainer2};
            }
        };

        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter, TFilterPtr processedContainersFilter, TFilterPtr portoExistsFilter, const THashSet<TPushContainer>& initContainers)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter, processedContainersFilter, portoExistsFilter, initContainers, 0)
            {}

        protected:
            void Test() override {
                auto logsTransmitterJob = GetLogsTransmitterJob();
                auto* stdoutTestOffsetHolder = (TTestLogsFileOffsetHolder*) StdoutHolderContext_->GetLogsFileOffsetHolder().Get();
                auto* stderrTestOffsetHolder = (TTestLogsFileOffsetHolder*) StderrHolderContext_->GetLogsFileOffsetHolder().Get();

                for (ui32 loop = 1; loop <= 100; ++loop) {
                    logsTransmitterJob->Run();
                }

                UNIT_ASSERT_EQUAL(0, stdoutTestOffsetHolder->SerializeOffsetCalls);
                UNIT_ASSERT_EQUAL(0, stderrTestOffsetHolder->SerializeOffsetCalls);
                UNIT_ASSERT(stdoutTestOffsetHolder->SerializedContainers_.empty());
                UNIT_ASSERT(stderrTestOffsetHolder->SerializedContainers_.empty());
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock, new TTestFilter, new TTestPortoExistsFilter, initContainers);
        test.DoTest();
    }


    Y_UNIT_TEST(TestShutdown) {
        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter)
            {}

        protected:
            void Test() override {
                GetLogsTransmitterJob()->Shutdown();
                UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)StdoutHolderContext_->GetPortoOffsetHolder().Get())->SerializeCalls);
                UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)StderrHolderContext_->GetPortoOffsetHolder().Get())->SerializeCalls);
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock);
        test.DoTest();
    }

    Y_UNIT_TEST(TestInitialize) {
        class TTest: public ITestLogsTransmitterJobCanon {
        public:
            TTest(TFileStreamHolderPtr stdoutFileStreamHolder, TFileStreamHolderPtr stderrFileStreamHolder, TNeedTransmitLogsDetectorPtr needTransmitLogsDetector, TActiveContainersGetterPtr activeContainerGetter, TLogsTransmitterStatisticsPrinterPtr statisticsPrinter)
                : ITestLogsTransmitterJobCanon(stdoutFileStreamHolder, stderrFileStreamHolder, needTransmitLogsDetector, activeContainerGetter, statisticsPrinter)
            {}

        protected:
            void Test() override {
                GetLogsTransmitterJob()->Initialize();
                UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)StdoutHolderContext_->GetPortoOffsetHolder().Get())->DeserializeCalls);
                UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)StderrHolderContext_->GetPortoOffsetHolder().Get())->DeserializeCalls);
            }
        };

        TTest test(new TTestFileStreamHolder, new TTestFileStreamHolder, new TMockNeedTransmitLogsDetector, new TTestActiveContainersGetter, new TLogsTransmitterStatisticsPrinterMock);
        test.DoTest();
    }

}

} // namespace NInfra::NPodAgent::NLogsTransmitterCollectorJobTest
