#include "logs_transmitter_multithread.h"

#include <util/generic/hash_set.h>

namespace NInfra::NPodAgent {

TLogsTransmitterMultithread::TLogsTransmitterMultithread(
    ui32 numOfThreads
    , TLogsTransmitterPtr logsTransmitter
    , TAtomicSharedPtr<IThreadPool> mtpQueue
    , TLogFramePtr logFrame
)
    : NumOfThreads_(numOfThreads)
    , LogsTransmitter_(logsTransmitter)
    , MtpQueue_(mtpQueue)
    , LogFrame_(logFrame)
{}

void TLogsTransmitterMultithread::TransmitLogs(const THashSet<TPushContainer>& pushContainers) {
    auto splittedContainers = Split(pushContainers, NumOfThreads_);

    TVector<NThreading::TFuture<void>> futures;

    for (const auto& packOfContainers : splittedContainers) {
        futures.push_back(
            NThreading::Async(
                [this, packOfContainers] () {
                    try{
                        LogsTransmitter_->TransmitLogs(packOfContainers);
                    } catch(...) {
                        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(TPushClientError{EPushClientError::Unspecified, CurrentExceptionMessage()}));
                    }
                }
                , *MtpQueue_
            )
        );
    }

    WaitExceptionOrAll(futures).Wait();
}

TVector<THashSet<TPushContainer>> TLogsTransmitterMultithread::Split(const THashSet<TPushContainer>& pushContainers, const ui32 numOfThreads) {
    TVector<THashSet<TPushContainer>> splittedContainers;
    splittedContainers.reserve(numOfThreads);

    ui32 step = (pushContainers.size() / numOfThreads) + 1;
    ui32 curPosition = 0;

    auto it = pushContainers.begin();
    while(curPosition < pushContainers.size()) {
        auto end = pushContainers.end();

        if (curPosition + step <= pushContainers.size()) {
            end = std::next(it, step);
        }

        splittedContainers.emplace_back(it, end);
        it = end;
        curPosition += step;
    }

    return splittedContainers;
}

}
