#include "logs_transmitter_porto_offsets_checker_impl.h"

#include <infra/libs/logger/log_frame.h>

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>

#include <util/generic/ptr.h>

namespace NInfra::NPodAgent {

TLogsTransmitterPortoOffsetsCheckerImpl::TLogsTransmitterPortoOffsetsCheckerImpl(
    TPortoClientPtr portoClient
    , THoldersContextPtr holdersContext
    , TLogFramePtr logFrame
    , ui32 checkPeriodSec
    , EPortoContainerProperty offsetTypeProperty
    , EPortoContainerProperty streamPathProperty
    , std::function<void(THoldersContextPtr, const TPushContainer&, ui64, TLogFramePtr)> portoRotatedDataHandler
)
    : LastCheckTime_(Now().Seconds())
    , PortoClient_(portoClient)
    , HoldersContext_(holdersContext)
    , LogFrame_(logFrame)
    , CheckPeriodSec_(checkPeriodSec)
    , OffsetTypeProperty_(offsetTypeProperty)
    , StreamPathProperty_(streamPathProperty)
    , PortoRotatedDataHandler_(portoRotatedDataHandler)
{
    Y_ENSURE((OffsetTypeProperty_ == EPortoContainerProperty::StdoutOffset) || (OffsetTypeProperty_ == EPortoContainerProperty::StderrOffset), TStringBuilder() << "Invalid offset property provided to logs transmitter porto offsets checker: " << ToString(OffsetTypeProperty_) << ", should be: " << ToString(EPortoContainerProperty::StdoutOffset) << " or " << ToString(EPortoContainerProperty::StderrOffset));
    Y_ENSURE((StreamPathProperty_ == EPortoContainerProperty::StdOutPath) || (StreamPathProperty_ == EPortoContainerProperty::StdErrPath), TStringBuilder() << "Invalid stream path property provided to logs transmitter porto offsets checker: " << ToString(StreamPathProperty_) << ", should be: " << ToString(EPortoContainerProperty::StdOutPath) << " or " << ToString(EPortoContainerProperty::StdErrPath));
}

void TLogsTransmitterPortoOffsetsCheckerImpl::Check(const THashSet<TPushContainer>& pushContainers) {
    ui64 now = Now().Seconds();

    if (now >= (LastCheckTime_ + CheckPeriodSec_)) {

        for (const auto& pushContainer: pushContainers) {

            auto checkResult = CheckContainer(pushContainer);
            if (std::holds_alternative<TPortoError>(checkResult)) {
                auto error = std::get<TPortoError>(checkResult);
                if (error.Code != EPortoError::ContainerDoesNotExist ) {
                    LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(error));
                }
            } else if (std::holds_alternative<TPushClientError>(checkResult)) {
                auto error = std::get<TPushClientError>(checkResult);
                LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(error));
            }
        }
        LastCheckTime_ = now;
    }
}

bool TLogsTransmitterPortoOffsetsCheckerImpl::IsPortoLogFile(const TString& streamPath) {
    return streamPath.EndsWith(TPathHolder::LOG_FILE_EXTENSION);
}

std::variant<TEmptySuccess, TPortoError, TPushClientError> TLogsTransmitterPortoOffsetsCheckerImpl::CheckContainer(const TPushContainer &pushContainer) {
    TString streamPath = OUTCOME_TRYX(PortoClient_->GetProperty(pushContainer.Container, StreamPathProperty_));
    if (IsPortoLogFile(streamPath)) {
        ui64 offset = FromString(OUTCOME_TRYX(PortoClient_->GetProperty(pushContainer.Container, OffsetTypeProperty_)));
        ui64 cachedOffset = OUTCOME_TRYX(HoldersContext_->GetPortoOffsetHolder()->GetOffset(pushContainer));

        if (offset != cachedOffset) {
            PortoRotatedDataHandler_(HoldersContext_, pushContainer, offset, LogFrame_);
        }
    }
    return TEmptySuccess();
}

} //namespace NInfra::NPodAgent
