#include "logs_transmitter_porto_offsets_checker_impl.h"

#include <infra/libs/logger/logger.h>

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/file_stream_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/holders_context.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/logs_file_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/serializable_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NLogsTransmitterOffsetsCheckerTest {

static TLogger logger({});

static const THashSet<TPushContainer> pushContainers = {
    TPushContainer({TPortoContainerName({"box_box1"}, "workload_workload1"), {"box1"}, {"workload1"}})
};

static std::function<void(THoldersContextPtr, const TPushContainer&, ui64, TLogFramePtr)> portoRotatedDataTestHandler =
    [](THoldersContextPtr holdersContext, const TPushContainer& container, ui64 offset, TLogFramePtr) {
        holdersContext->GetPortoOffsetHolder()->UpdateOffset(container, offset);
    };

static ui64 defaultOffsetFromPorto = 5;
static ui64 defaultCachedOffset = 10;
static ui64 defaultCheckPeriodSec = 0;
static TString defaultStreamPath = TStringBuilder() << "filename" << TPathHolder::LOG_FILE_EXTENSION;

struct TTestPortoOffsetHolder : public TMockSerializableOffsetHolder {

    TTestPortoOffsetHolder(ui64 currentOffset)
        : CurrentOffset(currentOffset)
    { }

    TExpected<void, TPushClientError> UpdateOffset(const TPushContainer&, ui64 offset) override {
        ++UpdateOffsetCalls;
        CurrentOffset = offset;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
        ++GetOffsetCalls;
        return CurrentOffset;
    }

    ui64 CurrentOffset;

    size_t UpdateOffsetCalls = 0;
    mutable size_t GetOffsetCalls = 0;
};

struct TTestPortoClient : public TMockPortoClient {
    TTestPortoClient(
        ui64 currentOffset
        , const TString& streamPath = defaultStreamPath
    )
        : CurrentOffset(currentOffset)
        , StreamPath(streamPath)
    { }

    TExpected<TString, TPortoError> GetProperty(const TPortoContainerName&, EPortoContainerProperty property, int) override {
        if (EPortoContainerProperty::StdoutOffset == property || EPortoContainerProperty::StderrOffset == property) {
            ++GetOffsetPropertyCalls;
            return ToString(CurrentOffset);
        }

        if (EPortoContainerProperty::StdOutPath == property || EPortoContainerProperty::StdErrPath == property) {
            ++GetStreamPathPropertyCalls;
            return StreamPath;
        }

        return TString("");
    }

    ui64 CurrentOffset;
    TString StreamPath;
    size_t GetOffsetPropertyCalls = 0;
    size_t GetStreamPathPropertyCalls = 0;
};


Y_UNIT_TEST_SUITE(LogsTransmitterOffsetsCheckerSuite) {

    Y_UNIT_TEST(UpdateOffsetBeforePeriodShouldNotBeInvoked) {

        TPortoClientPtr portoClient = new TTestPortoClient(defaultOffsetFromPorto);
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder(defaultCachedOffset);

        TLogsTransmitterPortoOffsetsCheckerImplPtr offsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(
             portoClient
             , new THoldersContext(
                 new TMockFileStreamHolder
                 , new TMockLogsFileOffsetHolder
                 , portoOffsetHolder
             )
             , logger.SpawnFrame()
             , 10 /*checkPeriodSec*/
             , EPortoContainerProperty::StdoutOffset
             , EPortoContainerProperty::StdOutPath
             , portoRotatedDataTestHandler
        );

        offsetChecker->Check(pushContainers);

        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetOffsetPropertyCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetStreamPathPropertyCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls, 0);
    }

    Y_UNIT_TEST(UpdateOffsetWhenOutStreamPathIsNotPortoLogFileShouldNotBeInvoked) {
        TString outStreamPath = "stdout";
        TPortoClientPtr portoClient = new TTestPortoClient(defaultOffsetFromPorto, outStreamPath);
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder(defaultCachedOffset);

        TLogsTransmitterPortoOffsetsCheckerImplPtr offsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(
                portoClient
                , new THoldersContext(
                        new TMockFileStreamHolder
                        , new TMockLogsFileOffsetHolder
                        , portoOffsetHolder
                )
                , logger.SpawnFrame()
                , defaultCheckPeriodSec
                , EPortoContainerProperty::StdoutOffset
                , EPortoContainerProperty::StdOutPath
                , portoRotatedDataTestHandler
        );

        offsetChecker->Check(pushContainers);

        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetStreamPathPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetOffsetPropertyCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls, 0);
    }

    Y_UNIT_TEST(UpdateOffsetAfterPeriodIfOffsetFromPortoNotEqualToCachedOk) {

        TPortoClientPtr portoClient = new TTestPortoClient(defaultOffsetFromPorto);
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder(defaultCachedOffset);

        TLogsTransmitterPortoOffsetsCheckerImplPtr offsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(
            portoClient
            , new THoldersContext(
                new TMockFileStreamHolder
                , new TMockLogsFileOffsetHolder
                , portoOffsetHolder
            )
            , logger.SpawnFrame()
            , defaultCheckPeriodSec
            , EPortoContainerProperty::StdoutOffset
            , EPortoContainerProperty::StdOutPath
            , portoRotatedDataTestHandler
        );

        offsetChecker->Check(pushContainers);

        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetOffsetPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetStreamPathPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls, 1);
    }

    Y_UNIT_TEST(UpdateOffsetAfterPeriodIfOffsetFromPortoEqualToCachedShouldNotBeInvoked) {

        ui64 offsetFromPorto = 5;
        ui64 cachedOffset = offsetFromPorto;

        TPortoClientPtr portoClient = new TTestPortoClient(offsetFromPorto);
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder(cachedOffset);

        TLogsTransmitterPortoOffsetsCheckerImplPtr offsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(
            portoClient
            , new THoldersContext(
                new TMockFileStreamHolder
                , new TMockLogsFileOffsetHolder
                , portoOffsetHolder
            )
            , logger.SpawnFrame()
            , defaultCheckPeriodSec
            , EPortoContainerProperty::StdoutOffset
            , EPortoContainerProperty::StdOutPath
            , portoRotatedDataTestHandler
        );

        offsetChecker->Check(pushContainers);

        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetOffsetPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetStreamPathPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls, 0);
    }

    Y_UNIT_TEST(UpdateOffsetAfterPeriodIfPortoReturnsErrorShouldNotBeInvoked) {

        struct TFailPortoClient : public TTestPortoClient {
            TFailPortoClient(ui64 currentOffset)
                : TTestPortoClient(currentOffset)
            { }

            TExpected<TString, TPortoError> GetProperty(const TPortoContainerName&, EPortoContainerProperty, int) override {
                ++GetStreamPathPropertyCalls;
                return TPortoError{EPortoError::Unknown, "List", "", "NO"};
            }
        };

        TPortoClientPtr portoClient = new TFailPortoClient(defaultOffsetFromPorto);
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder(defaultOffsetFromPorto);

        TLogsTransmitterPortoOffsetsCheckerImplPtr offsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(
             portoClient
             , new THoldersContext(
                 new TMockFileStreamHolder
                 , new TMockLogsFileOffsetHolder
                 , portoOffsetHolder
             )
             , logger.SpawnFrame()
             , defaultCheckPeriodSec
             , EPortoContainerProperty::StdoutOffset
             , EPortoContainerProperty::StdOutPath
             , portoRotatedDataTestHandler
        );

        offsetChecker->Check(pushContainers);

        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetStreamPathPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TFailPortoClient*)portoClient.Get())->GetOffsetPropertyCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls, 0);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls, 0);
    }

    Y_UNIT_TEST(UpdateOffsetAfterPeriodIfHolderReturnsErrorOnOffsetGetShouldNotBeInvoked) {

        struct TFailPortoOffsetHolder : public TTestPortoOffsetHolder {
            TFailPortoOffsetHolder(ui64 currentOffset)
                : TTestPortoOffsetHolder(currentOffset)
            { }

            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{EPushClientError::NoOffsetForWorkload, ""};
            }
        };

        TPortoClientPtr portoClient = new TTestPortoClient(defaultOffsetFromPorto);
        TSerializableOffsetHolderPtr portoOffsetHolder = new TFailPortoOffsetHolder(defaultOffsetFromPorto);

        TLogsTransmitterPortoOffsetsCheckerImplPtr offsetChecker = new TLogsTransmitterPortoOffsetsCheckerImpl(
             portoClient
             , new THoldersContext(
                 new TMockFileStreamHolder
                 , new TMockLogsFileOffsetHolder
                 , portoOffsetHolder
             )
             , logger.SpawnFrame()
             , defaultCheckPeriodSec
             , EPortoContainerProperty::StdoutOffset
             , EPortoContainerProperty::StdOutPath
             , portoRotatedDataTestHandler
        );

        offsetChecker->Check(pushContainers);

        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetOffsetPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoClient*)portoClient.Get())->GetStreamPathPropertyCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls, 1);
        UNIT_ASSERT_EQUAL(((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls, 0);
    }
}

}
