#include "porto_rotated_handlers_context.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/file_stream_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/logs_file_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/serializable_offset_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NPortoRotatedHandlersContextTest {

static const ui64 defaultLogsFileOffset = 100;
static const ui64 defaultPortoOffset = 100;
static const ui64 newOffset = 150;
static const TPushContainer defaultContainer = {{{"box_1"}, "workload_1"}, "box_1", "workload_1"};
static TLogger logger({});

struct TTestLogsFileOffsetHolder: public TMockLogsFileOffsetHolder {
    TExpected<void, TPushClientError> DecrementOffset(const TPushContainer&, ui64 delta) override {
        CurrentOffset_ -= delta;
        ++DecrementOffsetCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
        ++GetOffsetCalls;
        return CurrentOffset_;
    }

    ui64 CurrentOffset_ = defaultLogsFileOffset;
    ui64 DecrementOffsetCalls = 0;
    mutable ui64 GetOffsetCalls = 0;
};

struct TTestPortoOffsetHolder: public TMockSerializableOffsetHolder {
    TExpected<void, TPushClientError> UpdateOffset(const TPushContainer&, ui64 offset) override {
        CurrentOffset_ = offset;
        ++UpdateOffsetCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
        ++GetOffsetCalls;
        return CurrentOffset_;
    }

    ui64 CurrentOffset_ = defaultPortoOffset;
    ui64 UpdateOffsetCalls = 0;
    mutable ui64 GetOffsetCalls = 0;
};

Y_UNIT_TEST_SUITE(PortoRotatedHandlersContextSuite) {

    Y_UNIT_TEST(PortoRotatedOnPeriodicCheckHandlerCallTest) {
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(new TMockFileStreamHolder, new TMockLogsFileOffsetHolder, portoOffsetHolder);

        auto func = TPortoRotatedHandlersContext::GetPortoRotatedOnPeriodicCheckHandler();
        func(holdersContext, defaultContainer, newOffset, logger.SpawnFrame());

        UNIT_ASSERT_EQUAL(newOffset, portoOffsetHolder->GetOffset(defaultContainer).Success());
    }

    Y_UNIT_TEST(PortoRotatedOnStartHandlerCallWhenOldOffsetLessDeltaTest) {
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder;
        TLogsFileOffsetHolderPtr logsFileOffsetHolder = new TTestLogsFileOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(new TMockFileStreamHolder, logsFileOffsetHolder, portoOffsetHolder);

        ui64 bigNewOffset = 10000;

        auto func = TPortoRotatedHandlersContext::GetPortoRotatedOnStartHandler();
        func(holdersContext, defaultContainer, bigNewOffset, logger.SpawnFrame());

        UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->DecrementOffsetCalls);

        UNIT_ASSERT_EQUAL(bigNewOffset, portoOffsetHolder->GetOffset(defaultContainer).Success());
        UNIT_ASSERT_EQUAL(0, logsFileOffsetHolder->GetOffset(defaultContainer).Success());
    }

    Y_UNIT_TEST(PortoRotatedOnStartHandlerCallWhenOldOffsetGreaterDeltaTest) {
        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder;
        TLogsFileOffsetHolderPtr logsFileOffsetHolder = new TTestLogsFileOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(new TMockFileStreamHolder, logsFileOffsetHolder, portoOffsetHolder);

        auto func = TPortoRotatedHandlersContext::GetPortoRotatedOnStartHandler();
        func(holdersContext, defaultContainer, newOffset, logger.SpawnFrame());

        UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->DecrementOffsetCalls);

        UNIT_ASSERT_EQUAL(newOffset, portoOffsetHolder->GetOffset(defaultContainer).Success());
        UNIT_ASSERT_EQUAL(defaultLogsFileOffset - (newOffset - defaultPortoOffset), logsFileOffsetHolder->GetOffset(defaultContainer).Success());
}

    Y_UNIT_TEST(PortoRotatedOnStartHandlerCallWhenPortoGetOffsetReturnErrorTest) {
        struct TTestPortoOffsetHolderFail: public TTestPortoOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{};
            }
        };

        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolderFail;
        TLogsFileOffsetHolderPtr logsFileOffsetHolder = new TTestLogsFileOffsetHolder;
        THoldersContextPtr holdersContext = new THoldersContext(new TMockFileStreamHolder, logsFileOffsetHolder, portoOffsetHolder);

        auto func = TPortoRotatedHandlersContext::GetPortoRotatedOnStartHandler();
        func(holdersContext, defaultContainer, newOffset, logger.SpawnFrame());

        UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->DecrementOffsetCalls);

        UNIT_ASSERT_EQUAL(defaultPortoOffset, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->CurrentOffset_);
        UNIT_ASSERT_EQUAL(defaultLogsFileOffset, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->CurrentOffset_);
    }

    Y_UNIT_TEST(PortoRotatedOnStartHandlerCallWhenLogsFileGetOffsetReturnErrorTest) {
        struct TTestLogsFileOffsetHolderFail: public TTestLogsFileOffsetHolder {
            TExpected<ui64, TPushClientError> GetOffset(const TPushContainer&) const override {
                ++GetOffsetCalls;
                return TPushClientError{};
            }
        };

        TSerializableOffsetHolderPtr portoOffsetHolder = new TTestPortoOffsetHolder;
        TLogsFileOffsetHolderPtr logsFileOffsetHolder = new TTestLogsFileOffsetHolderFail;
        THoldersContextPtr holdersContext = new THoldersContext(new TMockFileStreamHolder, logsFileOffsetHolder, portoOffsetHolder);

        auto func = TPortoRotatedHandlersContext::GetPortoRotatedOnStartHandler();
        func(holdersContext, defaultContainer, newOffset, logger.SpawnFrame());

        UNIT_ASSERT_EQUAL(1, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->UpdateOffsetCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->GetOffsetCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->DecrementOffsetCalls);

        UNIT_ASSERT_EQUAL(defaultPortoOffset, ((TTestPortoOffsetHolder*)portoOffsetHolder.Get())->CurrentOffset_);
        UNIT_ASSERT_EQUAL(defaultLogsFileOffset, ((TTestLogsFileOffsetHolder*)logsFileOffsetHolder.Get())->CurrentOffset_);
    }

}

} //namespace NInfra::NPodAgent::NPortoRotatedHandlersContextTest
