#include "file_deserializer.h"

#include <infra/libs/logger/logger.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/file_system_utils_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent {

static TLogger logger({});

static TPathHolderPtr pathHolder = new TPathHolder(
    ""
    , {{"", ""}}
    , {{"", ""}}
    , ""
    , ""
    , ""
    , ""
    , ""
    , ""
);
static const TPushContainer defaultContainer = {{{"box"}, "workload"}, "box", "workload"};
static const int deserializableValue = 10;
static const int defaultValue = 10;
static std::function<ui64(TContainerMetadata)> defaultFunc =
    [](TContainerMetadata meta) {
        return meta.file_offset();
    };

struct TTestFileSystemUtils: public TMockFileSystemUtils {
    TExpected<void, TPushClientError> MakeDirectory(const TString&) const override {
        ++MakeDirectoryCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    bool Exists(const TString&) const override {
        ++ExistsCalls;
        return true;
    }

    mutable size_t MakeDirectoryCalls = 0;
    mutable size_t ExistsCalls = 0;
};

Y_UNIT_TEST_SUITE(FileDeserializerSuite) {

    Y_UNIT_TEST(RealDeserializeOkTest) {
        TFileSystemUtilsPtr fileSystemUtils = new TFileSystemUtils;
        TDeserializerPtr fileDeserializer = new TFileDeserializer(pathHolder, ELogType::Stdout, logger.SpawnFrame(), fileSystemUtils, defaultFunc);

        NFs::MakeDirectoryRecursive(pathHolder->GetWorkloadLogsMetaPathAtLogsVolumePath());
        TString filePath = NLogsTransmitterUtils::GetMetadataFilePath(pathHolder, defaultContainer, ELogType::Stdout);

        TUnbufferedFileOutput stream(filePath);
        stream << "{\n  \"file_offset\":" << deserializableValue << "\n}";
        stream.Finish();

        ui64 value = defaultValue;
        fileDeserializer->Deserialize(defaultContainer, value);

        UNIT_ASSERT_EQUAL(deserializableValue, value);
    }

    Y_UNIT_TEST(DeserializableFailWhenFileNotExistTest) {
        struct TTestFileSystemUtilsFails: public TTestFileSystemUtils {
            bool Exists(const TString&) const override {
                ++ExistsCalls;
                return false;
            }
        };

        TFileSystemUtilsPtr fileSystemUtils = new TTestFileSystemUtilsFails;
        TDeserializerPtr fileDeserializer = new TFileDeserializer(pathHolder, ELogType::Stdout, logger.SpawnFrame(), fileSystemUtils, defaultFunc);
        ui64 value = defaultValue;

        fileDeserializer->Deserialize(defaultContainer, value);
        UNIT_ASSERT_EQUAL(1, ((TTestFileSystemUtils*)fileSystemUtils.Get())->ExistsCalls);
        UNIT_ASSERT_EQUAL(defaultValue, value);
    }
}

} //namespace NInfra::NPodAgent
