#include "logs_transmitter_statistics_printer_impl.h"

#include <infra/libs/logger/log_frame.h>

#include <util/generic/ptr.h>

namespace NInfra::NPodAgent {

bool TLogsTransmitterStatisticsPrinterImpl::IsInitialized_ = false;
TRWMutex TLogsTransmitterStatisticsPrinterImpl::SignalMutex_;

TLogsTransmitterStatisticsPrinterImpl::TLogsTransmitterStatisticsPrinterImpl(
    TLogsTransmitterStatisticsPtr statistics
    , TSessionHolderPtr stdoutSessionHolder
    , TSessionHolderPtr stderrSessionHolder
    , TFileStreamHolderPtr stdoutFileStreamHolder
    , TFileStreamHolderPtr stderrFileStreamHolder
    , TLogFramePtr logFrame
    , ui32 printPeriodSec
)
    : LastPrintTime_(Now().Seconds())
    , Statistics_(statistics)
    , StdoutSessionHolder_(stdoutSessionHolder)
    , StderrSessionHolder_(stderrSessionHolder)
    , StdoutFileStreamHolder_(stdoutFileStreamHolder)
    , StderrFileStreamHolder_(stderrFileStreamHolder)
    , LogFrame_(logFrame)
    , PrintPeriodSec_(printPeriodSec) {

    InitSignals();
}

void TLogsTransmitterStatisticsPrinterImpl::Print() {
    ui64 now = Now().Seconds();

    if (now >= (LastPrintTime_ + PrintPeriodSec_)) {

        if (Statistics_->GetNumBytesRotated())
            SendSignal(COUNTER_LOST_LOGS_ROTATED_BY_POD_AGENT, Statistics_->GetNumBytesRotated());

        if (Statistics_->GetNumBytesRotatedByPorto())
            SendSignal(COUNTER_LOST_LOGS_ROTATED_BY_PORTO, Statistics_->GetNumBytesRotatedByPorto());

        if (Statistics_->GetNumRawBytesSent())
            SendSignal(COUNTER_TRANSFERED_RAW_LOGS, Statistics_->GetNumRawBytesSent());

        if (Statistics_->GetNumWrappedBytesSent())
            SendSignal(COUNTER_TRANSFERED_WRAPPED_LOGS, Statistics_->GetNumWrappedBytesSent());

        if (Statistics_->GetNumOfSessionActivationErrors())
            SendSignal(COUNTER_NUM_OF_SESSION_ACTIVATION_ERRORS, Statistics_->GetNumOfSessionActivationErrors());

        if (Statistics_->GetNumErrorsFromPushAgent())
            SendSignal(COUNTER_ERRORS_FROM_PUSH_AGENT, Statistics_->GetNumErrorsFromPushAgent());

        auto stdoutFilesBecameInvalid = StdoutFileStreamHolder_->FilesBecameInvalid();
        PrintPortoLogFilesBecameInvalidCause(stdoutFilesBecameInvalid);
        auto stderrFilesBecameInvalid = StderrFileStreamHolder_->FilesBecameInvalid();
        PrintPortoLogFilesBecameInvalidCause(stderrFilesBecameInvalid);
        ui64 numPortoLogFilesBecameInvalid = stdoutFilesBecameInvalid.size() + stderrFilesBecameInvalid.size();

        if (numPortoLogFilesBecameInvalid)
            SendSignal(COUNTER_NUM_OF_PORTO_LOG_FILES_BECAME_INVALID, numPortoLogFilesBecameInvalid);

        SendSignal(COUNTER_NUM_OF_STDOUT_LOG_SESSIONS, StdoutSessionHolder_->NumOfActiveSessions());

        SendSignal(COUNTER_NUM_OF_STDERR_LOG_SESSIONS, StderrSessionHolder_->NumOfActiveSessions());

        Statistics_->Reset();

        LastPrintTime_ = now;
    }
}

void TLogsTransmitterStatisticsPrinterImpl::InitSignals() {
    {
        TReadGuard guard(SignalMutex_);
        if (IsInitialized_) {
            return;
        }
    }

    TWriteGuard guard(SignalMutex_);
    if (IsInitialized_) {
        return;
    }

    IsInitialized_ = true;

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_LOST_LOGS_ROTATED_BY_POD_AGENT
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_TRANSFERED_RAW_LOGS
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_TRANSFERED_WRAPPED_LOGS
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_SESSION_ACTIVATION_ERRORS
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_ERRORS_FROM_PUSH_AGENT
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_STDOUT_LOG_SESSIONS
        , "aeee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_STDERR_LOG_SESSIONS
        , "aeee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_LOST_LOGS_ROTATED_BY_PORTO
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_PORTO_LOG_FILES_BECAME_INVALID
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
}

void TLogsTransmitterStatisticsPrinterImpl::SendSignal(const TString& signalType, ui64 value) {
    if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, signalType, value)) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(
            TPushClientError{
                EPushClientError::SignalSendError
                , TStringBuilder() << "PushSingalUnsafe(" << signalType << ", " << ToString(value) << ") returned false"
            }
        ));
    }
}

void TLogsTransmitterStatisticsPrinterImpl::PrintPortoLogFilesBecameInvalidCause(const TVector<TPushClientError>& filesBecameInvalidCause) {
    for (const auto& cause: filesBecameInvalidCause) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(cause));
    }
}


} //namespace NInfra::NPodAgent
