#include "logs_transmitter_statistics_impl.h"
#include "logs_transmitter_statistics.h"
#include "logs_transmitter_statistics_printer_impl.h"

#include <infra/libs/logger/logger.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/file_stream_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NLogsTransmitterStatisticsPrinterTest {

static TLogger logger({});

struct TTestSessionHolder : public TMockSessionHolder {
    size_t NumOfActiveSessions() const override { return NumOfSessions_; }

    size_t NumOfSessions_ = 10;
};

struct TTestFileStreamHolder : public TMockFileStreamHolder {
    TVector<TPushClientError> FilesBecameInvalid() override { return FilesBecameInvalid_; }

    TVector<TPushClientError> FilesBecameInvalid_ = {TPushClientError{EPushClientError::FileInodeChanged, ""}};
};

Y_UNIT_TEST_SUITE(LogsTransmitterStatisticsPrinterSuite) {

    Y_UNIT_TEST(PrintStatisticsAfterPeriodOk) {

        TMultiUnistat::Instance().Reset(TMultiUnistat::ESignalNamespace::INFRA);
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        ui64 rawBytesSent = 5;
        ui64 wrappedBytesSent = 6;
        ui64 numBytesRotated = 7;
        ui64 numBytesRotatedByPorto = 8;

        statistics->AddNumRawBytesSent(rawBytesSent);
        statistics->AddNumWrappedBytesSent(wrappedBytesSent);
        statistics->AddNumBytesRotated(numBytesRotated);
        statistics->AddNumBytesRotatedByPorto(numBytesRotatedByPorto);
        statistics->SessionActivationErrorOccured();
        statistics->ErrorFromPushAgentOccured();

        TSessionHolderPtr stdoutSessionHolder = new TTestSessionHolder();
        TSessionHolderPtr stderrSessionHolder = new TTestSessionHolder();

        TFileStreamHolderPtr stdoutFileStreamHolder = new TTestFileStreamHolder();
        TFileStreamHolderPtr stderrFileStreamHolder = new TTestFileStreamHolder();

        ui32 printPeriod = 0;

        TLogsTransmitterStatisticsPrinterPtr printer = new TLogsTransmitterStatisticsPrinterImpl(statistics, stdoutSessionHolder, stderrSessionHolder, stdoutFileStreamHolder, stderrFileStreamHolder, logger.SpawnFrame(), printPeriod);

        printer->Print();

        ui32 priority = TMultiUnistat::ESignalPriority::INFRA_INFO;
        TString infoFromUnistat = TMultiUnistat::Instance().CreateInfoDump(TMultiUnistat::ESignalNamespace::INFRA, priority - 1);

        TString holeInfo = TStringBuilder()
            << "{"

            << "\"" << "pod_agent_logs_transmitter_errors_from_push_agent" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":1,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_lost_logs_rotated_by_pod_agent" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":7,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_lost_logs_rotated_by_porto" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":8,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_num_of_porto_log_files_became_invalid" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":2,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_num_of_session_activation_errors" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":1,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_num_of_stderr_log_sessions" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":10,"
            <<"\"Type\":" << "\"trnsp" << "\","
            <<"\"Suffix\":\"" << "aeee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_num_of_stdout_log_sessions" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":10,"
            <<"\"Type\":" << "\"trnsp" << "\","
            <<"\"Suffix\":\"" << "aeee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_transfered_raw_logs" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":5,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"},"

            << "\"" << "pod_agent_logs_transmitter_transfered_wrapped_logs" << "\":{"
            <<"\"Priority\":" << priority << ","
            <<"\"Value\":6,"
            <<"\"Type\":" << "\"summ" << "\","
            <<"\"Suffix\":\"" << "deee" << "\","
            <<"\"Tags\":\"\"}"

            << "}";

        UNIT_ASSERT_EQUAL(holeInfo, infoFromUnistat);

        UNIT_ASSERT(!statistics->GetNumRawBytesSent());
        UNIT_ASSERT(!statistics->GetNumWrappedBytesSent());
        UNIT_ASSERT(!statistics->GetNumBytesRotated());
        UNIT_ASSERT(!statistics->GetNumBytesRotatedByPorto());
        UNIT_ASSERT(!statistics->GetNumOfSessionActivationErrors());
        UNIT_ASSERT(!statistics->GetNumErrorsFromPushAgent());
    }

    Y_UNIT_TEST(PrintStatisticsBeforePeriodOk) {

        TMultiUnistat::Instance().Reset(TMultiUnistat::ESignalNamespace::INFRA);
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        ui64 rawBytesSent = 5;
        ui64 wrappedBytesSent = 6;
        ui64 numBytesRotated = 7;
        ui64 numBytesRotatedByPorto = 8;

        statistics->AddNumRawBytesSent(rawBytesSent);
        statistics->AddNumWrappedBytesSent(wrappedBytesSent);
        statistics->AddNumBytesRotated(numBytesRotated);
        statistics->AddNumBytesRotatedByPorto(numBytesRotatedByPorto);
        statistics->SessionActivationErrorOccured();
        statistics->ErrorFromPushAgentOccured();

        TSessionHolderPtr stdoutSessionHolder = new TTestSessionHolder();
        TSessionHolderPtr stderrSessionHolder = new TTestSessionHolder();

        TFileStreamHolderPtr stdoutFileStreamHolder = new TTestFileStreamHolder();
        TFileStreamHolderPtr stderrFileStreamHolder = new TTestFileStreamHolder();

        ui32 printPeriod = 20;

        TLogsTransmitterStatisticsPrinterPtr printer = new TLogsTransmitterStatisticsPrinterImpl(statistics, stdoutSessionHolder, stderrSessionHolder, stdoutFileStreamHolder, stderrFileStreamHolder, logger.SpawnFrame(), printPeriod);

        printer->Print();

        ui32 priority = TMultiUnistat::ESignalPriority::DEBUG;
        TString infoFromUnistat = TMultiUnistat::Instance().CreateInfoDump(TMultiUnistat::ESignalNamespace::INFRA, priority - 1);

        TString holeInfo = TStringBuilder()
            << "{}" ;

        UNIT_ASSERT_EQUAL(holeInfo, infoFromUnistat);

        UNIT_ASSERT_EQUAL(statistics->GetNumRawBytesSent(), rawBytesSent);
        UNIT_ASSERT_EQUAL(statistics->GetNumWrappedBytesSent(), wrappedBytesSent);
        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotated(), numBytesRotated);
        UNIT_ASSERT_EQUAL(statistics->GetNumBytesRotatedByPorto(), numBytesRotatedByPorto);
        UNIT_ASSERT_EQUAL(statistics->GetNumOfSessionActivationErrors(), 1);
        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 1);
    }
}

}
