#include "active_containers_getter.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>
#include <infra/pod_agent/libs/push_client/utils/push_client_types.h>
#include <util/generic/hash_set.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent {

TActiveContainersGetter::TActiveContainersGetter(
    TWorkloadStatusRepositoryPtr workloadStatusRepository
    , TPathHolderPtr pathHolder
    , TLogFramePtr logFrame
    , const bool isBoxAgentMode
)
    : WorkloadStatusRepository_(workloadStatusRepository)
    , PathHolder_(pathHolder)
    , LogFrame_(logFrame)
    , IsBoxAgentMode_(isBoxAgentMode)
{}

THashSet<TPushContainer> TActiveContainersGetter::GetActivePushContainers() {
    TVector<TString> workloadIds = WorkloadStatusRepository_->GetObjectIds();
    THashSet<TPushContainer> pushContainers;

    for (TString& workloadId: workloadIds) {
        API::TWorkloadStatus workloadStatus = WorkloadStatusRepository_->GetObjectStatus(workloadId);
        TString boxId = workloadStatus.box_ref();
        TString stdoutFilePath = GetLogsFilePath(workloadId, ELogType::Stdout);
        TString stderrFilePath = GetLogsFilePath(workloadId, ELogType::Stderr);

        bool isStdoutExist = NFs::Exists(stdoutFilePath);
        bool isStderrExist = NFs::Exists(stderrFilePath);
        if (isStdoutExist && isStderrExist) {
            TString startWorkload = TStringBuilder() << WORKLOAD_PREFIX << workloadId << START_CONTAINER_SUFFIX;
            if (IsBoxAgentMode_) {
                pushContainers.insert({TPortoContainerName(CONTAINER_PREFIX + startWorkload), "", std::move(workloadId)});
            } else {
                TString boxWithPrefix = TStringBuilder() << CONTAINER_PREFIX << BOX_PREFIX << boxId;
                pushContainers.insert({TPortoContainerName(boxWithPrefix, startWorkload), std::move(boxId), std::move(workloadId)});
            }

        } else if (isStdoutExist || isStderrExist) {
            LogFrame_->LogEvent(TLOG_ERR, NLogsTransmitterUtils::ConstructExceptionEvent(
                TPushClientError{
                    EPushClientError::FileNotExists
                    , "One of stdout and stderr files not exist."
                }
            ));
        }
    }

    return pushContainers;
}

TString TActiveContainersGetter::GetLogsFilePath(const TString& workloadId, const ELogType logType) {
    if (IsBoxAgentMode_) {
        return PathHolder_->GetWorkloadLogsFilePathInBox(workloadId, ToString(logType));
    }
    return PathHolder_->GetWorkloadLogsFilePathAtLogsVolumePath(workloadId, ToString(logType));
}

} //namespace NInfra::NPodAgent
