#include "file_system_utils.h"

#include <errno.h>
#include <sys/stat.h>
#include <util/datetime/base.h>
#include <util/system/fs.h>
#include <util/system/fstat.h>

#ifndef FALLOC_FL_COLLAPSE_RANGE
#define FALLOC_FL_COLLAPSE_RANGE        0x08
#endif

namespace NInfra::NPodAgent {

TExpected<blksize_t, TPushClientError> TFileSystemUtils::GetFileSystemBlockSize() {
    struct stat st;
    if (stat(ROOT_PATH, &st)) {
        return TPushClientError{EPushClientError::FileStatError, TStringBuilder() << "stat error while generate block size of file system: " << strerror(errno)};
    }
    return st.st_blksize;
}

bool TFileSystemUtils::Exists(const TString& path) const {
    return NFs::Exists(path);
}

TExpected<void, TPushClientError> TFileSystemUtils::CutHeadOfFile(const TFile& file, off64_t len) const {
    if (fallocate(file.GetHandle(), FALLOC_FL_COLLAPSE_RANGE, ZERO_OFFSET, len)) {
        return TPushClientError{EPushClientError::FileRotateError, TStringBuilder() << "fallocate return error while rotating: " << strerror(errno)};
    }
    return TExpected<void, TPushClientError>::DefaultSuccess();
}

TExpected<void, TPushClientError> TFileSystemUtils::MakeDirectory(const TString& path) const {
    if (!NFs::MakeDirectory(path)) {
        return TPushClientError{EPushClientError::DirectoryCreatingError, TStringBuilder() << "mkdir returns error: " << strerror(errno)};
    }

    return TExpected<void, TPushClientError>::DefaultSuccess();
}

TExpected<ino_t, TPushClientError> TFileSystemUtils::GetFileInode(const TString& path) const {
    struct stat st;
    if (stat(path.c_str(), &st)) {
        return TPushClientError{EPushClientError::FileStatError, TStringBuilder() << "stat error while getting file inode: " << strerror(errno)};
    }
    return st.st_ino;
}

} //namespace NInfra::NPodAgent
