#pragma once

#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/generic/variant.h>
#include <util/generic/vector.h>

namespace NInfra::NPodAgent {

struct TObjectMeta {
    TObjectMeta(
        const TString& id
        , ui64 specTimestamp
        , ui32 revision
    )
        : Id_(id)
        , SpecTimestamp_(specTimestamp)
        , Revision_(revision)
    {}

    bool operator==(const TObjectMeta& other) const {
        return Id_ == other.Id_
            && SpecTimestamp_ == other.SpecTimestamp_
            && Revision_ == other.Revision_
        ;
    }

    const TString Id_;
    const ui64 SpecTimestamp_;
    const ui32 Revision_;
};

struct TBoxMeta : public TObjectMeta {
    TBoxMeta(
        const TString& id
        , ui64 specTimestamp
        , ui32 revision
        , const TVector<TString>& rootfsLayerRefs
        , const TVector<TString>& staticResourceRefs
        , const TVector<TString>& volumeRefs
        , const TString& metaContainer
        , const TVector<TString>& initContainers
        , const TString& boxSpecificType
    )
        : TObjectMeta(
            id
            , specTimestamp
            , revision
        )
        , RootfsLayerRefs_(rootfsLayerRefs)
        , StaticResourceRefs_(staticResourceRefs)
        , VolumeRefs_(volumeRefs)

        , MetaContainer_(metaContainer)
        , InitContainers_(initContainers)

        , BoxSpecificType_(boxSpecificType)
    {}

    bool operator==(const TBoxMeta& other) const {
        return this->TObjectMeta::operator==(*(TObjectMeta*)&other)
            && RootfsLayerRefs_ == other.RootfsLayerRefs_
            && StaticResourceRefs_ == other.StaticResourceRefs_
            && VolumeRefs_ == other.VolumeRefs_

            && MetaContainer_ == other.MetaContainer_
            && InitContainers_ == other.InitContainers_

            && BoxSpecificType_ == other.BoxSpecificType_
        ;
    }

    const TVector<TString> RootfsLayerRefs_;
    const TVector<TString> StaticResourceRefs_;
    const TVector<TString> VolumeRefs_;

    const TString MetaContainer_;
    const TVector<TString> InitContainers_;

    const TString BoxSpecificType_;
};

struct TLayerMeta : public TObjectMeta {
    TLayerMeta(
        const TString& id
        , ui64 specTimestamp
        , ui32 revision
        , const TString& downloadHash
        , bool removeSourceFileAfterImport
    )
        : TObjectMeta(
            id
            , specTimestamp
            , revision
        )
        , DownloadHash_(downloadHash)
        , RemoveSourceFileAfterImport_(removeSourceFileAfterImport)
    {}

    bool operator==(const TLayerMeta& other) const {
        return this->TObjectMeta::operator==(*(TObjectMeta*)&other)
            && DownloadHash_ == other.DownloadHash_
            && RemoveSourceFileAfterImport_ == other.RemoveSourceFileAfterImport_
        ;
    }

    const TString DownloadHash_;
    const bool RemoveSourceFileAfterImport_;
};

struct TStaticResourceMeta : public TObjectMeta {
    TStaticResourceMeta(
        const TString& id
        , ui64 specTimestamp
        , ui32 revision
        , const TString& downloadHash
        , ui64 checkPeriodMs
    )
        : TObjectMeta(
            id
            , specTimestamp
            , revision
        )
        , DownloadHash_(downloadHash)
        , CheckPeriodMs_(checkPeriodMs)
    {}

    bool operator==(const TStaticResourceMeta& other) const {
        return this->TObjectMeta::operator==(*(TObjectMeta*)&other)
            && DownloadHash_ == other.DownloadHash_
            && CheckPeriodMs_ == other.CheckPeriodMs_
        ;
    }

    const TString DownloadHash_;
    const ui64 CheckPeriodMs_;
};

struct TVolumeMeta : public TObjectMeta {
    TVolumeMeta(
        const TString& id
        , ui64 specTimestamp
        , ui32 revision
        , const TVector<TString>& layerRefs
        , const TVector<TString>& staticResourceRefs
    )
        : TObjectMeta(
            id
            , specTimestamp
            , revision
        )
        , LayerRefs_(layerRefs)
        , StaticResourceRefs_(staticResourceRefs)
    {}

    bool operator==(const TVolumeMeta& other) const {
        return this->TObjectMeta::operator==(*(TObjectMeta*)&other)
            && LayerRefs_ == other.LayerRefs_
            && StaticResourceRefs_ == other.StaticResourceRefs_
        ;
    }

    const TVector<TString> LayerRefs_;
    const TVector<TString> StaticResourceRefs_;
};

struct TWorkloadMeta : public TObjectMeta {
    struct TEmptyInfo {
        bool operator==(const TEmptyInfo& /* other */) const {
            return true;
        }
    };

    struct TContainerInfo {
        TContainerInfo(
            const TString& containerName
        )
            : ContainerName_(containerName)
        {}

        bool operator==(const TContainerInfo& other) const {
            return ContainerName_ == other.ContainerName_;
        }

        const TString ContainerName_;
    };

    struct THttpGetInfo {
        bool operator==(const THttpGetInfo& /* other */) const {
            return true;
        }
    };

    struct TTcpCheckInfo {
        bool operator==(const TTcpCheckInfo& /* other */) const {
            return true;
        }
    };

    struct TUnixSignalInfo {
        bool operator==(const TUnixSignalInfo& /* other */) const {
            return true;
        }
    };

    struct TStartContainerAliveCheckInfo {
        bool operator==(const TStartContainerAliveCheckInfo& /* other */) const {
            return true;
        }
    };

    using THookInfo = std::variant<
        TEmptyInfo
        , TContainerInfo
        , THttpGetInfo
        , TTcpCheckInfo
        , TUnixSignalInfo
        , TStartContainerAliveCheckInfo
    >;

    TWorkloadMeta(
        const TString& id
        , ui64 specTimestamp
        , ui32 revision
        , const TString& boxRef

        , const TContainerInfo& startContainer
        , const TVector<TContainerInfo>& initContainers

        , const THookInfo& readiness
        , const THookInfo& liveness
        , const THookInfo& stop
        , const THookInfo& destroy
    )
        : TObjectMeta(
            id
            , specTimestamp
            , revision
        )
        , BoxRef_(boxRef)

        , StartContainer_(startContainer)
        , InitContainers_(initContainers)

        , Readiness_(readiness)
        , Liveness_(liveness)
        , Stop_(stop)
        , Destroy_(destroy)
    {}

    bool operator==(const TWorkloadMeta& other) const {
        return this->TObjectMeta::operator==(*(TObjectMeta*)&other)
            && BoxRef_ == other.BoxRef_

            && StartContainer_ == other.StartContainer_
            && InitContainers_ == other.InitContainers_

            && Readiness_ == other.Readiness_
            && Liveness_ == other.Liveness_
            && Stop_ == other.Stop_
            && Destroy_ == other.Destroy_
        ;
    }

    const TString BoxRef_;

    const TContainerInfo StartContainer_;
    const TVector<TContainerInfo> InitContainers_;

    const THookInfo Readiness_;
    const THookInfo Liveness_;
    const THookInfo Stop_;
    const THookInfo Destroy_;
};

} // namespace NInfra::NPodAgent
